/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: bbs_thread_view.c,v 1.45 2004/01/23 18:51:30 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_thread.h"
#include "worker.h"

#include "ochusha_ui.h"
#include "bbs_thread_view.h"

#include "marshal.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#if HAVE_ONIG_ONIGPOSIX_H
# include <onig/onigposix.h>
#else
# include <onigposix.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static GtkTextTag *get_alt_link_tag(BBSThreadView *view);
static GtkTextTag *get_alt_sage_tag(BBSThreadView *view);
static GtkTextTag *get_alt_hidden_link_tag(BBSThreadView *view);
static void bbs_thread_view_class_init(BBSThreadViewClass *klass);
static void bbs_thread_view_init(BBSThreadView *view);
static void bbs_thread_view_finalize(GObject *object);
static void bbs_thread_view_destroy(GtkObject *object);

static void text_view_realize_cb(GtkTextView *text_view,
				 BBSThreadView *bbs_thread_view);

static gboolean text_view_get_iter_at_link_tag_from_event(
						GtkTextView *text_view,
						GtkTextIter *iter,
						GdkEventAny *event);
static gboolean bbs_thread_view_motion_notify(GtkWidget *widget,
					      GdkEventMotion *event);
static gboolean bbs_thread_view_button_press(GtkWidget *widget,
					     GdkEventButton *event);
static gboolean bbs_thread_view_button_release(GtkWidget *widget,
					       GdkEventButton *event);
static void emit_link_mouse_over(BBSThreadView *view, GdkEventMotion *event);
static void emit_link_mouse_out(BBSThreadView *view, GdkEventAny *event);

static void bbs_thread_view_scroll_to(BBSThreadView *view, int to_where);


GType
bbs_thread_view_get_type(void)
{
  static GType btv_type = 0;

  if (btv_type == 0)
    {
      static const GTypeInfo btv_info =
	{
	  sizeof(BBSThreadViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)bbs_thread_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(BBSThreadView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)bbs_thread_view_init,
	};

      btv_type = g_type_register_static(GTK_TYPE_TEXT_VIEW,
					"BBSThreadView", &btv_info, 0);
    }

  return btv_type;
}


enum {
  LINK_MOUSE_OVER_SIGNAL,
  LINK_MOUSE_OUT_SIGNAL,
  LINK_MOUSE_PRESS_SIGNAL,
  LINK_MOUSE_RELEASE_SIGNAL,
  SCROLL_TO_SIGNAL,
  SCROLL_VIEW_SIGNAL,
  INTERACTIVE_SEARCH_SIGNAL,
  WRITE_RESPONSE_SIGNAL,
  CLOSE_THREAD_SIGNAL,
  LAST_SIGNAL
};


enum {
  SCROLL_TO_BUFFER_START,
  SCROLL_TO_BUFFER_END
};


static GtkTextViewClass *parent_class = NULL;
static int bbs_thread_view_signals[LAST_SIGNAL] =
  { 0, 0, 0, 0, 0, 0, 0, 0, 0, };
static GtkTextTagTable *default_tag_table;

static GtkTextTag *link_tags[2];
static GtkTextTag *sage_tags[2];
static GtkTextTag *hidden_link_tags[2];


static GtkTextTag *
get_alt_link_tag(BBSThreadView *view)
{
  GtkTextTag *link_tag = link_tags[view->alt_link_tag];
  view->alt_link_tag = (view->alt_link_tag + 1) & 1;
  return link_tag;
}


static GtkTextTag *
get_alt_sage_tag(BBSThreadView *view)
{
  GtkTextTag *sage_tag = sage_tags[view->alt_sage_tag];
  view->alt_sage_tag = (view->alt_sage_tag + 1) & 1;
  return sage_tag;
}


static GtkTextTag *
get_alt_hidden_link_tag(BBSThreadView *view)
{
  GtkTextTag *hidden_link_tag = hidden_link_tags[view->alt_hidden_link_tag];
  view->alt_hidden_link_tag = (view->alt_hidden_link_tag + 1) & 1;
  return hidden_link_tag;
}


GtkTextTagTable *
bbs_thread_view_class_get_default_tag_table(BBSThreadViewClass *klass)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW_CLASS(klass), NULL);

  return default_tag_table;
}


static void
bbs_thread_view_class_init(BBSThreadViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;
  GtkBindingSet *binding_set;

  GtkTextTag *tag;

  parent_class = g_type_class_peek_parent(klass);
  binding_set = gtk_binding_set_by_class(klass);

  /* GObject signals */
  o_class->finalize = bbs_thread_view_finalize;

  /* GtkObject signals */
  object_class->destroy = bbs_thread_view_destroy;

  /* ǥեȤΥν */
  default_tag_table = gtk_text_tag_table_new();
  klass->tag_table = default_tag_table;

  /* TODO: طѹǽˤ */
  tag = gtk_text_tag_new("thread_title");
  g_object_set(G_OBJECT(tag),
	       "foreground", "red",
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  tag = gtk_text_tag_new("response_header");
  gtk_text_tag_table_add(default_tag_table, tag);

  tag = gtk_text_tag_new("response_name");
  g_object_set(G_OBJECT(tag),
	       "foreground", "forestgreen",
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  /*
   * ƱΥʣϢ³Ƥ硢GtkTextBufferǸʬĤʤ
   * Τǡ2(link0link1)ߤ˻ȤƱ˰
   */
  tag = gtk_text_tag_new("link0");
  g_object_set(G_OBJECT(tag),
	       "foreground", "blue",
	       "underline", PANGO_UNDERLINE_SINGLE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
  link_tags[0] = tag;

  tag = gtk_text_tag_new("link1");
  g_object_set(G_OBJECT(tag),
	       "foreground", "blue",
	       "underline", PANGO_UNDERLINE_SINGLE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
  link_tags[1] = tag;

  /*
   * ƱΥʣϢ³Ƥ硢GtkTextBufferǸʬĤʤ
   * Τǡ2(sage0sage1)ߤ˻ȤƱ˰
   */
  tag = gtk_text_tag_new("sage0");
  g_object_set(G_OBJECT(tag),
	       "foreground", "purple",
	       "underline", PANGO_UNDERLINE_SINGLE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
  sage_tags[0] = tag;

  tag = gtk_text_tag_new("sage1");
  g_object_set(G_OBJECT(tag),
	       "foreground", "purple",
	       "underline", PANGO_UNDERLINE_SINGLE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
  sage_tags[1] = tag;

  /*
   * ƱΥʣϢ³Ƥ硢GtkTextBufferǸʬĤʤ
   * Τǡ2(hidden0hidden1)ߤ˻ȤƱ˰
   */
  tag = gtk_text_tag_new("hidden0");
  gtk_text_tag_table_add(default_tag_table, tag);
  hidden_link_tags[0] = tag;

  tag = gtk_text_tag_new("hidden1");
  gtk_text_tag_table_add(default_tag_table, tag);
  hidden_link_tags[1] = tag;

  tag = gtk_text_tag_new("response_paragraph");
  g_object_set(G_OBJECT(tag),
	       "indent", 0,
	       "left-margin", 30,
	       "left-margin-set", TRUE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  tag = gtk_text_tag_new("bold");
  g_object_set(G_OBJECT(tag),
	       "weight", PANGO_WEIGHT_BOLD,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  tag = gtk_text_tag_new("italic");
  g_object_set(G_OBJECT(tag),
	       "style", PANGO_STYLE_ITALIC,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

#if 0
  tag = gtk_text_tag_new("match-string");
  g_object_set(G_OBJECT(tag),
	       "foreground", "red",
	       "background", "#ffffd0",
	       "weight", PANGO_WEIGHT_BOLD,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
#endif
			 
  bbs_thread_view_signals[LINK_MOUSE_OVER_SIGNAL] =
    g_signal_new("link_mouse_over",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_over),
		 NULL, NULL,
		 ochusha_marshal_VOID__BOXED_POINTER_STRING,
		 G_TYPE_NONE, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);
  bbs_thread_view_signals[LINK_MOUSE_OUT_SIGNAL] =
    g_signal_new("link_mouse_out",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_out),
		 NULL, NULL,
		 ochusha_marshal_VOID__BOXED_POINTER_STRING,
		 G_TYPE_NONE, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);
  bbs_thread_view_signals[LINK_MOUSE_PRESS_SIGNAL] =
    g_signal_new("link_mouse_press",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_press),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__BOXED_POINTER_STRING,
		 G_TYPE_BOOLEAN, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);
  bbs_thread_view_signals[LINK_MOUSE_RELEASE_SIGNAL] =
    g_signal_new("link_mouse_release",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_release),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__BOXED_POINTER_STRING,
		 G_TYPE_BOOLEAN, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);

  bbs_thread_view_signals[SCROLL_TO_SIGNAL] =
    g_signal_new("scroll_to",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BBSThreadViewClass, scroll_to),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  bbs_thread_view_signals[SCROLL_VIEW_SIGNAL] =
    g_signal_new("scroll_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BBSThreadViewClass, scroll_view),
		 NULL, NULL,
		 ochusha_marshal_VOID__ENUM,
		 G_TYPE_NONE, 1,
		 GTK_TYPE_SCROLL_TYPE);

  bbs_thread_view_signals[INTERACTIVE_SEARCH_SIGNAL] =
    g_signal_new("interactive_search",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BBSThreadViewClass, interactive_search),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);

  bbs_thread_view_signals[WRITE_RESPONSE_SIGNAL] =
    g_signal_new("write_response",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BBSThreadViewClass, write_response),
		 NULL, NULL,
		 ochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

  bbs_thread_view_signals[CLOSE_THREAD_SIGNAL] =
    g_signal_new("close_thread",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BBSThreadViewClass, close_thread),
		 NULL, NULL,
		 ochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

  gtk_binding_entry_add_signal(binding_set, GDK_Home, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_START);
  gtk_binding_entry_add_signal(binding_set, GDK_KP_Home, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_START);
  gtk_binding_entry_add_signal(binding_set, GDK_End, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_END);
  gtk_binding_entry_add_signal(binding_set, GDK_KP_End, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_END);

  gtk_binding_entry_add_signal(binding_set, GDK_space, 0,
			       "scroll_view", 1,
			       GTK_TYPE_SCROLL_TYPE, GTK_SCROLL_PAGE_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_Page_Down, 0,
			       "scroll_view", 1,
			       GTK_TYPE_SCROLL_TYPE, GTK_SCROLL_PAGE_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_BackSpace, 0,
			       "scroll_view", 1,
			       GTK_TYPE_SCROLL_TYPE, GTK_SCROLL_PAGE_BACKWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_Page_Up, 0,
			       "scroll_view", 1,
			       GTK_TYPE_SCROLL_TYPE, GTK_SCROLL_PAGE_BACKWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_Down, 0,
			       "scroll_view", 1,
			       GTK_TYPE_SCROLL_TYPE, GTK_SCROLL_STEP_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_Up, 0,
			       "scroll_view", 1,
			       GTK_TYPE_SCROLL_TYPE, GTK_SCROLL_STEP_BACKWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_S, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       BBS_THREAD_VIEW_SEARCH_ACTION_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_R, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       BBS_THREAD_VIEW_SEARCH_ACTION_BACKWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_A, 0,
			       "write_response", 0);

  gtk_binding_entry_add_signal(binding_set, GDK_W, GDK_CONTROL_MASK,
			       "close_thread", 0);

  widget_class->motion_notify_event = bbs_thread_view_motion_notify;
  widget_class->button_press_event = bbs_thread_view_button_press;
  widget_class->button_release_event = bbs_thread_view_button_release;

  klass->link_mouse_over = NULL;
  klass->link_mouse_out = NULL;
  klass->link_mouse_press = NULL;
  klass->link_mouse_release = NULL;

  klass->scroll_to = bbs_thread_view_scroll_to;
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
bbs_thread_view_init(BBSThreadView *view)
{
  GtkWidget *widget = GTK_WIDGET(view);
  GtkTextBuffer *buffer = gtk_text_buffer_new(default_tag_table);

  gtk_text_view_set_buffer(GTK_TEXT_VIEW(view), buffer);

  view->text_buffer = buffer;

  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(view), GTK_WRAP_CHAR);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(view), FALSE);
  gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(view), FALSE);

  g_signal_connect(G_OBJECT(widget), "realize",
		   G_CALLBACK(text_view_realize_cb), view);

  gtk_text_buffer_get_start_iter(buffer, &view->iter);

  view->tags = NULL;

  view->last_offset = -1;
  view->thread = NULL;
  view->link_table = g_hash_table_new_full(NULL, NULL, NULL, TRACE_FREE);
  view->child_table = g_hash_table_new(NULL, NULL);

  view->insert_mark = gtk_text_buffer_get_mark(buffer, "insert");
  view->selection_bound_mark = gtk_text_buffer_get_mark(buffer,
							"selection_bound");
}


static void
bbs_thread_view_finalize(GObject *object)
{
  BBSThreadView *view = BBS_THREAD_VIEW(object);

#if 0
  fprintf(stderr, "bbs_thread_view_finalize(%p)\n", object);
#endif

  if (view->text_buffer != NULL)
    {
      OCHU_OBJECT_UNREF(view->text_buffer);
      view->text_buffer = NULL;
    }
  if (view->tags != NULL)
    {
      g_slist_free(view->tags);
      view->tags = NULL;
    }

  if (view->link_table != NULL)
    {
      g_hash_table_destroy(view->link_table);
      view->link_table = NULL;
    }

  if (view->child_table != NULL)
    {
      g_hash_table_destroy(view->child_table);
      view->child_table = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
bbs_thread_view_destroy(GtkObject *object)
{
  BBSThreadView *view = BBS_THREAD_VIEW(object);

#if 0
  fprintf(stderr, "bbs_thread_view_destroy(%p)\n", object);
#endif

  if (view->ibeam_cursor != NULL)
    {
      gdk_cursor_unref(view->ibeam_cursor);
      view->ibeam_cursor = NULL;
    }

  if (view->hand_cursor != NULL)
    {
      gdk_cursor_unref(view->hand_cursor);
      view->hand_cursor = NULL;
    }

  if (view->search_key != NULL)
    {
      G_FREE(view->search_key);
      view->search_key = NULL;
    }

  if (view->regexp_available)
    {
      regfree(&view->regexp);
      view->regexp_available = FALSE;
      if (view->regexp_match != NULL)
	{
	  G_FREE(view->regexp_match);
	  view->regexp_match = NULL;
	}
    }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);

}


static void
text_view_realize_cb(GtkTextView *text_view, BBSThreadView *bbs_thread_view)
{
  GdkDisplay *display = gtk_widget_get_display(GTK_WIDGET(text_view));
  GdkWindow *window = gtk_text_view_get_window(text_view,
					       GTK_TEXT_WINDOW_TEXT);
  gdk_window_set_events(window,
			(gdk_window_get_events(window)
			 & ~GDK_POINTER_MOTION_HINT_MASK)
			| GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);

  if (bbs_thread_view->ibeam_cursor != NULL)
    gdk_cursor_unref(bbs_thread_view->ibeam_cursor);
  bbs_thread_view->ibeam_cursor = gdk_cursor_new_for_display(display,
							     GDK_XTERM);

  if (bbs_thread_view->hand_cursor != NULL)
    gdk_cursor_unref(bbs_thread_view->hand_cursor);
  bbs_thread_view->hand_cursor = gdk_cursor_new_for_display(display,
							    GDK_HAND2);
}


static gboolean
text_view_get_iter_at_link_tag_from_event(GtkTextView *text_view,
					  GtkTextIter *iter,
					  GdkEventAny *event)
{
  int buf_x;
  int buf_y;

  switch (event->type)
    {
    case GDK_MOTION_NOTIFY:
      {
	GdkEventMotion *me = (GdkEventMotion *)event;
	gtk_text_view_window_to_buffer_coords(text_view, GTK_TEXT_WINDOW_TEXT,
					      me->x, me->y, &buf_x, &buf_y);
	break;
      }

    case GDK_BUTTON_PRESS:
    case GDK_2BUTTON_PRESS:
    case GDK_3BUTTON_PRESS:
    case GDK_BUTTON_RELEASE:
      {
	GdkEventButton *be = (GdkEventButton *)event;
	gtk_text_view_window_to_buffer_coords(text_view, GTK_TEXT_WINDOW_TEXT,
					      be->x, be->y, &buf_x, &buf_y);
	break;
      }

    default:
      return FALSE;
    }

  gtk_text_view_get_iter_at_location(text_view, iter, buf_x, buf_y);

  if (gtk_text_iter_has_tag(iter, link_tags[0]))
    {	/* link_tags[0]ǰϤޤƤϰϤؤƤ */
      if (gtk_text_iter_begins_tag(iter, link_tags[0]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, link_tags[0]);
    }
  else if (gtk_text_iter_has_tag(iter, link_tags[1]))
    {	/* link_tags[1]ؤƤ */
      if (gtk_text_iter_begins_tag(iter, link_tags[1]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, link_tags[1]);
    }
  else if (gtk_text_iter_has_tag(iter, sage_tags[0]))
    {	/* sage_tags[0]ǰϤޤƤϰϤؤƤ */
      if (gtk_text_iter_begins_tag(iter, sage_tags[0]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, sage_tags[0]);
    }
  else if (gtk_text_iter_has_tag(iter, sage_tags[1]))
    {	/* sage_tags[1]ؤƤ */
      if (gtk_text_iter_begins_tag(iter, sage_tags[1]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, sage_tags[1]);
    }
  else if (gtk_text_iter_has_tag(iter, hidden_link_tags[0]))
    {	/* hidden_link_tags[0]ǰϤޤƤϰϤؤƤ */
      if (gtk_text_iter_begins_tag(iter, hidden_link_tags[0]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, hidden_link_tags[0]);
    }
  else if (gtk_text_iter_has_tag(iter, hidden_link_tags[1]))
    {	/* hidden_link_tags[1]ؤƤ */
      if (gtk_text_iter_begins_tag(iter, hidden_link_tags[1]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, hidden_link_tags[1]);
    }
  
  return FALSE;
}


static gboolean
bbs_thread_view_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
  gboolean result = FALSE;
  GtkTextIter iter;
  GtkTextView *text_view;
  BBSThreadView *bbs_thread_view;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(widget)
		       && event->type == GDK_MOTION_NOTIFY, FALSE);

  text_view = GTK_TEXT_VIEW(widget);
  bbs_thread_view = BBS_THREAD_VIEW(widget);

  if (text_view_get_iter_at_link_tag_from_event(text_view, &iter,
						(GdkEventAny *)event))
    {
      int start_offset = gtk_text_iter_get_offset(&iter);
      if (start_offset == bbs_thread_view->last_offset)
	{
	  goto finished;
	}

      if (bbs_thread_view->last_offset != -1)
	emit_link_mouse_out(bbs_thread_view, (GdkEventAny *)event);

      bbs_thread_view->last_offset = start_offset;
      emit_link_mouse_over(bbs_thread_view, event);
    }
  else if (bbs_thread_view->last_offset != -1)
    emit_link_mouse_out(bbs_thread_view, (GdkEventAny *)event);

 finished:
  if (GTK_WIDGET_CLASS(parent_class)->motion_notify_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->motion_notify_event)(widget,
								    event);

  return result;
}


static void
emit_link_mouse_over(BBSThreadView *view, GdkEventMotion *event)
{
  gchar *link;
  if (view->last_offset == -1)
    return;

  if (view->link_table == NULL)
    return;

  link = (gchar *)g_hash_table_lookup(view->link_table,
				      (gconstpointer)view->last_offset);

  if (link == NULL)
    return;

  gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(view),
						 GTK_TEXT_WINDOW_TEXT),
			view->hand_cursor);

  g_signal_emit(G_OBJECT(view),
		bbs_thread_view_signals[LINK_MOUSE_OVER_SIGNAL],
		0,
		event,
		view->thread,
		link);
}


static void
emit_link_mouse_out(BBSThreadView *view, GdkEventAny *event)
{
  gchar *link;
  if (view->last_offset == -1)
    return;

  if (view->link_table == NULL)
    return;

  link = (gchar *)g_hash_table_lookup(view->link_table,
				      (gconstpointer)view->last_offset);

  view->last_offset = -1;

  if (link == NULL)
    return;

  gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(view),
						 GTK_TEXT_WINDOW_TEXT),
			view->ibeam_cursor);

  g_signal_emit(G_OBJECT(view),
		bbs_thread_view_signals[LINK_MOUSE_OUT_SIGNAL],
		0,
		event,
		view->thread,
		link);
}


static gboolean
bbs_thread_view_button_press(GtkWidget *widget, GdkEventButton *event)
{
  GtkTextView *text_view;
  GtkTextIter iter;
  gboolean result = FALSE;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(widget)
		       && (event->type == GDK_BUTTON_PRESS
			   || event->type == GDK_2BUTTON_PRESS
			   || event->type == GDK_3BUTTON_PRESS), FALSE);
  text_view = GTK_TEXT_VIEW(widget);

  if (text_view_get_iter_at_link_tag_from_event(text_view, &iter,
						(GdkEventAny *)event))
    {
      BBSThreadView *bbs_thread_view = BBS_THREAD_VIEW(text_view);
      int start_offset = gtk_text_iter_get_offset(&iter);
      gchar *link = (bbs_thread_view->link_table != NULL)
	? (gchar *)g_hash_table_lookup(bbs_thread_view->link_table,
				       (gconstpointer)start_offset)
	: NULL;

      g_signal_emit(G_OBJECT(bbs_thread_view),
		    bbs_thread_view_signals[LINK_MOUSE_PRESS_SIGNAL],
		    0,
		    event,
		    bbs_thread_view->thread,
		    link,
		    &result);
    }

  if (!result && GTK_WIDGET_CLASS(parent_class)->button_press_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->button_press_event)(widget,
								   event);

  return result;
}


static gboolean
bbs_thread_view_button_release(GtkWidget *widget, GdkEventButton *event)
{
  GtkTextView *text_view;
  GtkTextIter iter;
  gboolean result = FALSE;

  g_return_val_if_fail(GTK_IS_TEXT_VIEW(widget)
		       && event->type == GDK_BUTTON_RELEASE, FALSE);
  text_view = GTK_TEXT_VIEW(widget);

  if (text_view_get_iter_at_link_tag_from_event(text_view, &iter,
						(GdkEventAny *)event))
    {
      BBSThreadView *bbs_thread_view = BBS_THREAD_VIEW(text_view);
      int start_offset = gtk_text_iter_get_offset(&iter);
      gchar *link = (bbs_thread_view->link_table != NULL)
	? (gchar *)g_hash_table_lookup(bbs_thread_view->link_table,
				       (gconstpointer)start_offset)
	: NULL;

      g_signal_emit(G_OBJECT(bbs_thread_view),
		    bbs_thread_view_signals[LINK_MOUSE_RELEASE_SIGNAL],
		    0,
		    event,
		    bbs_thread_view->thread,
		    link,
		    &result);
    }

  if (GTK_WIDGET_CLASS(parent_class)->button_release_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->button_release_event)(widget,
								     event);

  return result;
}


GtkWidget *
bbs_thread_view_new(OchushaBBSThread *thread)
{
  BBSThreadView *view = BBS_THREAD_VIEW(g_object_new(BBS_THREAD_VIEW_TYPE,
						     NULL));
  view->thread = thread;
  return GTK_WIDGET(view);
}


void
bbs_thread_view_set_thread(BBSThreadView *view, OchushaBBSThread *thread)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  view->thread = thread;
}


OchushaBBSThread *
bbs_thread_view_get_thread(BBSThreadView *view)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  return view->thread;
}


void
bbs_thread_view_clear(BBSThreadView *view)
{
  GtkTextIter end_iter;
  GHashTable *link_table;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  gtk_text_buffer_get_start_iter(view->text_buffer, &view->iter);
  gtk_text_buffer_get_end_iter(view->text_buffer, &end_iter);

  gtk_text_buffer_delete(view->text_buffer, &view->iter, &end_iter);

  link_table = view->link_table;
  view->link_table = NULL;
  if (link_table != NULL)
    g_hash_table_destroy(link_table);
  view->link_table = g_hash_table_new_full(NULL, NULL, NULL, TRACE_FREE);
}


int
bbs_thread_view_get_current_offset(BBSThreadView *view)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), 0);
  g_return_val_if_fail(view->text_buffer != NULL, 0);

  return gtk_text_iter_get_offset(&view->iter);
}


GtkTextMark *
bbs_thread_view_create_mark(BBSThreadView *view)
{
  GtkTextMark *mark;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);
  g_return_val_if_fail(view->text_buffer != NULL, NULL);

  mark = gtk_text_buffer_create_mark(view->text_buffer, NULL, &view->iter,
				     TRUE); 
  return mark;
}


void
bbs_thread_view_delete_mark(BBSThreadView *view, GtkTextMark *mark)
{
  GtkTextBuffer *buffer;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (mark == NULL)
    return;

  buffer = gtk_text_mark_get_buffer(mark);
  if (buffer == NULL)
    return;

  gtk_text_buffer_delete_mark(buffer, mark);
}


typedef struct _ApplyTagArgs
{
  GtkTextIter *start;
  GtkTextIter *end;
  GtkTextBuffer *buffer;
} ApplyTagArgs;


static void
apply_tag(gpointer data, gpointer user_data)
{
  GtkTextTag *tag = GTK_TEXT_TAG(data);
  ApplyTagArgs *args = (ApplyTagArgs *)user_data;
  if (tag == NULL)
    {
      fprintf(stderr, "Why data is NULL?\n");
      return;
    }
  gtk_text_buffer_apply_tag(args->buffer, tag, args->start, args->end);
}


void
bbs_thread_view_append_text(BBSThreadView *view, const gchar *text, int len)
{
  GtkTextIter start;
  ApplyTagArgs args;
  int start_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && text != NULL);

  args.start = &start;
  args.end = &view->iter;
  args.buffer = view->text_buffer;
  start_offset = gtk_text_iter_get_offset(&view->iter);

  if (view->last_is_new_line)
    {
      if (len > 0)
	while (len > 0 && *text == ' ')
	  {
	    text++;
	    len--;
	  }
      else
	while (*text != '\0' && *text == ' ')
	  {
	    text++;
	  }
    }

  if (len != 0)
    gtk_text_buffer_insert(view->text_buffer, &view->iter, text, len);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start, start_offset);
  g_slist_foreach(view->tags, apply_tag, &args);

  if (text[len - 1] == '\n')
    view->last_is_new_line = 1;
  else
    view->last_is_new_line = 0;
}


/*
 * MEMO: textlen byteʬ󥯤äݤɽʸȤƥХåե
 *       linkȤʸϸ˥ʥΰȤƻȤ롣
 *       textˤ"link"դtextƬʸΥХåե
 *       offsetʸlinkؤΥϥåơ֥Ͽ롣
 */
void
bbs_thread_view_append_text_as_link(BBSThreadView *view,
				    const gchar *text, int len,
				    const gchar *link)
{
  GtkTextIter start;
  int start_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && text != NULL);

  start_offset = gtk_text_iter_get_offset(&view->iter);
  bbs_thread_view_append_text(view, text, len);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start, start_offset);
  gtk_text_buffer_apply_tag(view->text_buffer, get_alt_link_tag(view),
			    &start, &view->iter);
  g_hash_table_insert(view->link_table,
		      (gpointer)start_offset, G_STRDUP(link));
}


/*
 * MEMO: textlen byteʬ󥯤äݤɽʸȤƥХåե
 *       linkȤʸϸ˥ʥΰȤƻȤ롣
 *       textˤ"sage"դtextƬʸΥХåե
 *       offsetʸlinkؤΥϥåơ֥Ͽ롣
 *       Ƥ"sage"ΤߤǤ뤳Ȥɽ뤿ᡢ̾Υ󥯤ȤϿѤ롣
 */
void
bbs_thread_view_append_text_as_sage(BBSThreadView *view,
				    const gchar *text, int len,
				    const gchar *link)
{
  GtkTextIter start;
  int start_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && text != NULL);

  start_offset = gtk_text_iter_get_offset(&view->iter);
  bbs_thread_view_append_text(view, text, len);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start, start_offset);
  gtk_text_buffer_apply_tag(view->text_buffer, get_alt_sage_tag(view),
			    &start, &view->iter);
  g_hash_table_insert(view->link_table,
		      (gpointer)start_offset, G_STRDUP(link));
}


/*
 * MEMO: textlen byteʬ򤳤ä󥯤äݤ갷٤ʸȤ
 *       ХåեƤϡ󥯤äݤʤ
 *       linkȤʸϸ˥ʥΰȤƻȤ롣
 *       textˤ"hidden_link"դtextƬʸΥХåե
 *       offsetʸlinkؤΥϥåơ֥Ͽ롣
 */
void
bbs_thread_view_append_text_as_hidden_link(BBSThreadView *view,
					   const gchar *text, int len,
					   const gchar *link)
{
  GtkTextIter start;
  int start_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && text != NULL);

  start_offset = gtk_text_iter_get_offset(&view->iter);
  bbs_thread_view_append_text(view, text, len);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start, start_offset);
  gtk_text_buffer_apply_tag(view->text_buffer, get_alt_hidden_link_tag(view),
			    &start, &view->iter);
  g_hash_table_insert(view->link_table,
		      (gpointer)start_offset, G_STRDUP(link));
}


void
bbs_thread_view_append_widget(BBSThreadView *view, GtkWidget *widget,
			      int hborder, int vborder)
{
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkTextChildAnchor *anchor;
  guint anchor_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && GTK_IS_WIDGET(widget));

  vbox = gtk_vbox_new(FALSE, 0);
  hbox = gtk_hbox_new(FALSE, 0);

  gtk_widget_show(widget);
  gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, hborder);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, vborder);
  gtk_widget_show(vbox);

  anchor_offset = gtk_text_iter_get_offset(&view->iter);

  anchor = gtk_text_buffer_create_child_anchor(view->text_buffer, &view->iter);
  gtk_text_view_add_child_at_anchor(GTK_TEXT_VIEW(view), vbox, anchor);

  anchor_offset = gtk_text_iter_get_offset(&view->iter);

  g_hash_table_insert(view->child_table, widget, (gpointer)anchor_offset);
}


typedef struct _AdjustOffsetArgs
{
  int offset;
  GHashTable *hash_table;
  GHashTable *new_hash_table;
} AdjustOffsetArgs;


static void
increment_values(gpointer key, int value, AdjustOffsetArgs *args)
{
  if (value >= args->offset)
    g_hash_table_replace(args->hash_table, key, (gpointer)(value + 1));
}


static gboolean
update_link_table_increment(int offset, char *url, AdjustOffsetArgs *args)
{
  if (offset >= args->offset)
    g_hash_table_insert(args->new_hash_table, (gpointer)(offset + 1), url);
  else
    g_hash_table_insert(args->new_hash_table, (gpointer)offset, url);

  return TRUE;
}


void
bbs_thread_view_insert_widget_at_offset(BBSThreadView *view, GtkWidget *widget,
					int offset)
{
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkTextChildAnchor *anchor;
  int anchor_offset;
  GtkTextIter iter;
  AdjustOffsetArgs args;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && GTK_IS_WIDGET(widget));

  vbox = gtk_vbox_new(FALSE, 0);
  hbox = gtk_hbox_new(FALSE, 0);

  gtk_widget_show(widget);
  gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, 30);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 20);
  gtk_widget_show(vbox);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter, offset);

  anchor = gtk_text_buffer_create_child_anchor(view->text_buffer, &iter);
  gtk_text_view_add_child_at_anchor(GTK_TEXT_VIEW(view), vbox, anchor);

  anchor_offset = gtk_text_iter_get_offset(&iter);

  args.offset = offset;
  args.hash_table = view->child_table;
  g_hash_table_foreach(view->child_table, (GHFunc)increment_values, &args);

  g_hash_table_insert(view->child_table, widget, (gpointer)anchor_offset);

  args.hash_table = view->link_table;
  args.new_hash_table = g_hash_table_new_full(NULL, NULL, NULL, TRACE_FREE);
  g_hash_table_foreach_steal(view->link_table,
			     (GHRFunc)update_link_table_increment, &args);
  g_hash_table_destroy(view->link_table);
  view->link_table = args.new_hash_table;

  gtk_text_buffer_get_end_iter(view->text_buffer, &view->iter);
}


static void
decrement_values(gpointer key, int value, AdjustOffsetArgs *args)
{
  if (value >= args->offset)
    g_hash_table_replace(args->hash_table, key, (gpointer)(value - 1));
}


static gboolean
update_link_table_decrement(int offset, char *url, AdjustOffsetArgs *args)
{
  if (offset >= args->offset)
    g_hash_table_insert(args->new_hash_table, (gpointer)(offset - 1), url);
  else
    g_hash_table_insert(args->new_hash_table, (gpointer)offset, url);

  return TRUE;
}


void
bbs_thread_view_remove_widget(BBSThreadView *view, GtkWidget *widget)
{
  GtkTextIter start;
  GtkTextIter end;
  int anchor_offset;
  AdjustOffsetArgs args;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));
  g_return_if_fail(GTK_IS_WIDGET(widget));

  anchor_offset = (guint)g_hash_table_lookup(view->child_table, widget);
  g_return_if_fail(anchor_offset != 0);

  g_hash_table_remove(view->child_table, widget);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start,
				     anchor_offset - 1);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &end, anchor_offset);
  gtk_text_buffer_delete(view->text_buffer, &start, &end);
  gtk_text_buffer_get_end_iter(view->text_buffer, &view->iter);

  args.offset = anchor_offset - 1;
  args.hash_table = view->child_table;
  g_hash_table_foreach(view->child_table, (GHFunc)decrement_values, &args);

  args.hash_table = view->link_table;
  args.new_hash_table = g_hash_table_new_full(NULL, NULL, NULL, TRACE_FREE);
  g_hash_table_foreach_steal(view->link_table,
			     (GHRFunc)update_link_table_decrement, &args);
  g_hash_table_destroy(view->link_table);
  view->link_table = args.new_hash_table;
}


GtkTextTag *
bbs_thread_view_get_tag_by_name(BBSThreadView *view, const gchar *name)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  return gtk_text_tag_table_lookup(default_tag_table, name);
}


void
bbs_thread_view_push_tag_by_name(BBSThreadView *view, const gchar *name)
{
  GtkTextTag *tag;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));
  
  tag = gtk_text_tag_table_lookup(default_tag_table, name);

  if (tag != NULL)
    bbs_thread_view_push_tag(view, tag);
}


void
bbs_thread_view_push_tag(BBSThreadView *view, GtkTextTag *tag)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (tag == NULL)
    return;

  g_return_if_fail(GTK_IS_TEXT_TAG(tag));

  view->tags = g_slist_prepend(view->tags, tag);
}


GtkTextTag *
bbs_thread_view_pop_tag(BBSThreadView *view)
{
  GtkTextTag *tag;
  GSList *head_entry;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  head_entry = view->tags;

  if (head_entry == NULL)
    return NULL;

  tag = GTK_TEXT_TAG(head_entry->data);
  view->tags = g_slist_remove_link(view->tags, head_entry);
  g_slist_free(head_entry);

  return tag;
}


void
bbs_thread_view_pop_tags(BBSThreadView *view, GtkTextTag *tag)
{
  GtkTextTag *popped_tag;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (tag == NULL)
    return;

  g_return_if_fail(GTK_IS_TEXT_TAG(tag));

  do
    {
      popped_tag = bbs_thread_view_pop_tag(view);
    } while (popped_tag != tag && popped_tag != NULL);
}


static void
bbs_thread_view_scroll_to(BBSThreadView *view, int to_where)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (to_where == SCROLL_TO_BUFFER_START)
    bbs_thread_view_scroll_to_start(view);
  else
    bbs_thread_view_scroll_to_end(view);
}


void
bbs_thread_view_scroll_to_start(BBSThreadView *view)
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  gboolean result;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view));
  g_return_if_fail(buffer != NULL);

  gtk_text_buffer_get_start_iter(buffer, &iter);
  gtk_text_buffer_place_cursor(buffer, &iter);
  result = gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(view),
					&iter, 0.0, FALSE, 0.0, 0.0);
}


void
bbs_thread_view_scroll_to_end(BBSThreadView *view)
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  gboolean result;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view));
  g_return_if_fail(buffer != NULL);

  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_place_cursor(buffer, &iter);
  result = gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(view),
					&iter, 0.0, FALSE, 0.0, 0.0);
}


int
bbs_thread_view_scroll_to_offset(BBSThreadView *view, int offset)
{
  GtkTextIter iter;
  int previous_offset;
  gboolean result;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_buffer != NULL,
		       0);

  gtk_text_view_place_cursor_onscreen(GTK_TEXT_VIEW(view));
  gtk_text_buffer_get_iter_at_mark(view->text_buffer, &iter,
				   gtk_text_buffer_get_insert(view->text_buffer));
  previous_offset = gtk_text_iter_get_offset(&iter);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter, offset);
  gtk_text_buffer_place_cursor(view->text_buffer, &iter);
  result = gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(view),
					&iter, 0.0, FALSE, 0.0, 0.0);

  return previous_offset;
}


void
bbs_thread_view_scroll_to_mark(BBSThreadView *view, GtkTextMark *mark)
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && mark != NULL);

  buffer = gtk_text_mark_get_buffer(mark);
  g_return_if_fail(buffer != NULL);

  gtk_text_buffer_get_iter_at_mark(buffer, &iter, mark);
  gtk_text_buffer_place_cursor(buffer, &iter);

  gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view), mark, 0.1, TRUE, 0.0, 0.1);
}


GtkTextMark *
bbs_thread_view_create_mark_at_offset(BBSThreadView *view, int offset)
{
  GtkTextIter iter;
  GtkTextMark *mark;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_buffer != NULL,
		       NULL);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter, offset);
  mark = gtk_text_buffer_create_mark(view->text_buffer, NULL, &iter, TRUE); 
  return mark;
}


void
bbs_thread_view_place_cursor_at_offset(BBSThreadView *view, int offset)
{
  GtkTextIter iter;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_buffer != NULL);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter, offset);
  gtk_text_buffer_place_cursor(view->text_buffer, &iter);
}


int
bbs_thread_view_get_visible_offset(BBSThreadView *view)
{
  GdkRectangle visible_rect;
  GtkTextIter iter;

  gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(view), &visible_rect);
  gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(view), &iter,
				     visible_rect.x, visible_rect.y);
  return gtk_text_iter_get_offset(&iter);
}


void
bbs_thread_view_make_offset_visible(BBSThreadView *view, int offset)
{
  GtkTextIter iter;
  gboolean result;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_buffer != NULL);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter, offset);
  gtk_text_buffer_place_cursor(view->text_buffer, &iter);
  result = gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(view),
					&iter, 0.0, TRUE, 0.0, 0.0);
}


/* Ϣ */
static void
bbs_thread_view_scroll_to_offsets(BBSThreadView *view,
				  int start_offset, int end_offset)
{
  /* ޥåʬ򤷡ʬɽ褦˥ */
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter,
				     start_offset);
  gtk_text_buffer_move_mark(view->text_buffer, view->insert_mark, &iter);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter,
				     + end_offset);
  gtk_text_buffer_move_mark(view->text_buffer, view->selection_bound_mark,
			    &iter);
  gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view), view->insert_mark,
			       0.1, TRUE, 0.2, 0.2);
}


static gboolean
bbs_thread_view_find_forward_first_match(BBSThreadView *view,
					 GtkTextIter *start_iter,
					 GtkTextIter *end_iter,
					 int *match_start_offset_p,
					 int *match_end_offset_p)
{
  gboolean result = FALSE;
  gchar *text;
  /*
   * MEMO: gtk_text_iter_get_text()䤽֤ϥԡ֤Ƥ뤳Ȥ
   *       աХåեΤоݤ˸ȤʤȡХåեݤȤΥԡ
   *       롣ٹʤΤǤʺ٤ʤȤ̵Ǥ
   */
  g_return_val_if_fail(!view->use_regexp || view->regexp_available, FALSE);

  text = gtk_text_iter_get_slice(start_iter, end_iter);
  g_return_val_if_fail(text != NULL, FALSE);

  if (!view->use_regexp && !view->match_case)
    {
      gchar *tmp_text = g_utf8_casefold(text, -1);
      g_free(text);
#if TRACE_MEMORY_USAGE
      text = G_STRDUP(tmp_text);
      g_free(tmp_text);
#else
      text = tmp_text;
#endif
    }
#if TRACE_MEMORY_USAGE
  else
    {
      gchar *tmp_text = text;
      text = G_STRDUP(text);
      g_free(tmp_text);
    }
#endif

  if (view->use_regexp)
    {
      g_assert(view->regexp_available);

      if (regexec(&view->regexp, text, 1, view->regexp_match, 0) == 0)
	{
	  int offset = gtk_text_iter_get_offset(start_iter);
	  *match_start_offset_p
	    = g_utf8_strlen(text, view->regexp_match[0].rm_so) + offset;
	  *match_end_offset_p
	    = g_utf8_strlen(text + view->regexp_match[0].rm_so,
			    (view->regexp_match[0].rm_eo
			     - view->regexp_match[0].rm_so))
	    + (*match_start_offset_p);
	  result = TRUE;
	}
    }
  else
    {
      gchar *match_start = strstr(text, view->search_key);
      if (match_start != NULL)
	{
	  int offset = gtk_text_iter_get_offset(start_iter);
	  *match_start_offset_p
	    = g_utf8_strlen(text, match_start - text) + offset;
	  *match_end_offset_p
	    = g_utf8_strlen(view->search_key, -1) + (*match_start_offset_p);
	  result = TRUE;
	}
    }

  if (text != NULL)
    G_FREE(text);

  return result;
}


static gboolean
bbs_thread_view_find_backward_first_match(BBSThreadView *view,
					  GtkTextIter *start_iter,
					  GtkTextIter *end_iter,
					  int *match_start_offset_p,
					  int *match_end_offset_p)
{
  gboolean result = FALSE;
  gchar *text;
  /*
   * MEMO: gtk_text_iter_get_text()䤽֤ϥԡ֤Ƥ뤳Ȥ
   *       աХåեΤоݤ˸ȤʤȡХåեݤȤΥԡ
   *       롣ٹʤΤǤʺ٤ʤȤ̵Ǥ
   */
  text = gtk_text_iter_get_slice(start_iter, end_iter);
  g_return_val_if_fail(text != NULL, FALSE);

  if (!view->use_regexp && !view->match_case)
    {
      gchar *tmp_text = g_utf8_casefold(text, -1);
      G_FREE(text);
      text = tmp_text;
    }

  if (view->use_regexp)
    {
      /*
       * MEMO: ɽεսޥåˤϤޤˡ̵ʤΤǡ٤
       *       (ޥåʸ¿)ǰξO(n)regexec()¹Ԥ롣
       */
      gboolean found = FALSE;
      gchar *tmp_pos = text;
      gchar *last_start_pos = NULL;
      gchar *last_end_pos = NULL;

      g_assert(view->regexp_available);

      while (*tmp_pos != '\0'
	     && regexec(&view->regexp, tmp_pos, 1, view->regexp_match, 0) == 0)
	{
	  found = TRUE;
	  last_start_pos = view->regexp_match[0].rm_so + tmp_pos;
	  last_end_pos = view->regexp_match[0].rm_eo + tmp_pos;
	  tmp_pos += view->regexp_match[0].rm_eo;
	}

      if (found)
	{
	  int offset = gtk_text_iter_get_offset(start_iter);
	  *match_start_offset_p
	    = g_utf8_strlen(text, last_start_pos - text) + offset;
	  *match_end_offset_p
	    = g_utf8_strlen(last_start_pos, last_end_pos - last_start_pos)
	    + (*match_start_offset_p);
	  result = TRUE;
	}
    }
  else
    {
      gchar *match_start = g_strrstr(text, view->search_key);
      if (match_start != NULL)
	{
	  int offset = gtk_text_iter_get_offset(start_iter);
	  *match_start_offset_p
	    = g_utf8_strlen(text, match_start - text) + offset;
	  *match_end_offset_p
	    = g_utf8_strlen(view->search_key, -1) + (*match_start_offset_p);
	  result = TRUE;
	}
    }

  if (text != NULL)
    G_FREE(text);

  return result;
}


static gboolean
bbs_thread_view_do_search_forward(BBSThreadView *view,
				  gboolean include_current_pos)
{
  GtkTextIter start;
  GtkTextIter end;
  int start_offset = 0;
  int end_offset = 0;

  if (view->use_regexp)
    {
      if (!view->regexp_available)
	return FALSE;
    }
  else
    {
      if (view->search_key == NULL)
	return FALSE;
    }

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start,
				     view->last_match_offset);
  if (!include_current_pos)
    {
      if (!gtk_text_iter_forward_char(&start))
	{
	  if (view->enable_wrap)
	    view->last_match_offset = 0;
	  return FALSE;
	}
    }
  gtk_text_buffer_get_end_iter(view->text_buffer, &end);

  if (bbs_thread_view_find_forward_first_match(view, &start, &end,
					       &start_offset, &end_offset))
    {
      bbs_thread_view_scroll_to_offsets(view, start_offset, end_offset);
      view->last_match_offset = start_offset;
      return TRUE;
    }
  else if (view->enable_wrap)
    view->last_match_offset = 0;

  return FALSE;
}


static gboolean
bbs_thread_view_do_search_backward(BBSThreadView *view,
				   gboolean include_current_pos)
{
  GtkTextIter start;
  GtkTextIter end;
  int start_offset = 0;
  int end_offset = 0;

  if (view->use_regexp)
    {
      if (!view->regexp_available)
	return FALSE;
    }
  else
    if (view->search_key == NULL)
      return FALSE;

  gtk_text_buffer_get_start_iter(view->text_buffer, &start);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &end,
				     view->last_match_offset);

  if (include_current_pos)
    {
      if (view->use_regexp)
	{
	  /* ʤ */
	  if (!gtk_text_iter_forward_line(&end))
	    gtk_text_buffer_get_end_iter(view->text_buffer, &end);
	}
      else
	gtk_text_iter_forward_chars(&end, g_utf8_strlen(view->search_key, -1));
    }
  else
    {
      if (!gtk_text_iter_backward_char(&end))
	{
	  if (view->enable_wrap)
	    {
	      gtk_text_buffer_get_end_iter(view->text_buffer, &end);
	      view->last_match_offset = gtk_text_iter_get_offset(&end);
	    }
	  return FALSE;
	}
    }

  if (bbs_thread_view_find_backward_first_match(view, &start, &end,
						&start_offset, &end_offset))
    {
      bbs_thread_view_scroll_to_offsets(view, start_offset, end_offset);
      view->last_match_offset = start_offset;
      return TRUE;
    }
  else if (view->enable_wrap)
    {
      GtkTextIter iter;
      gtk_text_buffer_get_end_iter(view->text_buffer, &iter);
      view->last_match_offset = gtk_text_iter_get_offset(&iter);
    }

  return FALSE;
}


static void
bbs_thread_view_unselect_text(BBSThreadView *view)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_mark(view->text_buffer, &iter,
				   view->insert_mark);
  gtk_text_buffer_move_mark(view->text_buffer, view->selection_bound_mark,
			    &iter);
}


gboolean
bbs_thread_view_find(BBSThreadView *view, const gchar *key,
		     BBSThreadViewSearchDirection direction,
		     gboolean enable_wrap, gboolean match_case,
		     gboolean use_regexp)
{
  gboolean result = TRUE;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view) && key != NULL, FALSE);

  /* ǽ˸νԤ */
  view->search_direction = direction;
  view->enable_wrap = enable_wrap;
  view->match_case = match_case;

  if (view->search_key != NULL)
    {
      G_FREE(view->search_key);
      view->search_key = NULL;
    }

  if (use_regexp)
    {
      int regcomp_result;
      if (view->regexp_available)
	{
	  regfree(&view->regexp);
	  if (view->regexp_match != NULL)
	    {
	      G_FREE(view->regexp_match);
	      view->regexp_match = NULL;
	    }
	}

      if (match_case)
	regcomp_result = regcomp(&view->regexp, key,
				 REG_EXTENDED | REG_NEWLINE);
      else
	regcomp_result = regcomp(&view->regexp, key,
				 REG_EXTENDED | REG_ICASE | REG_NEWLINE);
      if (regcomp_result != 0)
	{
	  view->regexp_available = FALSE;
	  regfree(&view->regexp);
	  result = FALSE;
	}
      else
	{
	  view->regexp_available = TRUE;
	  view->regexp_match = G_NEW0(regmatch_t, 1 + view->regexp.re_nsub);
	}
    }
  else
    {
      if (match_case)
	view->search_key = G_STRDUP(key);
      else
#if TRACE_MEMORY_USAGE
	{
	  gchar *tmp_key = g_utf8_casefold(key, -1);
	  view->search_key = G_STRDUP(tmp_key);
	  g_free(tmp_key);
	}
#else
	view->search_key = g_utf8_casefold(key, -1);
#endif
    }
  view->use_regexp = use_regexp;

  if (result)
    {
      if (view->search_direction == BBS_THREAD_VIEW_SEARCH_DIRECTION_FORWARD)
	result = bbs_thread_view_do_search_forward(view, TRUE);
      else
	result = bbs_thread_view_do_search_backward(view, TRUE);
    }

  if (!result)
    bbs_thread_view_unselect_text(view);

  return result;
}


gboolean
bbs_thread_view_find_next(BBSThreadView *view)
{
  gboolean result;
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), FALSE);

  if (view->search_direction == BBS_THREAD_VIEW_SEARCH_DIRECTION_FORWARD)
    result = bbs_thread_view_do_search_forward(view, FALSE);
  else
    result = bbs_thread_view_do_search_backward(view, FALSE);

  if (!result)
    bbs_thread_view_unselect_text(view);

  return result;
}


void
bbs_thread_view_invalidate_search_result(BBSThreadView *view)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));
  bbs_thread_view_unselect_text(view);

  if (view->search_key != NULL)
    {
      G_FREE(view->search_key);
      view->search_key = NULL;
    }

  if (view->regexp_available)
    {
      regfree(&view->regexp);
      view->regexp_available = FALSE;
      if (view->regexp_match != NULL)
	{
	  G_FREE(view->regexp_match);
	  view->regexp_match = NULL;
	}
    }

  view->last_match_offset = 0;
}
