/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: regex_utils.c,v 1.7 2004/03/21 21:35:11 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "regex_utils.h"

#include <gtk/gtk.h>

#if HAVE_ONIG_ONIGURUMA_H
# include <onig/oniguruma.h>
#else
# include <oniguruma.h>
#endif

#if (ONIGURUMA_VERSION_MAJOR >= 2) && (ONIGURUMA_VERSION_MINOR > 0)
# define ONIGURUMA_2_2_API		1
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static void regex_editor_class_init(RegexEditorClass *klass);
static void regex_editor_init(RegexEditor *editor);
static void regex_editor_finalize(GObject *object);
static void regex_editor_destroy(GtkObject *object);


GType
regex_editor_get_type(void)
{
  static GType regex_editor_type = 0;

  if (regex_editor_type == 0)
    {
      static const GTypeInfo regex_editor_info =
	{
	  sizeof(RegexEditorClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)regex_editor_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(RegexEditor),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)regex_editor_init,
	};

      regex_editor_type = g_type_register_static(GTK_TYPE_VBOX,
						 "RegexEditor",
						 &regex_editor_info, 0);
    }

  return regex_editor_type;
}


static GtkVBoxClass *parent_class;


static void
regex_editor_class_init(RegexEditorClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = regex_editor_finalize;
  object_class->destroy = regex_editor_destroy;
}


static void
regex_editor_init(RegexEditor *editor)
{
  GtkTextBuffer *regex_buffer = gtk_text_buffer_new(NULL);
  GtkWidget *regex_view = gtk_text_view_new_with_buffer(regex_buffer);
  GtkWidget *message = gtk_label_new("");
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#if 0	/* themeǤˤޤ٤ */
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window),
				      GTK_SHADOW_ETCHED_IN);
#endif
  gtk_widget_show(regex_view);
  gtk_container_add(GTK_CONTAINER(scrolled_window), regex_view);
  gtk_widget_show(scrolled_window);
  gtk_box_pack_start(GTK_BOX(editor), scrolled_window, TRUE, TRUE, 0);

  gtk_widget_show(message);
  gtk_box_pack_start(GTK_BOX(hbox), message, TRUE, TRUE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(editor), hbox, FALSE, FALSE, 0);

  editor->regex_buffer = regex_buffer;
  editor->message = GTK_LABEL(message);
}


static void
regex_editor_finalize(GObject *object)
{
  RegexEditor *editor = REGEX_EDITOR(object);

  if (editor->expression != NULL)
    {
#ifdef ONIGURUMA_2_2_API
      onig_free((regex_t *)editor->expression);
#else
      regex_free((regex_t *)editor->expression);
#endif
      editor->expression = NULL;
    }
  if (editor->regex_buffer != NULL)
    {
      OCHU_OBJECT_UNREF(editor->regex_buffer);
      editor->regex_buffer = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
regex_editor_destroy(GtkObject *object)
{
  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


/*
 * Ϳ줿textγƹԤbranchȤƷҤǰĤɽˤޤȤ᤿
 * ʸ֤
 *
 * ()ɽȤʤԤϼΤƤ롣
 */
static gchar *
regex_editor_synthesize_regular_expression(const gchar *text)
{
  gchar *result = NULL;
  gchar *tmp_pos;

  g_return_val_if_fail(text != NULL, G_STRDUP(""));
  if (*text == '\0')
    return G_STRDUP("");

  tmp_pos = (gchar *)text;	/* ޤä㤤*/
  while (*tmp_pos != '\0')
    {
      regex_t *regex;
      gchar *end_pos = strchr(tmp_pos, '\n');
      if (end_pos == NULL)
	end_pos = tmp_pos + strlen(tmp_pos) - 1;
      else
	end_pos--;	/* '\n'μޤ */

      if (tmp_pos == end_pos)
	goto next_line;

#ifdef ONIGURUMA_2_2_API
      if (onig_new(&regex, tmp_pos, end_pos, ONIG_OPTION_EXTEND,
		   ONIG_ENCODING_UTF8,
		   ONIG_SYNTAX_POSIX_EXTENDED, NULL) == ONIG_NORMAL)
#else
      if (regex_new(&regex, tmp_pos, end_pos, REG_OPTION_EXTEND,
# ifdef REG_ENCODING_UTF8
		    REG_ENCODING_UTF8,
# else /* old style */
		    REGCODE_UTF8,
# endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) == REG_NORMAL)
#endif
	{
	  gchar *sub_exp = G_STRNDUP(tmp_pos, end_pos - tmp_pos + 1);
	  if (result == NULL)
	    result = sub_exp;
	  else
	    {
	      /* ٹ롣*/
	      gchar *old_result = result;
	      result = g_strconcat(old_result, "|", sub_exp, NULL);
	      G_FREE(old_result);
	      G_FREE(sub_exp);
	    }
	}

#ifdef ONIGURUMA_2_2_API
      onig_free(regex);
#else
      regex_free(regex);
#endif

    next_line:
      tmp_pos = end_pos + 1;
      if (*tmp_pos == '\n')
	tmp_pos++;	/* '\n'ɤФ */
    }

  if (result == NULL)
    return G_STRDUP("");

  return result;
}


GtkWidget *
regex_editor_new(const gchar *regex_text)
{
  RegexEditor *editor = REGEX_EDITOR(g_object_new(REGEX_EDITOR_TYPE, NULL));

  if (regex_text != NULL)
    regex_editor_set_regex_text(editor, regex_text);

  return GTK_WIDGET(editor);
}


void
regex_editor_set_regex_text(RegexEditor *editor, const gchar *text)
{
  GtkTextIter current_iter;
  GtkTextIter end_iter;

  g_return_if_fail(IS_REGEX_EDITOR(editor));

  gtk_text_buffer_get_start_iter(editor->regex_buffer, &current_iter);
  gtk_text_buffer_get_end_iter(editor->regex_buffer, &end_iter);

  gtk_text_buffer_delete(editor->regex_buffer, &current_iter, &end_iter);

  if (text != NULL && *text != '\0')
    {
      /* branchԤˤФ餷ƥХåե */
      int nest = 0;
      gchar *branch_start = (gchar *)text;	/* ޤä㤤 */
      gchar *tmp_pos = branch_start;
      while (*tmp_pos != '\0')
	{
	  if (*tmp_pos == '\\')
	    tmp_pos++;
	  else if (*tmp_pos == '(')
	    nest++;
	  else if (*tmp_pos == ')')
	    nest--;
	  else if (*tmp_pos == '|' && nest == 0)
	    {
	      int len = tmp_pos - branch_start;
	      if (len > 0)
		{
		  gtk_text_buffer_insert(editor->regex_buffer, &current_iter,
					 branch_start, len);
		  gtk_text_buffer_insert(editor->regex_buffer, &current_iter,
					 "\n", -1);
		}
	      branch_start = tmp_pos + 1;
	    }
	  tmp_pos++;
	}

      if (*branch_start != '\0' && branch_start < tmp_pos)
	{
	  int len = tmp_pos - branch_start;
	  if (len > 0)
	    gtk_text_buffer_insert(editor->regex_buffer, &current_iter,
				   branch_start, len);
	}
    }
}


gchar *
regex_editor_get_regex_text(RegexEditor *editor)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  gchar *buffer_text;
  gchar *result;

  g_return_val_if_fail(IS_REGEX_EDITOR(editor), NULL);

  gtk_text_buffer_get_start_iter(editor->regex_buffer, &start_iter);
  gtk_text_buffer_get_end_iter(editor->regex_buffer, &end_iter);
  buffer_text = gtk_text_buffer_get_text(editor->regex_buffer, &start_iter,
					 &end_iter, FALSE);
  result = regex_editor_synthesize_regular_expression(buffer_text);
  g_free(buffer_text);

  return result;
}


gpointer
oniguruma_regex_new(const gchar *pattern, const char *encoding)
{
  regex_t *result;
#ifdef ONIGURUMA_2_2_API
  OnigEncoding code;
#else
  RegCharEncoding code;
#endif

  g_return_val_if_fail(pattern != NULL && *pattern != '\0', NULL);

  if (encoding != NULL)
    {
#ifdef ONIGURUMA_2_2_API
      if (strcmp(encoding, "CP932") == 0)
	code = ONIG_ENCODING_SJIS;
      else if (strcmp(encoding, "EUC-JP") == 0)
	code = ONIG_ENCODING_EUC_JP;
      else if (strcmp(encoding, "UTF-8") == 0)
	code = ONIG_ENCODING_UTF8;
      else if (strcmp(encoding, "Shift_JIS") == 0)
	code = ONIG_ENCODING_SJIS;
      else
	code = ONIG_ENCODING_UTF8;
    }
  else
    code = ONIG_ENCODING_UTF8;
#else
# ifdef REG_ENCODING_UNDEF
      if (strcmp(encoding, "CP932") == 0)
	code = REG_ENCODING_SJIS;
      else if (strcmp(encoding, "EUC-JP") == 0)
	code = REG_ENCODING_EUC_JP;
      else if (strcmp(encoding, "UTF-8") == 0)
	code = REG_ENCODING_UTF8;
      else if (strcmp(encoding, "Shift_JIS") == 0)
	code = REG_ENCODING_SJIS;
      else
	code = REG_ENCODING_UTF8;
    }
  else
    code = REG_ENCODING_UTF8;
# else /* old style */
      if (strcmp(encoding, "CP932") == 0)
	code = REGCODE_SJIS;
      else if (strcmp(encoding, "EUC-JP") == 0)
	code = REGCODE_EUCJP;
      else if (strcmp(encoding, "UTF-8") == 0)
	code = REGCODE_UTF8;
      else if (strcmp(encoding, "Shift_JIS") == 0)
	code = REGCODE_SJIS;
      else
	code = REGCODE_UTF8;
    }
  else
    code = REGCODE_UTF8;
# endif
#endif

#ifdef ONIGURUMA_2_2_API
  if (onig_new(&result, (UChar *)pattern, (UChar *)pattern + strlen(pattern),
	       ONIG_OPTION_EXTEND, code,
	       ONIG_SYNTAX_POSIX_EXTENDED, NULL) != ONIG_NORMAL)
    {
      onig_free(result);
      fprintf(stderr, "Invalid regular expression: \"%s\"\n", pattern);
      return NULL;
    }
#else
  if (regex_new(&result, (UChar *)pattern, (UChar *)pattern + strlen(pattern),
		REG_OPTION_EXTEND, code,
		REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
    {
      regex_free(result);
      fprintf(stderr, "Invalid regular expression: \"%s\"\n", pattern);
      return NULL;
    }
#endif

  return result;
}


void
oniguruma_regex_free(gpointer oniguruma_regex)
{
  g_return_if_fail(oniguruma_regex != NULL);

#ifdef ONIGURUMA_2_2_API
  onig_free((regex_t *)oniguruma_regex);
#else
  regex_free((regex_t *)oniguruma_regex);
#endif
}


gboolean
oniguruma_regex_match(gpointer oniguruma_regex, const char *text)
{
  UChar *text_end;

  g_return_val_if_fail(oniguruma_regex != NULL, FALSE);
  g_return_val_if_fail(text != NULL, FALSE);

  text_end = (UChar *)text + strlen(text);

#ifdef ONIGURUMA_2_2_API
  return onig_search((regex_t *)oniguruma_regex,
		     (UChar *)text, text_end,
		     (UChar *)text, text_end, NULL, 0) >= 0;
#else
  return regex_search((regex_t *)oniguruma_regex,
		      (UChar *)text, text_end,
		      (UChar *)text, text_end, NULL, 0) >= 0;
#endif
}
