/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_bbs_table.c,v 1.27 2004/04/13 15:38:05 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_table.h"
#include "ochusha_board_2ch.h"
#include "ochusha_board_jbbs.h"
#include "ochusha_board_category.h"
#include "ochusha_bulletin_board.h"
#include "ochusha_utils_2ch.h"

#include "marshal.h"

#include <glib.h>
#include <libxml/SAX.h>
#include <libxml/parser.h>

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <zlib.h>


static void ochusha_bbs_table_class_init(OchushaBBSTableClass *klass);
static void ochusha_bbs_table_init(OchushaBBSTable *table);
static void ochusha_bbs_table_finalize(GObject *object);


GType
ochusha_bbs_table_get_type(void)
{
  static GType bbs_table_type = 0;

  if (bbs_table_type == 0)
    {
      static const GTypeInfo bbs_table_info =
	{
	  sizeof(OchushaBBSTableClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_bbs_table_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBBSTable),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_bbs_table_init,
	};

      bbs_table_type = g_type_register_static(G_TYPE_OBJECT,
					      "OchushaBBSTable",
					      &bbs_table_info, 0);
    }

  return bbs_table_type;
}


enum {
  CATEGORY_NEW_SIGNAL,
  BOARD_NEW_SIGNAL,
  BOARDLIST_READ_CATEGORY_ELEMENT_SIGNAL,
  BOARDLIST_WRITE_CATEGORY_ELEMENT_SIGNAL,
  BOARDLIST_READ_BOARD_ELEMENT_SIGNAL,
  BOARDLIST_WRITE_BOARD_ELEMENT_SIGNAL,
  LAST_SIGNAL,
};


static GObjectClass *parent_class = NULL;
static int bbs_table_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0 };


static void
ochusha_bbs_table_class_init(OchushaBBSTableClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;

  parent_class = g_type_class_peek_parent(klass);
  o_class->finalize = ochusha_bbs_table_finalize;

  bbs_table_signals[CATEGORY_NEW_SIGNAL] =
    g_signal_new("category_new",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass, category_new),
		 NULL, NULL,
		 libochusha_marshal_OBJECT__STRING,
		 OCHUSHA_TYPE_BOARD_CATEGORY, 1,
		 G_TYPE_STRING);
  bbs_table_signals[BOARD_NEW_SIGNAL] =
    g_signal_new("board_new",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass, board_new),
		 NULL, NULL,
		 libochusha_marshal_OBJECT__INT_STRING_STRING,
		 OCHUSHA_TYPE_BULLETIN_BOARD, 3,
		 G_TYPE_INT,
		 G_TYPE_STRING,
		 G_TYPE_STRING);

  bbs_table_signals[BOARDLIST_READ_CATEGORY_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_read_category_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_read_category_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BOARD_CATEGORY,
		 G_TYPE_POINTER);
  bbs_table_signals[BOARDLIST_WRITE_CATEGORY_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_write_category_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_write_category_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BOARD_CATEGORY,
		 G_TYPE_POINTER);
  bbs_table_signals[BOARDLIST_READ_BOARD_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_read_board_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_read_board_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BULLETIN_BOARD,
		 G_TYPE_POINTER);
  bbs_table_signals[BOARDLIST_WRITE_BOARD_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_write_board_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_write_board_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BULLETIN_BOARD,
		 G_TYPE_POINTER);

  klass->category_new = NULL;
  klass->board_new = NULL;
  klass->boardlist_read_category_element = NULL;
  klass->boardlist_write_category_element = NULL;
  klass->boardlist_read_board_element = NULL;
  klass->boardlist_write_board_element = NULL;
}


static void
ochusha_bbs_table_init(OchushaBBSTable *table)
{
  table->category_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)OCHU_OBJECT_UNREF_FUNC);
  table->board_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)OCHU_OBJECT_UNREF_FUNC);
  table->board_name_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)OCHU_OBJECT_UNREF_FUNC);
}


static void
ochusha_bbs_table_finalize(GObject *object)
{
  OchushaBBSTable *table;
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(object));

  table = (OchushaBBSTable *)object;

  if (table->category_list != NULL)
    {
      g_slist_free(table->category_list);
      table->category_list = NULL;
    }

  if (table->category_table != NULL)
    {
      g_hash_table_destroy(table->category_table);
      table->category_table = NULL;
    }

  if (table->board_table != NULL)
    {
      g_hash_table_destroy(table->board_table);
      table->board_table = NULL;
    }

  if (table->board_name_table != NULL)
    {
      g_hash_table_destroy(table->board_name_table);
      table->board_name_table = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


OchushaBBSTable *
ochusha_bbs_table_new(void)
{
  return OCHUSHA_BBS_TABLE(g_object_new(OCHUSHA_TYPE_BBS_TABLE, NULL));
}


void
ochusha_bbs_table_prepend_category(OchushaBBSTable *table,
				   OchushaBoardCategory *category)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BOARD_CATEGORY(category));

  if (g_hash_table_lookup(table->category_table, category->name) != NULL)
    { /* ϿѤξ */
      table->category_list = g_slist_remove(table->category_list, category);
    }

  OCHU_OBJECT_REF(category);
  g_hash_table_insert(table->category_table, category->name, category);

  table->category_list = g_slist_prepend(table->category_list, category);
}


void
ochusha_bbs_table_add_category(OchushaBBSTable *table,
			       OchushaBoardCategory *category)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BOARD_CATEGORY(category));

  if (g_hash_table_lookup(table->category_table, category->name) != NULL)
    { /* ϿѤξ */
      table->category_list = g_slist_remove(table->category_list, category);
    }

  OCHU_OBJECT_REF(category);
  g_hash_table_insert(table->category_table, category->name, category);

  table->category_list = g_slist_append(table->category_list, category);
}


void
ochusha_bbs_table_remove_category(OchushaBBSTable *table,
				  OchushaBoardCategory *category)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BOARD_CATEGORY(category));
  g_return_if_fail(g_hash_table_lookup(table->category_table, category->name)
		   != NULL);
  g_hash_table_remove(table->category_table, category);
}


OchushaBoardCategory *
ochusha_bbs_table_lookup_category(OchushaBBSTable *table, const gchar *name)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table), NULL);

  return g_hash_table_lookup(table->category_table, name);
}


void
ochusha_bbs_table_add_board(OchushaBBSTable *table,
			    OchushaBulletinBoard *board)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BULLETIN_BOARD(board));

  OCHU_OBJECT_REF(board);
  g_hash_table_insert(table->board_table, board->base_url, board);

  OCHU_OBJECT_REF(board);
  g_hash_table_insert(table->board_name_table, board->name, board);

  return;
}


void
ochusha_bbs_table_remove_board(OchushaBBSTable *table,
			       OchushaBulletinBoard *board)
{
  OchushaBulletinBoard *entry;

  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BULLETIN_BOARD(board));

  OCHU_OBJECT_REF(board);

  entry = g_hash_table_lookup(table->board_table, board->base_url);
  if (entry == board)
    g_hash_table_remove(table->board_table, board->base_url);

  entry = g_hash_table_lookup(table->board_name_table, board->name);
  if (entry == board)
    g_hash_table_remove(table->board_name_table, board->name);
}


OchushaBulletinBoard *
ochusha_bbs_table_lookup_board_by_url(OchushaBBSTable *table, const char *url)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && url != NULL, NULL);

  return g_hash_table_lookup(table->board_table, url);
}


OchushaBulletinBoard *
ochusha_bbs_table_lookup_board_by_name(OchushaBBSTable *table,
				       const gchar *name)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && name != NULL, NULL);

  return g_hash_table_lookup(table->board_name_table, name);
}


/*
 * Ϳ줿URLϤбƤʷǼĤΤΤɤ
 * Ƚꤹ롣
 *
 * Ƚ˲äʲΤ褦Ѥ򵯤
 *
 * ȤͿ줿board_pNULLʾ:
 *   URLǼ줿ڡǼºݤбƤǼĤΤΤǡ
 *   ġĤФOchushaBulletinBoard֥ȤtableϿ
 *   ˤϡboard_pˤΥݥ󥿤Ǽ롣
 * ȤͿ줿board_url_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ĤURLȤ餷ʸΥԡ
 *   board_url_p˳Ǽ롣
 * ȤͿ줿bbs_type_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ļ̤Ȥ餷ͤ
 *   bbs_type_p˳Ǽ롣
 * ȤͿ줿thread_id_pfrom_pto_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ硢
 *   ID(thread_id)䡢쥹ϰϻƬ(from)(to)򤽤줾
 *   thread_id_pfrom_pto_p˳Ǽ롣thread_id_p˳ǼΤ
 *   IDʸΥԡ
 */
gboolean
ochusha_bbs_table_check_url(OchushaBBSTable *table, const char *url,
			    OchushaBulletinBoard **board_p,
			    char **board_url_p, int *bbs_type_p,
			    char **thread_id_p,
			    unsigned int *from_p, unsigned int *to_p,
			    gboolean *is_kako_html_p)
{
  char *board_url = NULL;
  gboolean result;

  if ((result = ochusha_utils_2ch_check_url(url, &board_url, bbs_type_p,
					    NULL, thread_id_p, from_p, to_p,
					    is_kako_html_p)))
    {
      if (board_p != NULL)
	{
	  *board_p = ochusha_bbs_table_lookup_board_by_url(table, board_url);
	  if (bbs_type_p != NULL)
	    {
	      if (*board_p != NULL)
		*bbs_type_p = (*board_p)->bbs_type;
	      else if (*bbs_type_p == OCHUSHA_BBS_TYPE_2CH)
		*bbs_type_p = OCHUSHA_BBS_TYPE_2CH_COMPATIBLE;
	    }
	}
    }

  if (board_url_p != NULL)
    *board_url_p = board_url;
  else if (board_url != NULL)
    G_FREE(board_url);

  return result;
}


typedef enum
{
  SAX_INITIAL,
  SAX_OCHUSHA,
  SAX_BOARDLIST,
  SAX_CATEGORY,
  SAX_CATEGORY_ATTRIBUTE,
  SAX_CATEGORY_ATTRIBUTE_BOOLEAN,
  SAX_CATEGORY_ATTRIBUTE_INT,
  SAX_CATEGORY_ATTRIBUTE_STRING,
  SAX_CATEGORY_BOARD,
  SAX_CATEGORY_BOARD_ATTRIBUTE,
  SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN,
  SAX_CATEGORY_BOARD_ATTRIBUTE_INT,
  SAX_CATEGORY_BOARD_ATTRIBUTE_STRING,
  SAX_ACCEPTED,
  SAX_ERROR
} SAXState;


typedef struct _SAXContext
{
  SAXState state;
  OchushaBBSTable *table;

  char *current_attr_name;
  char *current_attr_val;

  GSList *board_list;

  GHashTable *category_attributes;
  GHashTable *board_attributes;
} SAXContext;


static void
board_list_free(GSList *board_list)
{
  g_slist_foreach(board_list, (GFunc)OCHU_OBJECT_UNREF_FUNC, NULL);
  g_slist_free(board_list);
}


static void
cleanup_sax_context(SAXContext *context)
{
  if (context->current_attr_name != NULL)
    {
      G_FREE(context->current_attr_name);
      context->current_attr_name = NULL;
    }

  if (context->current_attr_val != NULL)
    {
      G_FREE(context->current_attr_val);
      context->current_attr_val = NULL;
    }

  if (context->board_list != NULL)
    {
      board_list_free(context->board_list);
      context->board_list = NULL;
    }

  if (context->category_attributes != NULL)
    {
      g_hash_table_destroy(context->category_attributes);
      context->category_attributes = NULL;
    }

  if (context->board_attributes != NULL)
    {
      g_hash_table_destroy(context->board_attributes);
      context->board_attributes = NULL;
    }
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
startElementHandler(void *context, const xmlChar *name, const xmlChar **atts)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_INITIAL:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_OCHUSHA:
      if (strcmp(name, "boardlist") == 0)
	{
	  sax_context->category_attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->board_attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->state = SAX_BOARDLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY:
    case SAX_CATEGORY_BOARD:
      if (strcmp(name, "attribute") == 0
	  && atts != NULL && strcmp(atts[0], "name") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE;
	      break;

	    case SAX_CATEGORY_BOARD:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE;
	      break;

	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
	  if (sax_context->current_attr_val != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "unexpected attribute found: %s=%s\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val);
#endif
	      G_FREE(sax_context->current_attr_name);
	      G_FREE(sax_context->current_attr_val);
	      sax_context->current_attr_name = NULL;
	      sax_context->current_attr_val = NULL;
	      break;	/* 顼 */
	    }
	  sax_context->current_attr_name = G_STRDUP(atts[1]);
	  return;
	}
      else if (sax_context->state == SAX_CATEGORY
	       && strcmp(name, "board") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE:
    case SAX_CATEGORY_BOARD_ATTRIBUTE:
      if (atts != NULL && strcmp(atts[0], "val") == 0)
	{
	  /* int/booleanβǽ */
	  if (strcmp(name, "int") == 0)
	    {
	      switch (sax_context->state)
		{
		case SAX_CATEGORY_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_ATTRIBUTE_INT;
		  break;

		case SAX_CATEGORY_BOARD_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_INT;
		  break;

		default:
		  fprintf(stderr, "Wrong implementation found.\n");
		  abort();
		}
	    }
	  else if (strcmp(name, "boolean") == 0)
	    {
	      switch (sax_context->state)
		{
		case SAX_CATEGORY_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_ATTRIBUTE_BOOLEAN;
		  break;

		case SAX_CATEGORY_BOARD_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN;
		  break;

		default:
		  fprintf(stderr, "Wrong implementation found.\n");
		  abort();
		}
	    }
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "element unexpected in state(%d) found: %s\n",
		      sax_context->state, name);
#endif
	      break;	/* 顼 */
	    }

	  if (sax_context->current_attr_val != NULL)
	    {
	      /* 顼ǤɤΤġ̯ */
#if DEBUG_SAX_HANDLER
	      fprintf(stderr,
		      "attribute %s=\"%s\" is overwritten by \"%s\"!\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val, atts[1]);
#endif
	      G_FREE(sax_context->current_attr_val);
	    }

	  sax_context->current_attr_val = G_STRDUP(atts[1]);
	  return;
	}
      else if (strcmp(name, "string") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY_ATTRIBUTE:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE_STRING;
	      return;

	    case SAX_CATEGORY_BOARD_ATTRIBUTE:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_STRING;
	      return;

	    default:
	      break;	/* 顼 */
	    }
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "string element is unexpected in state(%d).\n",
		  sax_context->state);
#endif
	}
      break;	/* 顼 */

    case SAX_BOARDLIST:
      if (strcmp(name, "category") == 0)
	{
	  if (sax_context->board_list != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "free board_list found!\n");
#endif
	      board_list_free(sax_context->board_list);
	    }
	  sax_context->board_list = NULL;
	  sax_context->state = SAX_CATEGORY;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_STRING:
    case SAX_CATEGORY_ATTRIBUTE_BOOLEAN:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_INT:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
      fprintf(stderr, "startHandler is called in unknown state: %d\n",
	      sax_context->state);
    }
  sax_context->state = SAX_ERROR;
}


static gboolean
hash_table_cleanup_func(gpointer key, gpointer value, gpointer unused)
{
  return TRUE;
}


static void
hash_table_cleanup(GHashTable *hash_table)
{
  g_hash_table_foreach_remove(hash_table, hash_table_cleanup_func, NULL);
}


static void
endElementHandler(void *context, const xmlChar *name)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_OCHUSHA:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_ACCEPTED; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE:
    case SAX_CATEGORY_BOARD_ATTRIBUTE:
      if (strcmp(name, "attribute") == 0)
	{
	  GHashTable *hash_table;
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY_ATTRIBUTE:
	      hash_table = sax_context->category_attributes;
	      sax_context->state = SAX_CATEGORY;
	      break;

	    case SAX_CATEGORY_BOARD_ATTRIBUTE:
	      hash_table = sax_context->board_attributes;
	      sax_context->state = SAX_CATEGORY_BOARD;
	      break;

	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
#if DEBUG_SAX_HANDLER_NEW
	  fprintf(stderr, "%s = \"%s\"\n", sax_context->current_attr_name,
		  sax_context->current_attr_val);
#endif
	  g_hash_table_insert(hash_table,
			      sax_context->current_attr_name,
			      sax_context->current_attr_val);
	  sax_context->current_attr_name = NULL;
	  sax_context->current_attr_val = NULL;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_STRING:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
      if (strcmp(name, "string") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY_ATTRIBUTE_STRING:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE;
	      break;

	    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE;
	      break;

	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
	  if (sax_context->current_attr_val == NULL)
	    sax_context->current_attr_val = G_STRDUP("");
	  return;
	}
      break;	/* 顼 */

    case SAX_BOARDLIST:
      if (strcmp(name, "boardlist") == 0)
	{
	  g_hash_table_destroy(sax_context->category_attributes);
	  sax_context->category_attributes = NULL;

	  g_hash_table_destroy(sax_context->board_attributes);
	  sax_context->board_attributes = NULL;

	  sax_context->state = SAX_OCHUSHA;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY:
      if (strcmp(name, "category") == 0)
	{
	  const gchar *category_name
	    = g_hash_table_lookup(sax_context->category_attributes, "name");
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "closing category\n");
#endif
	  if (sax_context->board_list != NULL)
	    {
	      OchushaBBSTable *table = sax_context->table;
	      OchushaBoardCategory *category
		= ochusha_bbs_table_lookup_category(table, category_name);

	      if (category == NULL)
		{
		  g_signal_emit(G_OBJECT(table),
				bbs_table_signals[CATEGORY_NEW_SIGNAL],
				0, category_name, &category);
		  if (category == NULL)
		    category = ochusha_board_category_new(category_name);
		}
	      else
		OCHU_OBJECT_REF(category);

#if DEBUG_LIBOCHUSHA_MORE
	      fprintf(stderr, "emitting read_boardlist_element for category.\n");
#endif
	      g_signal_emit_by_name(G_OBJECT(category),
				    "read_boardlist_element",
				    sax_context->category_attributes);
#if DEBUG_LIBOCHUSHA_MORE
	      fprintf(stderr, "emitting boardlist_read_category_element.\n");
#endif
	      g_signal_emit(G_OBJECT(table),
			    bbs_table_signals[BOARDLIST_READ_CATEGORY_ELEMENT_SIGNAL],
			    0,
			    category,
			    sax_context->category_attributes);

	      ochusha_board_category_append_boards(category,
						   sax_context->board_list);
	      sax_context->board_list = NULL;
	      ochusha_bbs_table_add_category(table, category);
	      OCHU_OBJECT_UNREF(category);
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "adding category: %s\n", category_name);
#endif
	    }
	  hash_table_cleanup(sax_context->category_attributes);
	  sax_context->state = SAX_BOARDLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_CATEGORY_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_CATEGORY_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD:
      if (strcmp(name, "board") == 0)
	{
	  gchar *board_name
	    = g_hash_table_lookup(sax_context->board_attributes, "name");
	  char *base_url
	    = g_hash_table_lookup(sax_context->board_attributes, "base_url");
	  OchushaBBSTable *table = sax_context->table;
	  OchushaBulletinBoard *board
	    = ochusha_bbs_table_lookup_board_by_url(table, base_url);
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "closing category/board\n");
#endif
	  if (board == NULL)
	    {
	      /*
	       * MEMO: deserializeݥȡǤbbs_typeΤߤˤꡢ
	       *       OchushaBulletinBoard֥Ȥconcreteʥ饹
	       *       Ƥ롣
	       */
	      int bbs_type = ochusha_utils_get_attribute_int(
						sax_context->board_attributes,
						"bbs_type");

	      g_signal_emit(G_OBJECT(table),
			    bbs_table_signals[BOARD_NEW_SIGNAL],
			    0, bbs_type, board_name, base_url, &board);

	      if (board == NULL)
		{
		  switch (bbs_type)
		    {
		    case OCHUSHA_BBS_TYPE_2CH:
		    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
		    case OCHUSHA_BBS_TYPE_2CH_HEADLINE:
		    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
		      board = ochusha_board_2ch_new(board_name, base_url);
		      break;

		    case OCHUSHA_BBS_TYPE_JBBS:
		    case OCHUSHA_BBS_TYPE_MACHIBBS:
		    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
		    case OCHUSHA_BBS_TYPE_MITINOKU:
		      board = ochusha_board_jbbs_new(board_name, base_url);
		      break;

		    default:
		      /* board = NULL;  XXX ȤꤢĤ̵ */
		      break;
		    }
		}

	      if (board != NULL)
		{
		  ochusha_bulletin_board_set_bbs_type(board, bbs_type);
#if DEBUG_SAX_HANDLER
		  fprintf(stderr, "adding board: %s\n", board_name);
#endif

		  g_signal_emit_by_name(G_OBJECT(board),
					"read_boardlist_element",
					sax_context->board_attributes);
		  g_signal_emit(G_OBJECT(table),
				bbs_table_signals[BOARDLIST_READ_BOARD_ELEMENT_SIGNAL],
				0,
				board,
				sax_context->board_attributes);

		  ochusha_bbs_table_add_board(table, board);
		  OCHU_OBJECT_UNREF(board);
		}
	    }

	  if (board != NULL)
	    sax_context->board_list
	      = g_slist_append(sax_context->board_list, board);

	  hash_table_cleanup(sax_context->board_attributes);
	  sax_context->state = SAX_CATEGORY;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_INITIAL:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
#if DEBUG_SAX_HANDLER
      fprintf(stderr, "endHandler called in unknown state: %d.\n",
	      sax_context->state);
#endif
      break;	/* 顼 */
    }
#if DEBUG_SAX_HANDLER
  fprintf(stderr, "Invalid document: </%s> appeared in state=%d\n",
	  name, sax_context->state);
#endif
  sax_context->state = SAX_ERROR;
}


static void
charactersHandler(void *context, const xmlChar *ch, int len)
{
  SAXContext *sax_context = (SAXContext *)context;

  if (sax_context->state == SAX_CATEGORY_ATTRIBUTE_STRING
      || sax_context->state == SAX_CATEGORY_BOARD_ATTRIBUTE_STRING)
    {
      if (sax_context->current_attr_val == NULL)
	sax_context->current_attr_val = G_STRNDUP(ch, len);
      else
	{
	  int old_len = strlen(sax_context->current_attr_val);
	  sax_context->current_attr_val
	    = G_REALLOC(sax_context->current_attr_val, old_len + len + 1);
	  strncat(sax_context->current_attr_val + old_len, ch, len);
	}
    }
}


static void
nopHandler(void *context)
{
}


static xmlEntityPtr
getEntityHandler(void *context, const xmlChar *name)
{
  return NULL;
}


gboolean
ochusha_bbs_table_read_boardlist_xml(OchushaBBSTable *table,
				     OchushaConfig *config,
				     const char *subdir)
{
  SAXContext context =
    {
      SAX_INITIAL,	/* state */
      table,		/* table */
      NULL, NULL,	/* current_attr_name, current_attr_val */
      NULL,		/* board_list */
      NULL,		/* category_attributes */
      NULL		/* board_attributes */
    };
  xmlSAXHandler sax_handler;
  char *pathname;
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && config != NULL, FALSE);

  pathname = ochusha_config_find_file(config, OCHUSHA_BOARDLIST_XML, subdir);
  if (pathname == NULL)
    return FALSE;

  memset(&sax_handler, 0, sizeof(xmlSAXHandler));
#if LIBXML_VERSION >= 20600
  xmlSAX2InitDefaultSAXHandler(&sax_handler, TRUE);
#else
  initxmlDefaultSAXHandler(&sax_handler, TRUE);
#endif

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
#if LIBXML_VERSION >= 20600
  sax_handler.startElementNs = NULL;
  sax_handler.endElementNs = NULL;
#endif
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  cleanup_sax_context(&context);

  if (context.state == SAX_ACCEPTED)
    {
      G_FREE(pathname);
      return TRUE;
    }

  fprintf(stderr, "%s is unacceptable as ochusha's boardlist.\n", pathname);
  return FALSE;
}


typedef struct _WriteBoardlistArgs
{
  OchushaBBSTable *table;
  gzFile boardlist_xml;
} WriteBoardlistArgs;


static void
write_board(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(data);
  WriteBoardlistArgs *args = (WriteBoardlistArgs *)user_data;
  gzFile boardlist_xml = args->boardlist_xml;

  if (board->killed)
    return;

  gzprintf(boardlist_xml, "      <board>\n");

  g_signal_emit_by_name(G_OBJECT(board),
			"write_boardlist_element",
			boardlist_xml);
#if DEBUG_LIBOCHUSHA_MORE
  fprintf(stderr, "table=%p, board=%p, boardlist_xml=%p\n",
	  args->table, board, boardlist_xml);
#endif
  g_signal_emit(G_OBJECT(args->table),
		bbs_table_signals[BOARDLIST_WRITE_BOARD_ELEMENT_SIGNAL],
		0,
		board,
		boardlist_xml);

  gzprintf(boardlist_xml, "      </board>\n");
}


static void
write_category(gpointer data, gpointer user_data)
{
  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(data);
  WriteBoardlistArgs *args = (WriteBoardlistArgs *)user_data;
  gzFile boardlist_xml = args->boardlist_xml;

  if (category->killed)
    return;

  gzprintf(boardlist_xml, "    <category>\n");

  g_signal_emit_by_name(G_OBJECT(category),
			"write_boardlist_element",
			boardlist_xml);
  g_signal_emit(G_OBJECT(args->table),
		bbs_table_signals[BOARDLIST_WRITE_CATEGORY_ELEMENT_SIGNAL],
		0,
		category,
		boardlist_xml);

  g_slist_foreach(category->board_list, write_board, args);

  gzprintf(boardlist_xml, "    </category>\n");
}


gboolean
ochusha_bbs_table_write_boardlist_xml(OchushaBBSTable *table,
				      OchushaConfig *config,
				      const char *subdir)
{
  WriteBoardlistArgs args;
  gzFile boardlist_xml;
  int fd;

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && config != NULL, FALSE);
  g_return_val_if_fail(config->home != NULL, FALSE);

  fd = ochusha_config_open_file(config, OCHUSHA_BOARDLIST_XML, subdir,
				O_WRONLY | O_TRUNC | O_CREAT);
  if (fd < 0)
    {
      fprintf(stderr, "Couldn't open \"%s/%s\" to write.\n",
	      config->home, OCHUSHA_BOARDLIST_XML);
      return FALSE;
    }

  boardlist_xml = gzdopen(fd, "w");
  if (boardlist_xml == NULL)
    {
      close(fd);
      fprintf(stderr, "Couldn't open fd to write.\n");
      return FALSE;
    }

  gzprintf(boardlist_xml, "<?xml version=\"1.0\"?>\n");
  gzprintf(boardlist_xml, "<ochusha>\n");
  gzprintf(boardlist_xml, "  <boardlist>\n");

  args.table = table;
  args.boardlist_xml = boardlist_xml;

  g_slist_foreach(table->category_list, write_category, &args);

  gzprintf(boardlist_xml, "  </boardlist>\n");
  gzprintf(boardlist_xml, "</ochusha>\n");

  return gzclose(boardlist_xml) == 0;
}
