/*
 * http_trans.c -- Functions for doing transport related stuff including
 * automatically extending buffers and whatnot.
 * Created: Christopher Blizzard <blizzard@appliedtheory.com>, 5-Aug-1998
 *
 * Copyright (C) 1998 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ghttp_config.h"

#include <fcntl.h>

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "http_trans.h"
#include "http_global.h"

#if defined(__FreeBSD_version) && (__FreeBSD_version < 501107)
#define NEED_RESOLVER_LOCK	1
#endif

#if !defined(HAVE_GETADDRINFO)
#define NEED_RESOLVER_LOCK	1
#endif

#if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
#include <pthread.h>
#endif

#ifdef ENABLE_OPENSSL
static int ssl_initialized = 0;

void
http_trans_check_init_ssl(void)
{
  if (ssl_initialized)
    return;

  SSL_library_init();
  SSL_load_error_strings();
  ssl_initialized = 1;
}
#endif

#if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
static int resolver_lock_initialized = 0;
static pthread_mutex_t resolver_lock;

static void
resolve_lock_check_init(void)
{
  if (resolver_lock_initialized)
    return;

  if (pthread_mutex_init(&resolver_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
  resolver_lock_initialized = 1;
}
#endif

static int
http_trans_buf_free(http_trans_conn *a_conn);


static int
http_trans_read_sock(http_trans_conn *a_conn, void *buf, size_t nbytes)
{
  return read(a_conn->sock, buf, nbytes);
}


static int
http_trans_write_sock(http_trans_conn *a_conn, void *buf, size_t nbytes)
{
  return write(a_conn->sock, buf, nbytes);
}


static int
http_trans_close_sock(http_trans_conn *a_conn)
{
  int result;
  if (a_conn->sock != -1)
    {
      result = close(a_conn->sock);
      a_conn->sock = -1;
    }
  else
    result = 0;
  return result;
}


#ifdef ENABLE_OPENSSL
static int
http_trans_read_ssl(http_trans_conn *a_conn, void *buf, size_t nbytes)
{
  return SSL_read(a_conn->ssl, buf, nbytes);
}


static int
http_trans_write_ssl(http_trans_conn *a_conn, void *buf, size_t nbytes)
{
  return SSL_write(a_conn->ssl, buf, nbytes);
}


static int
http_trans_close_ssl(http_trans_conn *a_conn)
{
  int result;
  if (a_conn->sock != -1)
    {
      SSL_shutdown(a_conn->ssl);
      result = close(a_conn->sock);
      SSL_free(a_conn->ssl);
      a_conn->ssl = NULL;
      SSL_CTX_free(a_conn->ssl_ctx);
      a_conn->ssl_ctx = NULL;
      a_conn->sock = -1;
    }
  else
    result = 0;
  return result;
}
#endif


int
http_trans_conn_close(http_trans_conn *a_conn)
{
#ifdef HAVE_GETADDRINFO
  if (a_conn->addrinfo != NULL)
    {
      freeaddrinfo(a_conn->addrinfo);
      a_conn->addrinfo = NULL;
    }
#endif
  if (a_conn->sock == -1)
    return 0;
  return (*a_conn->close)(a_conn);
}


int
http_trans_connect(http_trans_conn *a_conn)
{
#ifdef ENABLE_OPENSSL
  if (a_conn->ssl_ctx)
    {
      a_conn->read = http_trans_read_ssl;
      a_conn->write = http_trans_write_ssl;
      a_conn->close = http_trans_close_ssl;
    }
  else
    {
      a_conn->read = http_trans_read_sock;
      a_conn->write = http_trans_write_sock;
      a_conn->close = http_trans_close_sock;
    }
#else
  a_conn->read = http_trans_read_sock;
  a_conn->write = http_trans_write_sock;
  a_conn->close = http_trans_close_sock;
#endif
  if ((a_conn == NULL) || (a_conn->host == NULL))
    goto ec;

#ifdef HAVE_GETADDRINFO
  if (a_conn->addrinfo == NULL)
    {
      const char *host;
      char serv[16];
      struct addrinfo hints;
#if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
      resolve_lock_check_init();
      if (pthread_mutex_lock(&resolver_lock) != 0)
	{
	  fprintf(stderr, "Couldn't lock a mutex.\n");
	  abort();
	}
#endif

      if (a_conn->proxy_host)
	{
	  /* look up the name of the proxy if it's there. */
	  host = a_conn->proxy_host;
	  sprintf(serv, "%u", a_conn->proxy_port);
	}
      else
	{
	  host = a_conn->host;
	  sprintf(serv, "%u", a_conn->port);
	}

      memset(&hints, 0, sizeof(hints));
      hints.ai_family = AF_INET;
      hints.ai_socktype = SOCK_STREAM;
      if (getaddrinfo(host, serv, &hints, &a_conn->addrinfo))
	{
	  a_conn->error_type = http_trans_err_type_host;
	  a_conn->error = h_errno;
	  if (a_conn->addrinfo != NULL)
	    {
	      freeaddrinfo(a_conn->addrinfo);
	      a_conn->addrinfo = NULL;
	    }
# if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
	  if (pthread_mutex_unlock(&resolver_lock) != 0)
	    {
	      fprintf(stderr, "Couldn't unlock a mutex.\n");
	      abort();
	    }
# endif
	  goto ec;
	}
      a_conn->addr = a_conn->addrinfo->ai_addr;
      a_conn->addrlen = a_conn->addrinfo->ai_addrlen;

# if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
      if (pthread_mutex_unlock(&resolver_lock) != 0)
	{
	  fprintf(stderr, "Couldn't unlock a mutex.\n");
	  abort();
	}
# endif
    }
#else	/* !HAVE_GETADDRINFO */
  if (a_conn->hostinfo == NULL)
    {
      const char *host;
      u_short port;
# if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
      resolve_lock_check_init();
      if (pthread_mutex_lock(&resolver_lock) != 0)
	{
	  fprintf(stderr, "Couldn't lock a mutex.\n");
	  abort();
	}
# endif

      if (a_conn->proxy_host)
	{
	  host = a_conn->proxy_host;
	  port = a_conn->proxy_port;
	}
      else
	{
	  host = a_conn->host;
	  port = a_conn->port;
	}

      if ((a_conn->hostinfo = gethostbyname(host)) == NULL)
	{
	  a_conn->error_type = http_trans_err_type_host;
	  a_conn->error = h_errno;
# if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
	  if (pthread_mutex_unlock(&resolver_lock) != 0)
	    {
	      fprintf(stderr, "Couldn't unlock a mutex.\n");
	      abort();
	    }
# endif
	  goto ec;
	}

      /* set up the saddr */
      a_conn->saddr.sin_family = AF_INET;
      /* set the proxy port */
      a_conn->saddr.sin_port = htons(port);
      /* copy the name info */
      memcpy(&a_conn->saddr.sin_addr.s_addr,
	     a_conn->hostinfo->h_addr_list[0],
	     sizeof(unsigned long));
      a_conn->addr = (struct sockaddr *)&a_conn->saddr;
      a_conn->addrlen = sizeof(struct sockaddr_in);

# if defined(NEED_RESOLVER_LOCK) && defined(HAVE_PTHREAD_H)
      if (pthread_mutex_unlock(&resolver_lock) != 0)
	{
	  fprintf(stderr, "Couldn't unlock a mutex.\n");
	  abort();
	}
# endif
    }
#endif	/* HAVE_GETADDRINFO */

  /* set up the socket */
  if ((a_conn->sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      a_conn->error_type = http_trans_err_type_errno;
      a_conn->error = errno;
      goto ec;
    }

  /* set up the socket */
  if (connect(a_conn->sock, a_conn->addr, a_conn->addrlen) < 0)
    {
      a_conn->error_type = http_trans_err_type_errno;
      a_conn->error = errno;
      goto ec;
    }

  if (a_conn->sync == HTTP_TRANS_ASYNC)
    {
      int flags;
      flags = fcntl(a_conn->sock, F_GETFL) | O_NONBLOCK;
      fcntl(a_conn->sock, F_SETFL, flags);
    }

#ifdef ENABLE_OPENSSL
  if (a_conn->ssl_ctx != NULL)
    {
      int err;
      a_conn->ssl = SSL_new(a_conn->ssl_ctx);
      SSL_set_fd(a_conn->ssl, a_conn->sock);
      err = SSL_connect(a_conn->ssl);
      if (err != 1)
	{
	  a_conn->error_type = http_trans_err_type_ssl;
	  a_conn->error = SSL_get_error(a_conn->ssl, err);
	  http_trans_close_ssl(a_conn);
	  goto ec;
	}

      if (!a_conn->ssl_no_verify)
	{
	  /* verify server certificate */
	  long verify_result = SSL_get_verify_result(a_conn->ssl);

	  if (verify_result != X509_V_OK)
	    {
	      http_trans_close_ssl(a_conn);
	      a_conn->error_type = http_trans_err_type_ssl_verification;
	      a_conn->error = -1;
	      a_conn->ssl_verify_error = verify_result;
	      goto ec;
	    }
	}
    }
#endif
  
  return 0;
 ec:
  return -1;
}

http_trans_conn *
http_trans_conn_new(void)
{
  http_trans_conn *l_return = NULL;

  /* allocate a new connection struct */
  l_return = (http_trans_conn *)malloc(sizeof(http_trans_conn));
  memset(l_return, 0, sizeof(http_trans_conn));
  /* default to 80 */
  l_return->port = 80;
  /* default to 1000 bytes at a time */
  l_return->io_buf_chunksize = 1024;
  /* allocate a new trans buffer */
  l_return->io_buf = malloc(l_return->io_buf_chunksize);
  memset(l_return->io_buf, 0, l_return->io_buf_chunksize);
  l_return->io_buf_len = l_return->io_buf_chunksize;
  /* make sure the socket looks like it's closed */
  l_return->sock = -1;
  return l_return;
}

void
http_trans_conn_destroy(http_trans_conn *a_conn)
{
  /* destroy the connection structure. */
  if (a_conn == NULL)
    return;
  if (a_conn->io_buf)
    free(a_conn->io_buf);
  if (a_conn->sock != -1)
    (*a_conn->close)(a_conn);
  free(a_conn);
  return;
}

const char *
http_trans_get_host_error(int a_herror)
{
  switch (a_herror)
    {
    case HOST_NOT_FOUND:
      return "Host not found";
    case NO_ADDRESS:
      return "An address is not associated with that host";
    case NO_RECOVERY:
      return "An unrecoverable name server error occured";
    case TRY_AGAIN:
      return "A temporary error occurred on an authoritative name server.  Please try again later.";
    default:
      return "No error or error not known.";
    }
}

int
http_trans_append_data_to_buf(http_trans_conn *a_conn,
			      const char *a_data,
			      int   a_data_len)
{
  if (http_trans_buf_free(a_conn) < a_data_len)
    {
      a_conn->io_buf = realloc(a_conn->io_buf, a_conn->io_buf_len + a_data_len);
      a_conn->io_buf_len += a_data_len;
    }
  memcpy(&a_conn->io_buf[a_conn->io_buf_alloc], a_data, a_data_len);
  a_conn->io_buf_alloc += a_data_len;
  return 1;
}

int
http_trans_read_into_buf(http_trans_conn *a_conn)
{
  int l_read = 0;
  int l_bytes_to_read = 0;

  /* set the length if this is the first time */
  if (a_conn->io_buf_io_left == 0)
    {
      a_conn->io_buf_io_left = a_conn->io_buf_chunksize;
      a_conn->io_buf_io_done = 0;
    }
  /* make sure there's enough space */
  if (http_trans_buf_free(a_conn) < a_conn->io_buf_io_left)
    {
      a_conn->io_buf = realloc(a_conn->io_buf,
			       a_conn->io_buf_len + a_conn->io_buf_io_left);
      a_conn->io_buf_len += a_conn->io_buf_io_left;
    }
  /* check to see how much we should try to read */
  if (a_conn->io_buf_io_left > a_conn->io_buf_chunksize)
    l_bytes_to_read = a_conn->io_buf_chunksize;
  else
    l_bytes_to_read = a_conn->io_buf_io_left;
  /* read in some data */
  if ((a_conn->last_read = l_read
       = (*a_conn->read)(a_conn,
			 &a_conn->io_buf[a_conn->io_buf_alloc],
			 l_bytes_to_read)) < 0)
    {
      if (l_read == -1 && a_conn->sync == HTTP_TRANS_ASYNC && errno == EAGAIN)
	return HTTP_TRANS_NOT_DONE;
      if (errno == EINTR)
	l_read = 0;
      else
	return HTTP_TRANS_ERR;
    }
  else if (l_read == 0)
    return HTTP_TRANS_DONE;
  /* mark the buffer */
  a_conn->io_buf_io_left -= l_read;
  a_conn->io_buf_io_done += l_read;
  a_conn->io_buf_alloc += l_read;
  /* generate the result */
  if (a_conn->io_buf_io_left == 0)
    return HTTP_TRANS_DONE;
  return HTTP_TRANS_NOT_DONE;
}

int
http_trans_write_buf(http_trans_conn *a_conn)
{
  int l_written = 0;

  if (a_conn->io_buf_io_left == 0)
    {
      a_conn->io_buf_io_left = a_conn->io_buf_alloc;
      a_conn->io_buf_io_done = 0;
    }
  /* write out some data */
  if ((a_conn->last_read = l_written
       = (*a_conn->write)(a_conn,
			  &a_conn->io_buf[a_conn->io_buf_io_done],
			  a_conn->io_buf_io_left)) <= 0)
    {
      if (l_written == -1
	  && a_conn->sync == HTTP_TRANS_ASYNC && errno == EAGAIN)
	return HTTP_TRANS_NOT_DONE;
      if (errno == EINTR)
	l_written = 0;
      else
	return HTTP_TRANS_ERR;
    }
  if (l_written == 0)
    return HTTP_TRANS_DONE;
  /* advance the counters */
  a_conn->io_buf_io_left -= l_written;
  a_conn->io_buf_io_done += l_written;
  if (a_conn->io_buf_io_left == 0)
    return HTTP_TRANS_DONE;
  return HTTP_TRANS_NOT_DONE;
}

void
http_trans_buf_reset(http_trans_conn *a_conn)
{
  if (a_conn->io_buf)
    free(a_conn->io_buf);
  a_conn->io_buf = malloc(a_conn->io_buf_chunksize);
  memset(a_conn->io_buf, 0, a_conn->io_buf_chunksize);
  a_conn->io_buf_len = a_conn->io_buf_chunksize;
  a_conn->io_buf_alloc = 0;
  a_conn->io_buf_io_done = 0;
  a_conn->io_buf_io_left = 0;
}

void
http_trans_buf_clip(http_trans_conn *a_conn, char *a_clip_to)
{
  int l_bytes = 0;
  
  /* get the number of bytes to clip off of the front */
  l_bytes = a_clip_to - a_conn->io_buf;
  if (l_bytes > 0)
    {
      memmove(a_conn->io_buf, a_clip_to, a_conn->io_buf_alloc - l_bytes);
      a_conn->io_buf_alloc -= l_bytes;
    }
  a_conn->io_buf_io_done = 0;
  a_conn->io_buf_io_left = 0;
}

char *
http_trans_buf_has_patt(char *a_buf, int a_len,
			char *a_pat, int a_patlen)
{
  int i = 0;
  for ( ; i <= ( a_len - a_patlen ); i++ )
    {
      if (a_buf[i] == a_pat[0])
	{
	  if (memcmp(&a_buf[i], a_pat, a_patlen) == 0)
	    return &a_buf[i];
	}
    }
  return NULL;
}

/* static functions */

static int
http_trans_buf_free(http_trans_conn *a_conn)
{
  return (a_conn->io_buf_len - a_conn->io_buf_alloc);
}
