package java.util.regex;

import java.util.ArrayList;
import java.util.List;
import onig4j.OnigOptionType;
import onig4j.OnigRegex;
import onig4j.OnigRegion;
import static java.util.regex.Patterns.isNotEscaped;

/**
 *
 * @author calico
 */
public class Matcher extends MatchResultImpl {
    
    private Pattern pattern;
    private int start;
    private int range;
    
    private boolean hasAnchoringBounds;
    private CharSequence regionInput;
    
    Matcher(Pattern pattern, CharSequence input) {
        super(new OnigRegion(), input);
        this.pattern = pattern;
        this.range = input.length();
        this.hasAnchoringBounds = true;
    }
    
    public Pattern pattern() {
        return pattern;
    }

    public MatchResult toMatchResult() {
        return new MatchResultImpl(region.clone(), input, (hasAnchoringBounds ? offset : 0));
    }

    public Matcher usePattern(Pattern newPattern) {
        if (newPattern == null) {
            throw new IllegalArgumentException("newPattern is null");
        }
        
        pattern = newPattern;
        region.resizeClear(0);
        return this;
    }

    public Matcher reset() {
        start = offset = 0;
        range = input.length();
        region.resizeClear(0);
        return this;
    }

    public Matcher reset(CharSequence input) {
        this.input = input;
        return reset();
    }

    private int match() {
        int ret;
        if (regionInput == null) {
            ret = pattern.regex.match(input, range, 0, region);
            
        } else {
            if (hasAnchoringBounds) {
                ret = pattern.regex.match(regionInput, 0, region);
                
            } else {
                ret = pattern.regex.match(regionInput, 0, region,
                                        OnigOptionType.ONIG_OPTION_NOTBOL,
                                        OnigOptionType.ONIG_OPTION_NOTEOL);
            }
        }
        start = (ret >= 0 ? ret : 0);
        return ret;
    }
    
    public boolean matches() {
        if (match() == (range - offset)) {
            return true;
            
        } else {
            // 全体にマッチしなかった場合はregionの情報をクリアする
            region.resizeClear(0);
            return false;
        }
    }

    public boolean find() {
        int ret;
        if (regionInput == null) {
            ret = pattern.regex.search(input, start, region);
            
        } else {
            if (hasTransparentBounds) {
                final int st = start + offset;
                ret = pattern.regex.search(input, st, region);
                if (ret < st || region.end(0) > range) {
                    ret = OnigRegex.ONIG_MISMATCH;
                }
            } else if (hasAnchoringBounds) {
                ret = pattern.regex.search(regionInput, start, region);
                
            } else {
                ret = pattern.regex.search(regionInput, start, region,
                                        OnigOptionType.ONIG_OPTION_NOTBOL,
                                        OnigOptionType.ONIG_OPTION_NOTEOL);
            }
        }
        if (ret >= 0) {
            start = region.end(0);
            return true;
            
        } else {
            return false;
        }
    }

    public boolean find(int start) {
        if (start < 0 || start > input.length()) {
            throw new IndexOutOfBoundsException("the start parameter is out of range");
        }
        
        reset();
        this.start = start;
        return find();
    }

    public boolean lookingAt() {
        return (match() >= 0);
    }

    /**
     * 
     * @param s
     * @return
     */
    public static String quoteReplacement(String s) {
        final StringBuilder quoted = new StringBuilder(s.length());
        for (final char ch : s.toCharArray()) {
            if (ch == '\\' || ch == '$') {
                quoted.append('\\');
            }
            quoted.append(ch);
        }
        return quoted.toString();
    }

    public Matcher appendReplacement(StringBuffer sb, String replacement) {
        sb.append(replace(replacement, 1, false));
        return this;
    }

    public StringBuffer appendTail(StringBuffer sb) {
        final int end = input.length();
        if (start < end) {
            sb.append(input.subSequence(start, end));                
        }
        return sb;
    }

    private String replace(String replacement, int limit, boolean isAppendTail) {
        final Object replace = parseReplacement(replacement);
        final StringBuilder replaced = new StringBuilder();
        final OnigRegex regex = pattern.regex;
        final int end = input.length();
        int off = 0;
        int ret;
        while ((ret = regex.search(input, end, start + off, range, region)) >= 0) {
            replaced.append(input.subSequence(start, ret).toString());
            start = region.end(0);
            replaced.append(replace.toString());
            --limit;
            if (limit == 0) {
                break;
            }
            if (ret == start) { // region.begin(0) == region.end(0)
                // 行末記号などの境界正規表現や幅ゼロの先読み・後読みにマッチした場合は
                // 無限ループに陥るのを防ぐために検索開始位置を1文字ずらす
                off = 1;
                if (start + off >= end) {
                    break;
                }
            }
        }
        if (isAppendTail && start < end) {
            replaced.append(input.subSequence(start, end).toString());                
        }
        return replaced.toString();
    }
    
    /**
     * エスケープ処理を行う。
     * @param str
     * @return
     */
    private static String deleteEscapeChars(String str) {
        int pos = str.indexOf('\\');
        if (pos < 0) {
            return str;
        }
        
        final StringBuilder buff = new StringBuilder(str.length());
        final int last = str.length() - 1;
        int start = 0;
        do {
            if (pos < last && str.charAt(pos + 1) == '\\') {
                // エスケープ記号をエスケープしている場合
                ++pos;
            }
            buff.append(str.substring(start, pos));
            start = pos + 1;

        } while ((pos = str.indexOf('\\', start)) != -1);

        return buff.append(str.substring(start)).toString();
    }

    /** 前方参照オブジェクト */
    private class PreviousReferenceObject {
        private final int group;
        
        private PreviousReferenceObject(int group) {
            this.group = group;
        }

        @Override
        public String toString() {
            return group(group);
        }
    }
    
    private class ReplacementObject {
        private final List<Object> replace = new ArrayList<Object>();
        
        public void add(Object e) {
            replace.add(e);
        }
        
        public boolean isEmpty() {
            return replace.isEmpty();
        }
        
        @Override
        public String toString() {
            final StringBuilder buff = new StringBuilder();
            for (final Object e : replace) {
                buff.append(e.toString());
            }
            return buff.toString();
       }
    }
    
    private Object parseReplacement(String replacement) {
        final OnigRegex regex = new OnigRegex("\\$[0-9]");
        try {
            final int e = replacement.length();
            int s = 0;
            int ret = regex.search(replacement, e, s, e);
            if (ret < 0) {
                // 置換文字列に前方参照記号（'$n'）を含まない場合
                return deleteEscapeChars(replacement);

            } else {
                // 置換文字列に前方参照記号（'$n'）を含んでいる場合
                boolean hasPrevReference = false;
                ReplacementObject replace = new ReplacementObject();
                do {
                    final String literal = deleteEscapeChars(replacement.substring(s, ret));
                    replace.add(literal);
                    s = ret + 2;    // 2 = "$g".length();
                    if (isNotEscaped(replacement, ret)) {
                        // '$'がエスケープされていない場合はPreviousReferenceObjectを格納する
                        final int group = Integer.parseInt(replacement.substring(ret + 1, s));
                        replace.add(new PreviousReferenceObject(group));
                        hasPrevReference = true;
                        
                    } else {
                        // '$'がエスケープされている場合は、文字列として格納する
                        replace.add(replacement.substring(ret, s));
                    }
                } while ((ret = regex.search(replacement, e, s, e))>= 0);
                
                if (hasPrevReference) {
                    if (s < e && !replace.isEmpty()) {
                        replace.add(replacement.substring(s, e));
                    }
                    return replace;
                    
                } else {
                    // 置換文字列に前方参照が含まれていなかった場合
                    return deleteEscapeChars(replacement);
                }
            }
        } finally {
            regex.close();
        }
    }
    
    public String replaceAll(String replacement) {
        reset();
        return replace(replacement, -1, true);
     }

    public String replaceFirst(String replacement) {
        reset();
        return replace(replacement, 1, true);
    }

    public Matcher region(int start, int end) {
        final int len = input.length();
        if (start < 0 || start > end || start > len) {
            throw new IndexOutOfBoundsException("the start parameter is out of range");
        }
        if (end < 0 || end > len) {
            throw new IndexOutOfBoundsException("the end parameter is out of range");
        }
        
        this.start = 0;
        regionInput = (start > 0 || end < len ? input.subSequence(start, end) : null);
        offset = start;
        range = end;
        region.resizeClear(0);
        return this;
    }

    public int regionStart() {
        return offset;
    }

    public int regionEnd() {
        return range;
    }

    public boolean hasTransparentBounds() {
        return hasTransparentBounds;
    }

    public Matcher useTransparentBounds(boolean b) {
        hasTransparentBounds = b;
        return this;
    }

    public boolean hasAnchoringBounds() {
        return hasAnchoringBounds;
    }

    public Matcher useAnchoringBounds(boolean b) {
        if (hasAnchoringBounds && !b) {
            start += offset;
            
        } else if (!hasAnchoringBounds && b) {
            start -= offset;
        }
        
        hasAnchoringBounds = b;
        return this;
    }

    @Override
    public String toString() {
        return Matcher.class.getName()
                + "[pattern=" + pattern.pattern()
                + " region=" + offset + "," + range
                + " lastmatch="+ (region.count() > 0 ? group() : "") + "]";
    }

    /**
     * This method is experimentation phase, and implementation has not been completed yet.
     * @return
     * @deprecated
     */
    @Deprecated
    public boolean hitEnd() {
        return (hasAnchoringBounds ? (range == region.end(0)) : (input.length() == end()));
    }

//    /**
//     * Not supported yet.
//     * @return
//     * @throws UnsupportedOperationException
//     */
//    public boolean hitEnd() {
//        throw new UnsupportedOperationException("Not supported yet");
//    }
    
    /**
     * Not supported yet.
     * @return
     * @throws UnsupportedOperationException
     */
    public boolean requireEnd() {
        throw new UnsupportedOperationException("Not supported yet");
    }
}
