package java.util.regex;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import onig4j.OnigOptionType;
import onig4j.OnigRegex;
import onig4j.OnigSyntaxType;

/**
 *
 * @author calico
 */
public final class Pattern implements Serializable {
    private static final long serialVersionUID = 1L;
    
    /**
     * NOTE: CANON_EQ option requires the Unicode Normalizer class.
     * @see java.text.Normalizer
     * @see <a href="http://www.icu-project.org/">ICU - ICU4J, ICU4JNI</a>
     */
    public static final int CANON_EQ = 128;
    public static final int CASE_INSENSITIVE = 2;
    public static final int COMMENTS = 4;
    public static final int DOTALL = 32;
    public static final int LITERAL = 16;
    /*
     * MEMO: 鬼車の行末記号は「\n」にのみマッチする
     */
    public static final int MULTILINE = 8;
    public static final int UNICODE_CASE = 64;
    public static final int UNIX_LINES = 1;
    
    final OnigRegex regex;
    private final String pattern;
    private final int flags;
   
    private Pattern(String pattern, int flags) {
        this.pattern = pattern;
        this.flags = flags;

        if ((flags & LITERAL) == 0) {
            // LITERALモードでない場合は正規表現に含まれるUnicodeエスケープシーケンスを文字に変換して置き換える
            pattern = Patterns.unescapeUnicode(pattern);
            
            if ((flags & CANON_EQ) != 0) {
                // CANON_EQモードの場合は正規表現を正規分解した表現に変換する
                try {
                    pattern = Patterns.normalize(pattern);
                    
                } catch (UnsupportedOperationException ex) {
                    throw new UnsupportedOperationException(
                                "CANON_EQ option requires the Unicode Normalizer",
                                ex
                            );
                }
            }
        } else {
            // LITERALモードの場合は前処理は不要
            pattern = quote(pattern);
        }
        
        List<OnigOptionType> options = null;
        if (flags != 0) {
            options = new ArrayList<OnigOptionType>(6);
            if ((flags & CASE_INSENSITIVE) != 0) {
                options.add(OnigOptionType.ONIG_OPTION_IGNORECASE);
            }
            if ((flags & UNICODE_CASE) != 0) {
                options.add(OnigOptionType.ONIG_OPTION_NEGATE_CASE_FOLD_ASCII_ONLY);
            }            
            if ((flags & MULTILINE) != 0) {
                options.add(OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
                
                if ((flags & UNIX_LINES) != 0) {
                    options.add(OnigOptionType.ONIG_OPTION_NEGATE_JAVA_LINES);
                }
            }
            if ((flags & DOTALL) != 0) {
                options.add(OnigOptionType.ONIG_OPTION_MULTILINE);
            }
            if ((flags & COMMENTS) != 0) {
                options.add(OnigOptionType.ONIG_OPTION_EXTEND);
            }
        }
        
//        OnigRegex.setWarningListener(
//                    new OnigRegex.WarningListener() {
//                        private final java.util.logging.Logger log
//                                = java.util.logging.Logger.getLogger(Pattern.class.getName());
//                        public void warning(String msg) {
//                            log.warning(msg);
//                        }
//                    }
//                );
        this.regex = new OnigRegex(pattern, OnigSyntaxType.ONIG_SYNTAX_JAVA, options);
    }

    public static Pattern compile(String regex) {
        return new Pattern(regex, 0);
    }

    public static Pattern compile(String regex, int flags) {
        return new Pattern(regex, flags);
    }

    public int flags() {
        return flags;
    }

    public  Matcher matcher(CharSequence input) {
        return new Matcher(this, input);
    }

    public static boolean matches(String regex, CharSequence input) {
        return compile(regex).matcher(input).matches();
    }
    
    public String pattern() {
        return pattern;
    }
    
    /**
     * 
     * @param s 
     * @return "\\Q" + s + "\\E"
     */
    public static String quote(String s) {
        return "\\Q" + s + "\\E";
    }
    
    /**
     * 
     * @param input
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int)
     */
    public String[] split(CharSequence input) {
        return split(input, 0);
    }
    
    /**
     * 
     * @param input
     * @param limit
     * @return
     * @see onig4j.OnigRegex#split(java.lang.CharSequence, int, int)
     */
    public String[] split(CharSequence input, int limit) {
        final List<String> split = regex.split(input, limit);
        return split.toArray(new String[split.size()]);

//        // MEMO onig_split()のJavaによる実装
//        final List<String> list = new ArrayList<String>();
//        final OnigRegion region = new OnigRegion();
//        final boolean isTrim = (limit == 0);
//        int start = 0;
//        final int end = input.length();
//        int offset = 0;
//        int ret;
//        while ((ret = regex.search(input, end, start + offset, end, region)) >= 0) {
//            list.add(input.subSequence(start, ret).toString());
//            start = region.end(0);
//            --limit;
//            if (limit == 0) {
//                break;
//            }
//            if (ret == start) { // region.begin(0) == region.end(0)
//                // 行末記号などの境界正規表現や幅ゼロの先読み・後読みにマッチした場合は
//                // 無限ループに陥るのを防ぐために検索開始位置を1文字ずらす
//                offset = 1;
//                if (start + offset >= end) {
//                    break;
//                }
//            }
//        }
//        if (start <= end) {
//            list.add(input.subSequence(start, end).toString());
//        }
//        region.close();
//        
//        if (isTrim && !list.isEmpty()) {
//            // 後続の空の文字列を破棄する
//            for (final ListIterator<String> iter = list.listIterator(list.size());
//                    iter.hasPrevious() && iter.previous().length() == 0;
//                    iter.remove()) {
//                // nothing
//            }
//        }
//        return list.toArray(new String[list.size()]);
    }
    
    /**
     * 
     * @return The string representation of this pattern
     * @see #pattern()
     */
    @Override
    public String toString() {
        return pattern();
    }
}
