package java.util.regex;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import onig4j.OnigRegex;

/**
 *
 * @author calico
 */
final class Patterns {
    
    /**
     * Unicodeエスケープシーケンス '\\uXXXX' を抽出する為の正規表現オブジェクト
     */
    private static final OnigRegex unicode = new OnigRegex("\\\\u\\h{4}");
    
    /**
     * 正規表現にUnicodeエスケープシーケンスが存在する場合は文字に変換して置き換える
     * @param pattern
     * @return
     */
    static String unescapeUnicode(String pattern) {
        final int end = pattern.length();
        int pos = unicode.search(pattern, end, 0, end);
        if (pos < 0) {
            // Unicodeエスケープシーケンスが存在しない場合
            return pattern;
        }
        
        // 正規表現にUnicodeエスケープシーケンスが存在する場合は文字に変換して置き換える
        final StringBuilder newPattern = new StringBuilder(end);
        int start = 0;
        do {
            newPattern.append(pattern.substring(start, pos));
            start = pos + 6;    // 6 = "\\uXXXX".length();
            newPattern.append((char) Integer.parseInt(pattern.substring(pos + 2, start), 16));
        } while ((start < end) && (pos = unicode.search(pattern, end, start, end)) >= 0);
        if (start < end) {
            newPattern.append(pattern.substring(start));
        }
        return newPattern.toString();
    }
    
    /**
     * 前方のエスケープ記号をカウントする
     * @param cs
     * @param index
     * @return
     */
    private static int countPrevEscapeChar(CharSequence cs, int i) {
        int cnt = 0;
        for (; i > 0 && cs.charAt(--i) == '\\'; ++cnt) {
            // nothing
        }
        return cnt;
    }
    
    /**
     * 指定されたインデックス位置の文字がエスケープされていない場合はtrueを返す。
     * @param cs 文字シーケンス
     * @param index インデックス位置
     * @return
     */
    static boolean isNotEscaped(CharSequence cs, int index) {
        return ((countPrevEscapeChar(cs, index) % 2) == 0);
    }

    /**
     * Unicode Normalizer クラス
     */
    private static final class Normalizer {
        
        private static Normalizer instance;

        /**
         * 
         * @see <a href="http://java.sun.com/javase/6/docs/api/java/text/Normalizer.html">java.text.Normalizer</a>
         * @see <a href="http://www.icu-project.org/">ICU</a>
         * @see <a href="http://www.icu-project.org/apiref/icu4j/com/ibm/icu/text/Normalizer.html">Normalizer - ICU4J</a>
         * @see <a href="http://www.icu-project.org/apiref/icu4jni/com/ibm/icu4jni/text/Normalizer.html">Normalizer - ICU4JNI</a>
         */
        private static Normalizer getInstance() throws ClassNotFoundException {
            if (instance != null) {
                return instance;
            }

            // java.text.Normalizer、ICU4J、ICU4JNIの順でNormalizerを検索する
            final String[][] normalizers
                    = new String[][] {
                            // java.text.Normalizer
                            {
                                "java.text.Normalizer$Form",
                                "NFC",
                                "NFD",
                                "java.text.Normalizer",
                                "normalize",
                                "java.lang.CharSequence",
                            },
                            // ICU4J
                            {
                                "com.ibm.icu.text.Normalizer",
                                "NFC",
                                "NFD",
                                "com.ibm.icu.text.Normalizer",
                                "normalize",
                                "java.lang.String",
                            },
                            // ICU4JNI
                            {
                                "com.ibm.icu4jni.text.Normalizer",
                                "UNORM_NFC",
                                "UNORM_NFD",
                                "com.ibm.icu4jni.text.Normalizer",
                                "normalize",
                                "java.lang.String",
                            },
                        };
            for (String[] param : normalizers) {
                try {
                    Class clazz = Class.forName(param[0]);
                    final Field nfc = clazz.getField(param[1]);
                    final Field nfd = clazz.getField(param[2]);

                    clazz = Class.forName(param[3]);
                    Method method
                            = clazz.getMethod(
                                    param[4],
                                    Class.forName(param[5]),
                                    nfc.getType()
                                );
                    instance = new Normalizer(method, nfc.get(null), nfd.get(null));
                    return instance;

    //            } catch (ClassNotFoundException ex) {
    //            } catch (NoSuchFieldException ex) {
    //            } catch (NoSuchMethodException ex) {
    //            } catch (IllegalAccessException ex) {
    //            } catch (IllegalArgumentException ex) {
                } catch (Exception ex) {
                    // nothing
                }
            }
            throw new ClassNotFoundException(
                            "Requires 'java.text.Normalizer' "
                            + "or 'com.ibm.icu.text.Normalizer' "
                            + "or 'com.ibm.icu4jni.text.Normalizer'"
                        );
        }
        
        private final Object NFC;
        private final Object NFD;
        private final Method normalize;
        
        private Normalizer(Method method, Object nfc, Object nfd) {
            normalize = method;
            NFC = nfc;
            NFD = nfd;
        }
        
        private String normalizeNFC(CharSequence src) {
            return normalize(src, NFC);
        }
        
        private String normalizeNFD(CharSequence src) {
            return normalize(src, NFD);
        }
        
        private String normalize(CharSequence src, Object form) {
            try {
                return (String) normalize.invoke(null, src, form);
                
            } catch (IllegalAccessException ex) {
                throw new UnsupportedOperationException(ex);

            } catch (IllegalArgumentException ex) {
                throw new UnsupportedOperationException(ex);

            } catch (InvocationTargetException ex) {
                throw new UnsupportedOperationException(ex);
            }
        }
    }
    
    /**
     * 正規分解（NFD）した正規表現を返す。
     * @param pattern
     * @return
     * @throws UnsupportedOperationException Requires Unicode Normalizer
     * @see <a href="http://java.sun.com/javase/6/docs/api/java/text/Normalizer.html">java.text.Normalizer</a>
     * @see <a href="http://www.icu-project.org/">ICU</a>
     * @see <a href="http://www.icu-project.org/apiref/icu4j/com/ibm/icu/text/Normalizer.html">Normalizer - ICU4J</a>
     * @see <a href="http://www.icu-project.org/apiref/icu4jni/com/ibm/icu4jni/text/Normalizer.html">Normalizer - ICU4JNI</a>
     */
    static String normalize(String pattern) {
        try {
            final Normalizer normalizer = Normalizer.getInstance();
            final String nfc = normalizer.normalizeNFC(pattern);
            final String nfd = normalizer.normalizeNFD(pattern);
            
//            final String nfc = Normalizer.normalize(pattern, Form.NFC);
//            final String nfd = Normalizer.normalize(pattern, Form.NFD);

            final int len = nfc.length();
            final int capacity = len + (nfd.length() - len) * 7;
            final StringBuilder newPattern = new StringBuilder(capacity);
            int i = 0;
            for (final char c : nfc.toCharArray()) {
                final char d = nfd.charAt(i++);
                if (c == d) {
                    newPattern.append(c);
                } else {
                    newPattern.append("(?:");
                    newPattern.append(c);
                    newPattern.append('|');
                    newPattern.append(d);
                    newPattern.append(nfd.charAt(i++));
                    newPattern.append(')');
                }
            }
            return newPattern.toString();
            
        } catch (ClassNotFoundException ex) {
            throw new UnsupportedOperationException("Requires Unicode Normalizer", ex);
        }
    }

//    /** 前処理済みの正規表現パターン文字列 */
//    private final String pattern;
//    
//    /** 前処理済みの正規表現オプションフラグ */
//    private int flags;
//    
//    private Patterns(String regex, int flags) {
//        // 正規表現に含まれるUnicodeエスケープシーケンスを文字に変換して置き換える
//        regex = unescapeUnicode(regex);
//        
//        // 鬼車が対応していない埋め込みフラグ表現を削除する
//        this.flags = flags;
//        this.pattern = deleteUnsupportedEmbeddedFlag(regex);
////        this.pattern = deleteUnsupportedEmbeddedFlag(flags, regex);
////        this.pattern = regex;
//    }
//    
//    /**
//     * 前処理済みの正規表現パターン文字列を返す。
//     * @return
//     */
//    public String pattern() {
//        return pattern;
//    }
//    
//    /**
//     * 前処理済みの正規表現オプションフラグを返す。
//     * @return
//     */
//    public int flags() {
//        return flags;
//    }
//
//    /**
//     * 正規表現とフラグに対して前処理を行った結果を持つオブジェクトを生成する。
//     * @param regex 正規表現パターン文字列
//     * @param flags 正規表現オプションフラグ
//     * @return 前処理済み正規表現パラメータオブジェクト
//     */
//    static Patterns preprocess(String regex, int flags) {
//        return new Patterns(regex, flags);
//    }
//    
//    /**
//     * 正規表現に含まれる行末記号をJavaの行末記号表現に置き換える
//     * @param pattern
//     * @return
//     */
//    static String replaceLineTerminator(String pattern) {
//        int pos = pattern.indexOf('$');
//        if (pos < 0) {
//            return pattern;
//        }
//        
//        final int end = pattern.length();
//        // 25 = "(?:(?=[\r\u0085\u2028\u2029]|(?<!\r)\n)|\\Z)".length();
//        final StringBuilder newPattern = new StringBuilder(end + 25);
//        int start = 0;
//        do {
//            if (isNotEscaped(pattern, pos)) {
//                newPattern.append(pattern.substring(start, pos));
//                // 行末記号がエスケープされていない場合は正規表現を置き換える
//                newPattern.append("(?:(?=[\r\u0085\u2028\u2029]|(?<!\r)\n)|\\Z)");
//                
//            } else {
//                // 行末記号がエスケープされている場合は置き換えない
//                newPattern.append(pattern.substring(start, pos + 1));
//            }
//            start = pos + 1;
//        } while ((start < end) && (pos = pattern.indexOf("$", start)) != -1);
//        if (start < end) {
//            newPattern.append(pattern.substring(start));
//        }
//        return newPattern.toString();
//    }
//
//    /**
//     * 'd'また'u'を含んだ埋め込みフラグ表現'(?idsmux-idsmux)'を抽出する為の正規表現オブジェクト
//     */
//    private static final OnigRegex embeddedFlag
//            = new OnigRegex("\\(\\?(?=.*[du].*)([idsmux]*(?:-[idsmux]+)?)\\)");
//            // MEMO 埋め込みフラグをonとoffで分けて取得する場合の正規表現
////            = new OnigRegex("\\(\\?(?=.*[du].*)(([idsmux]*)(-[idsmux]+)?)\\)");
//    
//    /**
//     * 鬼車が対応していない埋め込みフラグ表現'(?d)'及び'(?u)'を削除する。<br/>
//     * NOTE: 埋め込みフラグ表現が文字列の先頭に存在する場合しか対応しない。
//     * @param pattern
//     * @return 埋め込みフラグ表現を削除した正規表現パターン文字列
//     */
//    private String deleteUnsupportedEmbeddedFlag(String pattern) {
//        final int end = pattern.length();
//        final int len = embeddedFlag.match(pattern, end, 0);
//        if (len < 0) {
//            // 埋め込みフラグ表現が存在しない場合
//            return pattern;
//        }
//        
//        // 正規表現に埋め込みフラグ表現'(?d)'または'(?u)'が存在する場合
//        int last = len - 1;
//        final StringBuilder flag = new StringBuilder(last - 2);
//        boolean isActive = true;
//        for (int i = 2; i < last; ++i) {
//            final char c = pattern.charAt(i);
//            if (c == '-') {
//                isActive = false;
//
//            } else if (c == 'd') {
//                if (isActive) {
//                    flags |= Pattern.UNIX_LINES;
//                } else {
//                    flags &= ~Pattern.UNIX_LINES;
//                }
//                continue;
//                
//            } else if (c == 'u') {
//                if (isActive) {
//                    flags |= Pattern.UNICODE_CASE;
//                } else {
//                    flags &= ~Pattern.UNICODE_CASE;
//                }
//                continue;
//            }
//            flag.append(c);
//        }
//        last = flag.length() - 1;
//        if (last >= 0 && flag.charAt(last) == '-') {
//            // '-'で終わっている場合は'-'を削除する
//            flag.deleteCharAt(last);
//        }
//        
//        final StringBuilder newPattern = new StringBuilder(end);
//        if (flag.length() > 0) {
//            //　埋め込みフラグ表現が空でない場合
//            newPattern.append("(?").append(flag).append(")");
//        }
//        newPattern.append(pattern.substring(len));
//        return newPattern.toString();
//    }
//    
//    /**
//     * 埋め込みフラグ表現 '(?idsmux-idsmux:X)' を抽出する為の正規表現オブジェクト
//     * TODO 閉じ括弧がエスケープされていた場合に対応していないため要改修！
//     * MEMO '(?idsmux-idsmux)'と'(?idsmux-idsmux:X)'を分けて検索すれば対応可能か？
//     */
//    private static final OnigRegex embeddedFlag
//            = new OnigRegex(
//                    "\\(\\?([idsmux]*-?[idsmux]+)(:([^)]*))?\\)",
//                    OnigSyntaxType.ONIG_SYNTAX_JAVA,
//                    OnigOptionType.ONIG_OPTION_MULTILINE
//                );
//        
//    /**
//     * 鬼車が対応していない埋め込みフラグ表現'(?d)'を削除する。<br/>
//     * NOTE: 埋め込みフラグ表現'(?u)'は鬼車本体を改造して対応している。
//     * @param pattern
//     * @return 埋め込みフラグ表現を削除した正規表現パターン文字列
//     */
//    private String deleteUnsupportedEmbeddedFlag(String pattern) {
//        final int end = pattern.length();
//        final OnigRegion region = new OnigRegion();
//        int pos = embeddedFlag.search(pattern, end, 0, end, region);
//        if (pos < 0) {
//            // 埋め込みフラグ表現が存在しない場合
//            return pattern;
//        }
//        
//        // 正規表現に埋め込みフラグ表現が存在する場合は削除する
//        final StringBuilder ptn = new StringBuilder(end);
//        int start = 0;
//        do {
//            final int regionEnd = region.end(0);
//            if (isNotEscaped(pattern, pos)) {
//                final int begin1 = region.begin(1);
//                final int end1 = region.end(1);
//                final String flag = pattern.substring(begin1, end1);
//                final boolean hasUnixLinesFlag = (flag.indexOf('d') >= 0);
//                if (!hasUnixLinesFlag) {
//                    // 対応していない埋め込みフラグ表現が含まれていない場合
//                    ptn.append(pattern.substring(start, regionEnd));
//
//                } else {
//                    // 対応している埋め込みフラグ表現と混ざっている場合
//                    // 対応していない埋め込みフラグ表現のみを削除する
//                    boolean isActive = true;
//                    final StringBuilder deleted = new StringBuilder(flag.length());
//                    for (final char c : flag.toCharArray()) {
//                        if (c == '-') {
//                            isActive = false;
//                            
//                        } else if (c == 'd') {
//                            if (isActive) {
//                                flags |= Pattern.UNIX_LINES;
//                            } else {
//                                flags &= ~Pattern.UNIX_LINES;
//                            }
//                            // TODO ここでフラグがUNIX_LINESモードでない場合はreplaceLineTerminator()で行末記号を置換すべき！
//                            continue;
//                            // 'd'を削除しない場合、OnigRegex#setWarningListener()でセットした
//                            // WarningListenerのwarning()が呼ばれるように鬼車本体を改造している
//                        }
//                        deleted.append(c);
//                    }
//                    final int last = deleted.length() - 1;
//                    if (last >= 0 && deleted.charAt(last) == '-') {
//                        // '-'で終わっている場合は'-'を削除する
//                        deleted.deleteCharAt(last);
//                    }
//                    if (deleted.length() != 0) {
//                        ptn.append(pattern.substring(start, begin1));
//                        ptn.append(deleted);
//                        ptn.append(pattern.substring(end1, regionEnd));
//                        
//                    } else {
//                        if (region.count() > 2) {
//                            final int begin = region.begin(2);
//                            if (begin != OnigRegion.ONIG_REGION_NOTPOS) {
//                                ptn.append(pattern.substring(start, begin1));
//                                ptn.append(pattern.substring(begin, regionEnd));
//                            }
//                        }
//                    }
//                }
//            }
//            start = regionEnd;
//        } while ((start < end) && (pos = embeddedFlag.search(pattern, end, start, end, region)) >= 0);
//        if (start < end) {
//            ptn.append(pattern.substring(start));
//        }
//        return ptn.toString();
//    }
    
//    /**
//     * 正規表現に含まれる行末記号をJavaの行末記号表現に置き換える
//     * @param flags
//     * @param pattern
//     * @return
//     */
//    private static String replaceLineTerminator(int flags, String pattern) {
//        if (((flags & Pattern.MULTILINE) != 0)
//                && ((flags & Pattern.UNIX_LINES) == 0)) {
//            // MULTILINEモード且つUNIX_LINESモードでない場合は行末記号を置き換える
//            return replaceLineTerminator(pattern);
//        }
//        return pattern;
//    }
//    
//    /**
//     * 埋め込みフラグ表現 '(?idsmux-idsmux:X)' を抽出する為の正規表現オブジェクト
//     * TODO 閉じ括弧がエスケープされていた場合に対応していないため要改修！
//     * MEMO '(?idsmux-idsmux)'と'(?idsmux-idsmux:X)'を分けて検索すれば対応可能か？
//     */
//    private static final OnigRegex embeddedFlag
//            = new OnigRegex(
////                    "\\(\\?([idsmux]*-?[idsmux]+)\\)|\\(\\?([idsmux]*-?[idsmux]+):.*\\)",
//                    "\\(\\?([idsmux]*-?[idsmux]+)",
//                    OnigSyntaxType.ONIG_SYNTAX_JAVA,
//                    OnigOptionType.ONIG_OPTION_MULTILINE
//                );
//    
//    /**
//     * 鬼車が対応していない埋め込みフラグ表現'(?d)'及び'(?u)'を削除する。
//     * @param pattern
//     * @return 埋め込みフラグ表現を削除した正規表現パターン文字列
//     */
//    private static String deleteUnsupportedEmbeddedFlag(final int flags, String pattern) {
//        final int end = pattern.length();
//        final OnigRegion region = new OnigRegion();
//        int pos = embeddedFlag.search(pattern, end, 0, end, region);
//        if (pos < 0) {
//            // 埋め込みフラグ表現が存在しない場合
//            return replaceLineTerminator(flags, pattern);
//        }
//        
//        // 正規表現に埋め込みフラグ表現が存在する場合は削除する
//        final StringBuilder ptn = new StringBuilder(end);
//        int start = 0;
//        do {
//            final int regionEnd = region.end(0);
//            if (isNotEscaped(pattern, pos)) {
//                // 埋め込みフラグ表現がエスケープされていない場合
//                final int begin1 = region.begin(1);
//                final int end1 = region.end(1);
//                boolean isActive = true;
//                int option = flags;
//                final StringBuilder deleted = new StringBuilder(end1 - begin1);
//                for (int i = begin1; i < end1; ++i) {
//                    final char c = pattern.charAt(i);
//                    if (c == '-') {
//                        isActive = false;
//
//                    } else if (c == 'm') {
//                        if (isActive) {
//                            option |= Pattern.MULTILINE;
//                        } else {
//                            option &= ~Pattern.MULTILINE;
//                        }
//                    } else if (c == 'd') {
//                        if (isActive) {
//                            option |= Pattern.UNIX_LINES;
//                        } else {
//                            option &= ~Pattern.UNIX_LINES;
//                        }
//                        continue;
//
//                    } else if (c == 'u') {
//                        if (isActive) {
//                            option |= Pattern.UNICODE_CASE;
//                        } else {
//                            option &= ~Pattern.UNICODE_CASE;
//                        }
//                        continue;
//                    }
//                    deleted.append(c);
//                }
//                final int last = deleted.length() - 1;
//                if (last >= 0 && deleted.charAt(last) == '-') {
//                    // '-'で終わっている場合は'-'を削除する
//                    deleted.deleteCharAt(last);
//                }
//                if (deleted.length() != 0) {
//                    String substr = pattern.substring(start, begin1);
//                    if (pos > start) {
//                        if ((option & Pattern.UNIX_LINES) == 0) {
//                            // UNIX_LINESモードでない場合は行末記号を置き換える
//                            substr = replaceLineTerminator(substr);
//                        }
//                    }
//                    ptn.append(substr);
//                    ptn.append(deleted);
//                    ptn.append(
//                            deleteUnsupportedEmbeddedFlag(option, pattern.substring(end1))
//                        );
//                } else {
//                    if (pos > start) {
//                        String substr = pattern.substring(start, pos);
//                        if ((option & Pattern.UNIX_LINES) == 0) {
//                            // UNIX_LINESモードでない場合は行末記号を置き換える
//                            substr = replaceLineTerminator(substr);
//                        }
//                        ptn.append(substr);
//                    }
//                    ptn.append(
//                            deleteUnsupportedEmbeddedFlag(option, pattern.substring(end1))
//                        );
//
//                }
//            }
//            start = regionEnd;
//        } while ((start < end) && (pos = embeddedFlag.search(pattern, end, start, end, region)) >= 0);
//        if (start < end) {
//            ptn.append(pattern.substring(start));
//        }
//        return ptn.toString();
//    }
//    
//    private static String parseEnclose(final int flags, StringBuilder pattern, int i) {
//        return null;
//    }
//    
//    private static String preprocessUnsupportedEmbeddedFlag(final int flags, String pattern) {
//        for (final char c : pattern.toCharArray()) {
//            
//        }
//        final int end = pattern.length();
//        final OnigRegion region = new OnigRegion();
//        int pos = embeddedFlag.search(pattern, end, 0, end, region);
//        if (pos < 0) {
//            // 埋め込みフラグ表現が存在しない場合
//            return replaceLineTerminator(flags, pattern);
//        }
//        
//        // 正規表現に埋め込みフラグ表現が存在する場合は削除する
//        final StringBuilder ptn = new StringBuilder(end);
//        int start = 0;
//        do {
//            final int regionEnd = region.end(0);
//            if (isNotEscaped(pattern, pos)) {
//                // 埋め込みフラグ表現がエスケープされていない場合
//                final int begin1 = region.begin(1);
//                final int end1 = region.end(1);
//                boolean isActive = true;
//                int option = flags;
//                final StringBuilder deleted = new StringBuilder(end1 - begin1);
//                for (int i = begin1; i < end1; ++i) {
//                    final char c = pattern.charAt(i);
//                    if (c == '-') {
//                        isActive = false;
//
//                    } else if (c == 'm') {
//                        if (isActive) {
//                            option |= Pattern.MULTILINE;
//                        } else {
//                            option &= ~Pattern.MULTILINE;
//                        }
//                    } else if (c == 'd') {
//                        if (isActive) {
//                            option |= Pattern.UNIX_LINES;
//                        } else {
//                            option &= ~Pattern.UNIX_LINES;
//                        }
//                        continue;
//
//                    } else if (c == 'u') {
//                        if (isActive) {
//                            option |= Pattern.UNICODE_CASE;
//                        } else {
//                            option &= ~Pattern.UNICODE_CASE;
//                        }
//                        continue;
//                    }
//                    deleted.append(c);
//                }
//                final int last = deleted.length() - 1;
//                if (last >= 0 && deleted.charAt(last) == '-') {
//                    // '-'で終わっている場合は'-'を削除する
//                    deleted.deleteCharAt(last);
//                }
//                if (deleted.length() != 0) {
//                    String substr = pattern.substring(start, begin1);
//                    if (pos > start) {
//                        if ((option & Pattern.UNIX_LINES) == 0) {
//                            // UNIX_LINESモードでない場合は行末記号を置き換える
//                            substr = replaceLineTerminator(substr);
//                        }
//                    }
//                    ptn.append(substr);
//                    ptn.append(deleted);
//                    ptn.append(
//                            deleteUnsupportedEmbeddedFlag(option, pattern.substring(end1))
//                        );
//                } else {
//                    if (pos > start) {
//                        String substr = pattern.substring(start, pos);
//                        if ((option & Pattern.UNIX_LINES) == 0) {
//                            // UNIX_LINESモードでない場合は行末記号を置き換える
//                            substr = replaceLineTerminator(substr);
//                        }
//                        ptn.append(substr);
//                    }
//                    ptn.append(
//                            deleteUnsupportedEmbeddedFlag(option, pattern.substring(end1))
//                        );
//
//                }
//            }
//            start = regionEnd;
//        } while ((start < end) && (pos = embeddedFlag.search(pattern, end, start, end, region)) >= 0);
//        if (start < end) {
//            ptn.append(pattern.substring(start));
//        }
//        return ptn.toString();
//        
//    }
}
