/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.tools;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Map;

import org.onion_lang.onion.compiler.CompilationConfiguration;
import org.onion_lang.onion.compiler.CompiledClass;
import org.onion_lang.onion.compiler.OnionCompiler;
import org.onion_lang.onion.compiler.utility.Messages;
import org.onion_lang.onion.compiler.utility.Systems;

/**
 * @author Kota Mizushima
 * 
 */
public class OnionCompilerFrontend {
  private CommandLineParser commandLineParser;
  
  private static String message(String key){
    return Messages.get(key);
  }
  
  private static String format(String template, String parameter){
    return MessageFormat.format(template, new Object[]{parameter});
  }
  
  private static OptionConfiguration conf(String optionName, boolean requireArgument){
    return new OptionConfiguration(optionName, requireArgument);
  }
    
  public OnionCompilerFrontend(){
    commandLineParser = new CommandLineParser(
      new OptionConfiguration[]{
        conf("-classpath", true),
        conf("-sourcepath", true),
        conf("-encoding", true),
        conf("-d", true)});
  }
  
  public int run(String[] commandLine){
    ParseSuccess result = parseCommandLine(commandLine);
    if(result == null) return -1;
    CompiledClass[] binaries = startCompile((ParseSuccess)result);
    if(binaries == null) return -1;
    boolean generated = generateFile(binaries);
    if(!generated) return -1;
    return 0;
  }
  
  private String getSimpleName(String fqcn){
    int index = fqcn.lastIndexOf(".");
    if(index < 0){
      return fqcn;
    }else{
      return fqcn.substring(index + 1, fqcn.length());
    }
  }
  
  public String getOutputPath(String outDir, String fqcn){
    String name = getSimpleName(fqcn);
    return outDir + Systems.getFileSeparator() + name + ".class";
  }
   
  private boolean generateFile(CompiledClass[] binaries){
    for (int i = 0; i < binaries.length; i++) {
      CompiledClass binary = binaries[i];
      String outDir = binary.getOutputPath();
      new File(outDir).mkdirs();
      String outPath = getOutputPath(outDir, binary.getClassName());
      File targetFile = new File(outPath);
      try{
        if(!targetFile.exists()) targetFile.createNewFile();
        BufferedOutputStream out = 
          new BufferedOutputStream(new FileOutputStream(targetFile));
        out.write(binary.getContent());
        out.close();
      }catch(FileNotFoundException e){
        e.printStackTrace();
        return false;
      }catch(IOException e){
        e.printStackTrace();
        return false;
      }
    }
    return true;
  }
  
  private ParseSuccess parseCommandLine(String[] commandLine){
    ParseResult result = commandLineParser.parse(commandLine);
    if(result.getStatus() == ParseResult.FAILURE){
      ParseFailure failure = (ParseFailure)result;
      String[] lackedOptions = failure.getLackedOptions();
      String[] invalidOptions = failure.getInvalidOptions();
      for (int i = 0; i < invalidOptions.length; i++) {
        System.err.println(format(message("option.invalidArgument"), invalidOptions[i]));
      }
      for (int i = 0; i < lackedOptions.length; i++) {
        System.err.println(format(message("option.noArgument"), lackedOptions[i]));
      }
      return null;
    }
    return (ParseSuccess)result;
  }
      
  private CompiledClass[] startCompile(ParseSuccess success){
    Map option = success.getOptions();
    String sourcepath = (String)option.get("-sourcepath");
    String classpath = (String)option.get("-classpath");
    String encoding = (String)option.get("-encoding");
    String outputDirectory = (String)option.get("-d");
    String[] fileNames = (String[])success.getArguments().toArray(new String[0]);
    OnionCompiler compiler = new OnionCompiler(
        new CompilationConfiguration(
          sourcepath != null ? pathArray(sourcepath) : new String[]{"."},
          classpath != null ? pathArray(classpath) : new String[]{"."},
          encoding != null ? encoding : "Shift_JIS",
          outputDirectory != null ? outputDirectory : ".",
          1));
    return compiler.compile(fileNames);    
  }
    
  private static String[] pathArray(String path){
    return path.split(Systems.getPathSeparator());
  }
  
  public static void main(String[] args) throws Exception{
    new OnionCompilerFrontend().run(args);
  }
}
