module ActiveRecord

  # 編集画面の情報を記録するモジュール。
  module DisplayMonitor

    def self.included(base)
      base.class_eval do
        alias_method_chain :create, :display
        alias_method_chain :update, :display
      end
    end

    # <tt>create</tt> を置き換える。
    # カラム <tt>created_in</tt> や <tt>updated_in</tt> が存在する場合は当該画面を記録する。
    def create_with_display
      write_attribute(:created_in, Display.current_id) if self.class.column_names.include?("created_in") && created_in.nil?
      write_attribute(:updated_in, Display.current_id) if self.class.column_names.include?("updated_in")
      create_without_display
    end

    # <tt>update</tt> を置き換える。
    # カラム <tt>updated_in</tt> が存在する場合は当該画面を記録する。
    def update_with_display
      write_attribute(:updated_in, Display.current_id) if self.class.column_names.include?("updated_in")
      update_without_display
    end

    # 作成画面を <tt>Display</tt> インスタンスとして返す。
    # 存在しない場合は nil を返す。
    def created_in
      Display.find(read_attribute(:created_in))
    rescue ActiveRecord::RecordNotFound
      nil
    end

    # 更新画面を <tt>Display</tt> インスタンスとして返す。
    # 存在しない場合は nil を返す。
    def updated_in
      Display.find(read_attribute(:updated_in))
    rescue ActiveRecord::RecordNotFound
      nil
    end
  end
end

ActiveRecord::Base.class_eval do
  include ActiveRecord::DisplayMonitor
end
