# -*- coding: utf-8 -*-
# app/controllers/list_controller.rb のクラス ListController をテストする。

require File.dirname(__FILE__) + '/../test_helper'
require 'list_controller'

# Re-raise errors caught by the controller.
class ListController; def rescue_action(e) raise e end; end

class ListControllerTest < Test::Unit::TestCase
  fixtures(:languages, :domains, :people, :users,
           :groups, :companies, :organizations,
           :products, :displays, :items,
           :grant_ons, :permissions)

  def setup
    @controller = ListController.new
    @request    = ActionController::TestRequest.new
    @response   = ActionController::TestResponse.new

    # set logged in
    @request.session[:user_id] = users(:tesla)

    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  # index に id なしでアクセスした場合にエラーを返す。
  def test_index__without_id
    assert_raise(ArgumentError) do
      get :index
    end
  end

  # index に存在しない id でアクセスした場合にエラーを返す。
  def test_index__no_such_id
    assert_raise(ActiveRecord::RecordNotFound) do
      get :index, :id => 9876543210
    end
  end

  # add に適切な id を与えると新しい一覧を追加する。
  def test_add__success
    xhr :post, :add, :id => 8
    assert_response :success
    assert_template "index"
    assert_select "div"
  end

  # destroy に適切な id および display_id を与えると削除する。
  def test_destroy__success
    xhr :post, :destroy, :id => 8, :display_id => 20002
    assert_response :success
    assert_equal "text/javascript", @response.content_type
    assert_raise(ActiveRecord::RecordNotFound) do
      Display.find(20002)
    end
  end

  # edit で既存の一覧を編集するための画面を開く。
  def test_edit__xhr
    xhr :post, :edit, :id => 8, :display_id => 20002
    assert_response :success
    assert_template "edit"
    assert_select "table"
  end

  # update で既存の一覧を更新する。
  def test_update
    post :update, :id => 8, :display_id => 20002, :display => {:name => "変更後"}
    assert_response :redirect
    assert_redirected_to :action => "index", :id => 8
  end

  # order_selected で選択された項目の並び替える。
  def test_order_selected
    xhr :post, :order_selected, :id => 8, :display_id => 20002, :selected_items => []
    assert_response :success
    assert_equal " ", @response.body
  end

  # order_unselected で選択されていない項目に関する情報を取得する。
  def test_order_unselected
    xhr :post, :order_unselected, :id => 8, :display_id => 20002, :unselected_items => []
    assert_response :success
    assert_equal " ", @response.body
  end

  # import で共有一覧をインポートするための画面を表示する。
  def test_import
    xhr :post, :import, :id => 8
    assert_response :success
    assert_template "_import"
    assert_select "div"
  end

  # search_to_import では target 情報が与えられていない場合はそのまま表示する。
  def test_search_to_import__without_target
    xhr :post, :search_to_import, :id => 8
    assert_response :success
    assert_template "_import"
    assert_select "div"
  end

  # search_to_import では target 情報に従って検索を行う。
  def test_search_to_import__success
    xhr :post, :search_to_import, :id => 8, :target => {:display_sub => "", :create_sub => ""}
    assert_response :success
    assert_template "_import"
    assert_select "div"
  end

  # glance_to_import で共有一覧の項目の情報を表示する。
  def test_glance_to_import
    xhr :post, :glance_to_import, :id => 8, :shared_list_id => 1
    assert_response :success
    assert_template "_glance"
    assert_select "div"
  end

  # select_to_import で公開先でないユーザが共有一覧を選択しようとするとエラーになる。
  def test_select_to_import__not_allowed
    assert_raise(ArgumentError) do
      xhr :post, :select_to_import, :id => 8, :shared_list_id => 1
    end
  end

  # select_to_import で共有一覧を選択する。
  def test_select_to_import__success
    post :select_to_import, :id => 8, :shared_list_id => 2
    assert_response :redirect
    assert_redirected_to :action => "index", :id => 8
  end
end
