# -*- coding: utf-8 -*-
# lib/custom_validations.rb のモジュール CustomValidations をテストする。

require 'test_helper'
$KCODE = "u"
require "custom_validations"

class CustomValidationsTest < ActiveSupport::TestCase
  fixtures :domains, :users, :companies

  def define_dummy_class(&block)
    @class = Class.new
    @class.class_eval do
      include CustomValidations::Validateable
      include CustomValidations::OnDestroy
      extend CustomValidations
      attr_accessor :data

      def initialize(options = nil)
        options.each {|k, v| self.instance_variable_set("@#{k}", v)} if options
      end
    end
    @class.class_eval(&block)
  end

  def assert_valid(data, on = nil, options = {}, &block)
    o = @class.new(options)
    o.data = data
    case on
    when :destroy
      assert o.valid_on_destroy?, "#{data.inspect} should be valid"
    else
      assert o.valid?, "#{data.inspect} should be valid"
    end
    block.call(o) if block_given?
  end

  def assert_not_valid(data, on = nil, options = {}, &block)
    o = @class.new(options)
    o.data = data
    case on
    when :destroy
      assert !o.valid_on_destroy?, "#{data.inspect} should not be valid"
    else
      assert !o.valid?, "#{data.inspect} should not be valid"
    end
    block.call(o) if block_given?
  end

  TEST_DATA = [
    nil,
    STRING_TEST_DATA = [
      "",
      INTEGER_STRING_TEST_DATA = [
        "0",
        "-0",
        "+0",
        "100",
        "-10",
        "+10",
        "5",
      ],
      INVALID_FLOAT_STRING_TEST_DATA = [
        "0.",
        ".0",
        "+0.",
        "-0.",
        "+.0",
        "-.0",
      ],
      FLOAT_STRING_TEST_DATA = [
        "0.0",
        "-0.0",
        "+0.0",
        "10.0",
        "-10.0",
        "+10.0",
        "0.01",
        "-0.01",
        "+0.01",
        "0.0123",
        "-0.0123",
        "+0.0123",
      ],
      "string",
      "String",
      "StrinG",
      "StRinG",
      "STRING",
      "string123",
      "String123",
      "StrinG123",
      "StRinG123",
      "STRING123",
      "123string123",
      "123String123",
      "123StrinG123",
      "123StRinG123",
      "123STRING123",
      "Abc",
      INCLUSION_CHARS_TEST_DATA = [
        "A",
        "b",
        "Ab",
        "Ab5",
        "A5b",
        "5Ab",
        "b5A5",
      ],
    ].flatten,
    INTEGER_TEST_DATA = [
      0,
      5,
      100,
      -10,
    ],
    FLOAT_TEST_DATA = [
      0.0,
      5.0,
      100.0,
      -10.0,
      10.01,
      -1.01,
    ],
    HALFWIDTH_KATAKANA_TEST_DATA = [
      [0xff83, 0xff7d, 0xff84].pack('U*'),
      [0xff83, 0xff9e, 0xff70, 0xff80].pack('U*'),
    ],
    FULLWIDTH_KATAKANA_TEST_DATA = [
      [0x30c6, 0x30b9, 0x30c8].pack('U*'),
      [0x30c7, 0x30fc, 0x30bf].pack('U*'),
    ],
    [0xa1].pack('U*'), # can not convert to cp932
  ].flatten

  def assert_validates_ok(ok_pattern, &block)
    (TEST_DATA - ok_pattern).each do |data|
      assert_not_valid(data, nil, {}, &block)
    end
    ok_pattern.each do |data|
      assert_valid(data)
    end
  end

  def assert_validates_ng(ng_pattern)
    (TEST_DATA - ng_pattern).each do |data|
      assert_valid(data)
    end
    ng_pattern.each do |data|
      assert_not_valid(data)
    end
  end

  # ActiveRecord::Validations の validates_presence_of の動作をテストする。
  def test_validates_presence_of
    define_dummy_class do
      validates_presence_of :data
    end
    assert_validates_ng([
        nil,
        "",
      ])
  end

  # ActiveRecord::Validations の validates_numericality_of の動作をテストする。
  def test_validates_numericality_of
    define_dummy_class do
      validates_numericality_of :data
    end
    begin
      Float("0.")
      ng_zero_point_data = []
    rescue ArgumentError
      ng_zero_point_data = ["0.", "+0.", "-0."]
    end
    assert_validates_ok([
        INTEGER_STRING_TEST_DATA,
        INVALID_FLOAT_STRING_TEST_DATA - ng_zero_point_data,
        FLOAT_STRING_TEST_DATA,
        INTEGER_TEST_DATA,
        FLOAT_TEST_DATA,
      ].flatten)
  end

  # CustomValidations の validates_non_negative_integer_of の動作をテストする。
  def test_validates_non_negative_integer_of
    define_dummy_class do
      validates_non_negative_integer_of :data
    end
    assert_validates_ok([
        "0",
        "+0",
        "100",
        "+10",
        "5",
        0,
        5,
        100,
      ])
  end

  # CustomValidations の validates_integer_of の動作をテストする。
  def test_validates_integer_of
    define_dummy_class do
      validates_integer_of :data
    end
    assert_validates_ok([
        INTEGER_STRING_TEST_DATA,
        INTEGER_TEST_DATA,
      ].flatten)
  end

  # CustomValidations の validates_non_negative_float_of の動作をテストする。
  def test_validates_non_negative_float_of
    define_dummy_class do
      validates_non_negative_float_of :data
    end
    assert_validates_ok([
        "0.0",
        "+0.0",
        "10.0",
        "+10.0",
        "0.01",
        "+0.01",
        "0.0123",
        "+0.0123",
        0.0,
        5.0,
        100.0,
        10.01,
      ])
  end

  # CustomValidations の validates_float_of の動作をテストする。
  def test_validates_float_of
    define_dummy_class do
      validates_float_of :data
    end
    assert_validates_ok([
        FLOAT_STRING_TEST_DATA,
        FLOAT_TEST_DATA,
      ].flatten)
  end

  # CustomValidations の validates_zero_of の動作をテストする。
  def test_validates_zero_of
    define_dummy_class do
      stubs(:warn) { } # suspends warning message in testings
      validates_zero_of :data
    end
    # ok_pattern = ["0", "+0", "-0", "0.0", "+0.0", "-0.0", 0, 0.0]
    assert_validates_ok(NONZERO_OK_PATTERN)
  end

  NONZERO_OK_PATTERN = [
    "100",
    "-10",
    "+10",
    "5",

    "10.0",
    "-10.0",
    "+10.0",
    "0.01",
    "-0.01",
    "+0.01",
    "0.0123",
    "-0.0123",
    "+0.0123",

    "123string123",
    "123String123",
    "123StrinG123",
    "123StRinG123",
    "123STRING123",

    "5Ab",

    5,
    100,
    -10,

    5.0,
    100.0,
    -10.0,
    10.01,
    -1.01,
  ]

  # CustomValidations の validates_nonzero_of の動作をテストする。
  def test_validates_nonzero_of
    define_dummy_class do
      validates_nonzero_of :data
    end
    assert_validates_ok(NONZERO_OK_PATTERN)
  end

  # CustomValidations の validates_upper_case_of の動作をテストする。
  def test_validates_upper_case_of
    define_dummy_class do
      validates_upper_case_of :data
    end
    assert_validates_ok([
        "STRING",
        "A",
      ])
  end

  # CustomValidations の validates_lower_case_of の動作をテストする。
  def test_validates_lower_case_of
    define_dummy_class do
      validates_lower_case_of :data
    end
    assert_validates_ok([
        "string",
        "b",
      ])
  end

  # CustomValidations の validates_alphabetic_of の動作をテストする。
  def test_validates_alphabetic_of
    define_dummy_class do
      validates_alphabetic_of :data
    end
    assert_validates_ok([
        "string",
        "String",
        "StrinG",
        "StRinG",
        "STRING",
        "Abc",
        "A",
        "b",
        "Ab",
      ])
  end

  # CustomValidations の validates_alphanumeric_of の動作をテストする。
  def test_validates_alphanumeric_of
    define_dummy_class do
      validates_alphanumeric_of :data
    end
    ok_data = STRING_TEST_DATA - ["", "+0", "-0", "+10", "-10"]
    ok_data -= INVALID_FLOAT_STRING_TEST_DATA
    ok_data -= FLOAT_STRING_TEST_DATA
    ok_data -= FLOAT_TEST_DATA
    ok_data += [0, 5, 100]
    assert_validates_ok(ok_data)
  end

  # CustomValidations の validates_halfwidth_katakana_of の動作をテストする。
  def test_validates_halfwidth_katakana_of
    halfwidth_ideographic_full_stop = [0xff61].pack('U')
    halfwidth_katakana_semi_voiced_sound_mark = [0xff9f].pack('U')
    define_dummy_class do
      validates_halfwidth_katakana_of :data
    end
    assert_validates_ok(HALFWIDTH_KATAKANA_TEST_DATA)
    assert_valid(halfwidth_ideographic_full_stop)
    assert_valid(halfwidth_katakana_semi_voiced_sound_mark)
  end

  # CustomValidations の validates_fullwidth_of の動作をテストする。
  # FIXME: how to check width of chars
  def test_validates_fullwidth_of
    define_dummy_class do
      validates_fullwidth_of :data
    end
    assert_validates_ok(FULLWIDTH_KATAKANA_TEST_DATA)
  end

  # CustomValidations の validates_fullwidth_katakana_of の動作をテストする。
  def test_validates_fullwidth_katakana_of
    katakana_hiragana_double_hyphen = [0x30a0].pack('U')
    katakana_letter_small_ro = [0x31ff].pack('U')
    define_dummy_class do
      validates_fullwidth_katakana_of :data
    end
    assert_validates_ok(FULLWIDTH_KATAKANA_TEST_DATA)
    assert_valid(katakana_hiragana_double_hyphen)
    assert_valid(katakana_letter_small_ro)
  end

  # CustomValidations の validates_inclusion_chars_of の動作をテストする。
  def test_validates_inclusion_chars_of
    actual = ["A", "b", 5]
    define_dummy_class do
      validates_inclusion_chars_of :data, :chars => actual
    end
    assert_validates_ok(INCLUSION_CHARS_TEST_DATA + ["5", 5]) do |o|
      assert_equal "rfw|message|error|Data can contain \"Ab5\" only", o.errors.on(:data)
    end
  end

  # ActiveRecord::Validations の validates_length_of の動作をテストする。
  def test_validates_length_of
    define_dummy_class do
      validates_length_of :data, :minimum => 4
      validates_length_of :data, :maximum => 5
      # or validates_length_of :data, :in => 4..5
    end
    assert_not_valid(nil)
    assert_not_valid("")
    assert_not_valid("1")
    assert_not_valid("12")
    assert_not_valid("123")
    assert_valid("1234")
    assert_valid("12345")
    assert_not_valid("123456")
    assert_not_valid(HALFWIDTH_KATAKANA_TEST_DATA[0])
    assert_valid(HALFWIDTH_KATAKANA_TEST_DATA[1])
    assert_not_valid(FULLWIDTH_KATAKANA_TEST_DATA[0])
    inverted_exclamation_mark = [0xa1].pack('U')
    assert_not_valid(inverted_exclamation_mark*3)
    assert_valid(inverted_exclamation_mark*4)
    surrogate_pair = [0xd842, 0xdf9f].pack('U*') # U+20B9F
    assert_not_valid(surrogate_pair)
    assert_valid(surrogate_pair*2)
  end

  # CustomValidations の validates_integral_length_of の動作をテストする。
  def test_validates_integral_length_of
    actual = 2..4
    define_dummy_class do
      validates_integral_length_of :data, :in => actual
    end
    assert_not_valid("0")
    assert_not_valid("1")
    assert_valid("12")
    assert_valid("123")
    assert_valid("1234")
    assert_not_valid("12345")
    assert_not_valid("123456")
    assert_not_valid("0.1")
    assert_not_valid("1.1")
    assert_valid("12.1")
    assert_valid("123.1")
    assert_valid("1234.1")
    assert_not_valid("12345.1")
    assert_not_valid("123456.1")
  end

  # CustomValidations の validates_fractional_length_of の動作をテストする。
  def test_validates_fractional_length_of
    actual = 2..4
    define_dummy_class do
      validates_fractional_length_of :data, :in => actual
    end
    assert_not_valid("0")
    assert_not_valid("0.1")
    assert_valid("0.12")
    assert_valid("0.123")
    assert_valid("0.1234")
    assert_valid("99999.123")
    assert_not_valid("0.12345")
    assert_not_valid("0.123456")
  end

  # CustomValidations の validates_year_of の動作をテストする。
  def test_validates_year_of
    define_dummy_class do
      validates_year_of :data
    end
    assert_valid(nil)
    assert_valid("")
    assert_not_valid(-1)
    assert_not_valid(0)
    assert_valid(1)
    assert_not_valid("1")
    assert_not_valid("010")
    assert_valid("0001")
    assert_valid(999)
    assert_valid(1000)
    assert_valid(1970)
    assert_valid(2000)
    assert_valid(9999)
    assert_valid("9999")
    assert_not_valid("09999")
    assert_not_valid(10000)
    assert_not_valid("010000")
    assert_not_valid("abcd")
  end

  # CustomValidations の validates_year_month_of の動作をテストする。
  def test_validates_year_month_of
    define_dummy_class do
      validates_year_month_of :data
    end
    assert_valid(nil)
    assert_valid("")
    assert_not_valid(-1)
    assert_not_valid(0)
    assert_not_valid(2000)
    assert_not_valid(107) # 10年7月でも1年07月でもない。
    assert_valid(200007)
    assert_valid("2000/07")
    assert_valid("2000-07")
    assert_valid("2000/7")
    assert_valid("2000-7")
    assert_not_valid(2000007)
    assert_not_valid("20000/07")
    assert_not_valid("20000-07")
    assert_not_valid("0-0")
    assert_not_valid("0/0")
    assert_not_valid("0-1")
    assert_not_valid("0/1")
    assert_not_valid("1-0")
    assert_not_valid("1/0")
    assert_not_valid("1/1")
    assert_not_valid("1-1")
    assert_not_valid("1/12")
    assert_not_valid("1-12")
    assert_not_valid("1/13")
    assert_not_valid("1-13")
    assert_not_valid(999900)
    assert_not_valid("9999/00")
    assert_not_valid("9999-00")
    assert_valid(999901)
    assert_valid("9999/01")
    assert_valid("9999-01")
    assert_valid(999912)
    assert_valid("9999/12")
    assert_valid("9999-12")
    assert_not_valid(999913)
    assert_not_valid("9999/13")
    assert_not_valid("9999-13")
    assert_not_valid(1000001)
    assert_not_valid("10000/01")
    assert_not_valid("10000-01")
    assert_not_valid("10000/1")
    assert_not_valid("10000-1")
    assert_not_valid("09/004")
    assert_not_valid("09-004")
    assert_not_valid("008/005")
    assert_not_valid("008-005")
    assert_not_valid("2009/009")
    assert_not_valid("2009-009")
    assert_valid("200911")
  end

  # CustomValidations の validates_year_month_day_of の動作をテストする。
  def test_validates_year_month_day_of
    define_dummy_class do
      validates_year_month_day_of :data
    end
    assert_valid(nil)
    assert_valid("")
    assert_not_valid(-1)
    assert_not_valid(0)
    assert_not_valid(2000)
    assert_not_valid(200007)
    assert_valid(20000709)
    assert_valid("2000/07/09")
    assert_valid("2000-07-09")
    assert_not_valid("2000-07/09")
    assert_not_valid("2000/07-09")
    assert_not_valid(200709)
    assert_not_valid("20/07/09")
    assert_not_valid("20-07-09")
    assert_not_valid("1/1/1")
    assert_not_valid("1-1-1")
    assert_not_valid("1/1/0")
    assert_not_valid("1-1-0")
    assert_not_valid("1/0/1")
    assert_not_valid("1-0-1")
    assert_not_valid("0/1/1")
    assert_not_valid("0-1-1")
    assert_valid(99991231)
    assert_valid("9999/12/31")
    assert_valid("9999-12-31")
    assert_not_valid(99991232)
    assert_not_valid("9999/12/32")
    assert_not_valid("9999-12-32")
    assert_not_valid(99991331)
    assert_not_valid("9999/13/31")
    assert_not_valid("9999-13-31")
    assert_not_valid(100001231)
    assert_not_valid("10000/12/31")
    assert_not_valid("10000-12-31")
    assert_valid(20000229)
    assert_valid("2000/2/29")
    assert_valid("2000-2-29")
    assert_not_valid(19990229)
    assert_not_valid("1999/2/29")
    assert_not_valid("1999-2-29")
    assert_not_valid("2009/aa/01")
    assert_not_valid("2009-aa-01")
    assert_not_valid("009/02/28")
    assert_not_valid("009-02-28")
  end

  # CustomValidations の validates_hour_of の動作をテストする。
  def test_validates_hour_of
    define_dummy_class do
      validates_hour_of :data
    end
    0.upto(23) do |i|
      assert_not_valid(i.to_s) if i < 10
      assert_valid("%02d" % i)
    end
    assert_not_valid('-1')
    assert_not_valid('24')
    assert_not_valid('25')
    assert_not_valid("000")
  end

  # CustomValidations の validates_hour_minute_of の動作をテストする。
  def test_validates_hour_minute_of
    define_dummy_class do
      validates_hour_minute_of :data
    end
    assert_not_valid("20:60")
    assert_not_valid("24:00")
    assert_not_valid("24:59")
    assert_not_valid("25:00")
    assert_not_valid("0:0")
    assert_not_valid("12:0")
    assert_valid("23:59")
    0.upto(23) do |hour|
      0.upto(59) do |minute|
        assert_valid("%02d:%02d" % [hour, minute])
      end
    end
  end

  # CustomValidations の validates_postal_code_of の動作をテストする。
  def test_validates_postal_code_of
    define_dummy_class do
      validates_postal_code_of :data
    end
    assert_valid(-1)
    assert_valid(0)
    assert_valid(24)
    assert_valid("000")
    assert_valid("000-0000")
    assert_valid("123-4567")
    assert_not_valid("0123(45)6789")
    assert_not_valid("123+4567")
  end

  # CustomValidations の validates_phone_number_of の動作をテストする。
  def test_validates_phone_number_of
    define_dummy_class do
      validates_phone_number_of :data
    end
    assert_valid(-1)
    assert_valid(0)
    assert_valid(24)
    assert_valid("000")
    assert_valid("000-0000")
    assert_valid("123-4567")
    assert_valid("0123(45)6789")
    assert_not_valid("123+4567")
  end

  # CustomValidations の validates_email_of の動作をテストする。
  def test_validates_email_of
    define_dummy_class do
      validates_email_of :data
    end
    assert_not_valid("invalid@f@@.b")
    assert_valid("valid@foo.bar")
  end

  # CustomValidations の validates_url_of の動作をテストする。
  def test_validates_url_of
    define_dummy_class do
      validates_url_of :data
    end
    assert_not_valid("http:// ")
    assert_valid("http://localhost:3000/?query=string#fragment")
  end

  # CustomValidations の validates_minimum_number_of の動作をテストする。
  def test_validates_minimum_number_of
    define_dummy_class do
      validates_minimum_number_of :data, :minimum => 5
    end
    assert_not_valid(0)
    assert_not_valid(1)
    assert_not_valid(2)
    assert_not_valid(3)
    assert_not_valid(4)
    assert_valid(5)
    assert_valid(6)
    assert_valid(7)
    assert_valid(8)
    assert_valid(9)
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__in
    define_dummy_class do
      validates_period_of :data, :in => "20010203".."20010205"
    end
    assert_not_valid("2000/11/30")
    assert_not_valid("20010202")
    assert_valid("20010203")
    assert_valid("20010204")
    assert_valid("20010205")
    assert_valid("2001/02/05")
    assert_not_valid("20010206")
    assert_not_valid("2008/01/31")
    assert_valid("2008/02/aa") # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__in_without_first
    define_dummy_class do
      validates_period_of :data, :in => "".."20010205"
    end
    assert_valid("2000/11/30")
    assert_valid("20010202")
    assert_valid("20010203")
    assert_valid("20010204")
    assert_valid("20010205")
    assert_valid("2001/02/05")
    assert_not_valid("20010206")
    assert_not_valid("2008/01/31")
    assert_valid("2008/02/aa") # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__in_without_last
    define_dummy_class do
      validates_period_of :data, :in => "20010203"..""
    end
    assert_not_valid("2000/11/30")
    assert_not_valid("20010202")
    assert_valid("20010203")
    assert_valid("20010204")
    assert_valid("20010205")
    assert_valid("2001/02/05")
    assert_valid("20010206")
    assert_valid("2008/01/31")
    assert_valid("2008/02/aa") # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__in_with_eval
    define_dummy_class do
      validates_period_of :data, :in => "x".."y"
      attr_accessor :x
      attr_accessor :y
    end
    options = {
      :x => "20010203",
      :y => "20010205",
    }
    assert_not_valid("2000/11/30", nil, options)
    assert_not_valid("20010202", nil, options)
    assert_valid("20010203", nil, options)
    assert_valid("20010204", nil, options)
    assert_valid("20010205", nil, options)
    assert_valid("2001/02/05", nil, options)
    assert_not_valid("20010206", nil, options)
    assert_not_valid("2008/01/31", nil, options)
    assert_valid("2008/02/aa", nil, options) # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__minimum
    define_dummy_class do
      validates_period_of :data, :minimum => "2001/02/03"
    end
    assert_not_valid("20010202")
    assert_not_valid("2000/02/01")
    assert_valid("20010203")
    assert_valid("20010204")
    assert_valid("20010205")
    assert_valid("20010206")
    assert_valid("2008/02/06")
    assert_valid("2008/02/aa") # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__minimum_blank
    define_dummy_class do
      validates_period_of :data, :minimum => ""
    end
    assert_valid("20010202")
    assert_valid("2000/02/01")
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__minimum_with_eval
    define_dummy_class do
      validates_period_of :data, :minimum => "x.y"
      attr_accessor :x
    end
    options = {:x => stub(:y => "20010203")}
    assert_not_valid("20010202", nil, options)
    assert_not_valid("2000/02/01", nil, options)
    assert_valid("20010203", nil, options)
    assert_valid("20010204", nil, options)
    assert_valid("20010205", nil, options)
    assert_valid("20010206", nil, options)
    assert_valid("2008/02/06", nil, options)
    assert_valid("2008/02/aa", nil, options) # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__maximum
    define_dummy_class do
      validates_period_of :data, :maximum => "2001-02-05"
    end
    assert_valid("20010202")
    assert_valid("20010203")
    assert_valid("20010204")
    assert_valid("20010205")
    assert_valid("2001/02/05")
    assert_not_valid("20010206")
    assert_not_valid("2008/02/05")
    assert_valid("2008/02/aa") # ok with invalid format
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__maximum_blank
    define_dummy_class do
      validates_period_of :data, :maximum => ""
    end
    assert_valid("20010202")
    assert_valid("2008/02/05")
  end

  # CustomValidations の validates_period_of の動作をテストする。
  def test_validates_period_of__maximum_with_eval
    define_dummy_class do
      validates_period_of :data, :maximum => "abc"
      attr_accessor :abc
    end
    options = {:abc => "20010205"}
    assert_valid("20010202", nil, options)
    assert_valid("20010203", nil, options)
    assert_valid("20010204", nil, options)
    assert_valid("20010205", nil, options)
    assert_valid("2001/02/05", nil, options)
    assert_not_valid("20010206", nil, options)
    assert_not_valid("2008/02/05", nil, options)
    assert_valid("2008/02/aa", nil, options) # ok with invalid format
  end

  # CustomValidations の validates_later_date_of の動作をテストする。
  def test_validates_later_date_of
    define_dummy_class do
      validates_later_date_of :data, :than => "2001/02/03"
    end
    assert_not_valid("20010201")
    assert_not_valid("20010202")
    assert_not_valid("20010203")
    assert_not_valid("2000/12/31") do |o|
      assert_match /2001[-\/]*02[-\/]*03/, o.errors.on(:data)
    end
    assert_valid("20010204")
    assert_valid("20010205")
    assert_valid("20010206")
    assert_valid("2001/12/01")
    assert_valid("2001/aa/01") # ok with invalid format
    assert_valid("2001/12/bb") # ok with invalid format
    assert_valid("2001/02/29") # ok with invalid date
  end

  # CustomValidations の validates_earlier_date_of の動作をテストする。
  def test_validates_earlier_date_of
    define_dummy_class do
      validates_earlier_date_of :data, :than => "20010204"
    end
    assert_valid("2001/01/31")
    assert_valid("20010201")
    assert_valid("20010202")
    assert_valid("20010203")
    assert_not_valid("20010204")
    assert_not_valid("20010205")
    assert_not_valid("20010206")
    assert_not_valid("2001/02/20") do |o|
      assert_match /2001[-\/]*02[-\/]*04/, o.errors.on(:data)
    end
    assert_valid("2001/aa/01") # ok with invalid format
    assert_valid("2001/12/bb") # ok with invalid format
    assert_valid("1999/02/29") # ok with invalid date
  end

  # CustomValidations の validates_row_of の動作をテストする。
  def test_validates_row_of
    define_dummy_class do
      validates_row_of :data, :table => "domains", :column => "name"
    end
    assert_valid("wonda")
    assert_valid("fire")
    assert_not_valid("ice")
    assert_not_valid("wanda")
  end

  # ActiveRecord::Validations の validates_presence_of で検証する項目の値が
  # 変更された場合の動作をテストする。
  def test_once_validates_presence_of
    define_dummy_class do
      validates_length_of :data, :maximum => 5, :allow_nil => true
    end
    data = nil

    o = @class.new
    o.extend CustomValidations::SingletonValidateable
    o.validates_presence_of :data
    o.data = data
    assert !o.valid?, "#{data.inspect} should not be valid temporary"
    o.data = "abc"
    assert o.valid?, "should be valid on length"
    o.data = "xxxxxxx"
    assert !o.valid?, "should not be valid on length"

    o = @class.new
    o.data = data
    assert o.valid?, "#{data.inspect} should be valid again"
    o.data = "abc"
    assert o.valid?, "should be valid on length"
    o.data = "xxxxxxx"
    assert !o.valid?, "should not be valid on length"
  end

  # CustomValidations の validates_isolated_of の動作をテストする。
  def test_validates_isolated_of__ok
    define_dummy_class do
      validates_isolated_of :data, :from => 'User#domain_id,Company#domain_id'
    end
    assert_valid(99999, :destroy)
  end

  # CustomValidations の validates_isolated_of の動作をテストする。
  def test_validates_isolated_of__ng
    define_dummy_class do
      validates_isolated_of :data, :from => 'User#domain_id,Company#domain_id'
    end
    assert_not_valid(1, :destroy)
  end
end
