# -*- coding: utf-8 -*-
# Filters added to this controller apply to all controllers in the application.
# Likewise, all the methods added will be available for all controllers.

require "fragment_hash"
require "display_monitor"
require "user_system"

if defined?(GetText) && !File.exist?(File.join(RAILS_ROOT, "locale"))
  # makemo
  require "gettext/utils"
  GetText.create_mofiles(true, "po", "locale")
end

class ApplicationController < ActionController::Base
  #helper :all # include all helpers, all the time

  include ExceptionNotifiable if Rails.env.production?
  around_filter :catch_client_error

  secret_path = Pathname.new(RAILS_ROOT) + ".secret.csrf.txt"
  unless secret_path.exist?
    require 'rails_generator/secret_key_generator'
    secret_path.open("w", 0400) do |f|
      f.puts Rails::SecretKeyGenerator.new(App::PACKAGE).generate_secret
    end
  end
  secret = secret_path.read.chomp
  protect_from_forgery :secret => secret

  # See ActionController::Base for details
  # Uncomment this to filter the contents of submitted sensitive data parameters
  # from your application log (in this case, all? { |fields with names like "password").
  filter_parameter_logging :password

  # avoid error before install gettext gem
  if defined?(GetText) && defined?(after_init_gettext)
    # FIXME: after_init_gettextのチェック
    # ApplicationController の autoload まわりがあやしいのか、
    # なぜか after_init_gettext でエラーになるので、after_init_gettext の
    # チェックも追加した。
    if ENV["RAILS_ENV"] == "development"
      after_init_gettext do
        lang = GetText.locale.to_s[0,2]
        if test(?>,
            File.join(RAILS_ROOT, "po", lang, "#{App::PACKAGE}.po"),
            File.join("locale", lang, "LC_MESSAGES", "#{App::PACKAGE}.mo"))
          require "gettext/utils"
          GetText.create_mofiles(true, "po", "locale")
        end
      end
    end

    before_init_gettext do
      lang = GetText.locale.to_s[0,2]
      po_translation_name = "po_translation_#{lang}"
      class_name = po_translation_name.classify
      po_translation_class = nil
      begin
        po_translation_class = class_name.constantize
      rescue NameError => e
        logger.info "#{po_translation_name} not found: #{e.inspect}"
      end
      if po_translation_class
        po_translation_class.database_only_update_po_and_make_mo
      end
    end

    init_gettext ::App::GETTEXT_DOMAINNAME
    init_gettext "database"
  end

  include UserSystem
  include FragmentHash::ControllerHelper
  include FragmentHash::UrlHelper
  include DisplayNarrowingHelper::Controller
  include Rfw::UserAgent

  helper :user
  before_filter :authenticate_user
  before_filter :convert_fragment

  if ENV['RAILS_ENV'] == "development"
    after_filter do |c|
      case c.response.content_type
      when "text/html"
        ids = {}
        errors = []
        c.response.body.scan(/id=(['"])(.+?)\1/) do |quote,id|
          if ids.key?(id)
            errors << "<div class=\"formError\">duplicated id=#{id.dump}</div>"
          end
          ids[id] = true
        end
        unless errors.empty?
          c.response.body.sub!(/<div/) do
            errors.join("\n") + $&
          end
        end
      end
    end
  end

  private

  # 1リクエストの中でのキャッシュを初期化する。
  def clear_current_cache
    CacheEachRequest.clear
  end
  prepend_before_filter :clear_current_cache

  SESSION_RESERVE_KEYS = [:return_to, :fragment, :uuid]

  # セッションを初期化する。ただしフレームワークで予約しているキー <tt>SESSION_RESERVE_KEYS</tt> のデータについては引き継ぐ。
  def reset_session() #:doc:
    data = Hash.new
    SESSION_RESERVE_KEYS.each do |key|
      data[key] = session[key] if session[key]
    end
    super
#     session[:rails_protection_session_id] = ::UserSystem::RailsProtectionDigestClass.hexdigest("#{Time.now.to_i}-#{session.session_id}")
    data.each do |key, value|
      session[key] = value
    end
    @session_reset_done = true
  end

  # セッションを初期化する。<tt>reset_session</tt> と異なり、フレームワークで予約しているキー <tt>SESSION_RESERVE_KEYS</tt> のデータも消す。
  def reset_session_all() #:doc:
    session[:return_to] = nil
    session[:fragment] = nil
    unless @session_reset_done
      reset_session
    end
  end

  # Web クライアントに関するエラー。
  class ClientError < Exception; end
  # クライアントに対して利用が許されていない場合のエラー。
  class ForbiddenException < ClientError; end
  # 該当するデータが存在しない場合のエラー。
  class NotFoundException < ClientError; end
  # クライアントに対して権限によって許可されていない場合のエラー。
  class PermissionDenied < ForbiddenException; end
  # クライアントに対して無効にされている場合のエラー。
  class DisabledException < ForbiddenException; end
  # Web サーバに関するエラー。
  class ServerError < Exception; end
  # 依存するサーバが動いていないなどのサービスが提供できないエラー。
  class ServiceUnavailable < ServerError; end

  def catch_client_error
    yield
  rescue ForbiddenException => e
    logger.error("ERROR:ClientError:ForbiddenException: #{e.inspect}")
    @error_type = s_("rfw|ClientError|Forbidden")
    @error = e
    render :template => "client_error", :status => 403
  rescue NotFoundException => e
    logger.error("ERROR:ClientError:NotFoundException: #{e.inspect}")
    @error_type = "Not Found"
    @error = e
    render :template => "client_error", :status => 404
  rescue ServiceUnavailable => e
    logger.error("ERROR:ServerError:ServiceUnavailable: #{e.inspect}")
    @error_type = "Service Unavailable"
    @error = e
    # FIXME: テンプレートの名前をどうにかする。
    render :template => "client_error", :status => 503
  end

  def admin_only
    unless User.admin?
      raise PermissionDenied, "admin only"
    end
  end

  # ajax update view or redirect
  def x_redirect_to(view, ajax_link, location, options={})
    if request.xhr?
      view = view.sub(/\Aview_/, "")
      render :update do |page|
        if ajax_link == :close
          page.call("closeView", view, url_for(location))
        elsif ajax_link
          page.call("onAjaxLink", "link_#{view}_#{ajax_link}")
        end
      end
    else
      redirect_to location
    end
  end

  def x_close_or_redirect_to(location, options={})
    x_redirect_to(@current_view, :close, location, options)
  end
end
