# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: users
#
#  id                    :integer       not null, primary key
#  domain_id             :integer       not null
#  person_id             :integer       not null
#  login                 :string(80)    not null
#  salted_password       :string(128)   not null
#  admin                 :boolean       not null
#  list_default_per_page :integer       default(10), not null
#  list_header_per_line  :integer       default(10), not null
#  menu_history_max      :integer       default(10), not null
#  salt                  :string(128)   not null
#  verified              :boolean       not null
#  security_token        :string(128)
#  token_expiry          :datetime
#  deleted               :boolean       not null
#  failed_login_count    :integer       default(0), not null
#  password_updated_on   :string(8)
#  created_at            :string(14)
#  updated_at            :string(14)
#  created_by            :integer
#  updated_by            :integer
#  created_in            :integer
#  updated_in            :integer
#  lock_version          :integer       default(0), not null
#

require "digest/sha2"

# ユーザーのモデル
# this model expects a certain database layout and its based on the name/login pattern.
class User < ActiveRecord::Base
  PasswordDigestClass = Digest::SHA512

  untranslate_all
  timestamps_as_string
  user_monitor

  CHANGEABLE_FIELDS = []
  attr_accessor :password_needs_confirmation

  belongs_to :domain

  delegate :name, :mail_address, :to => :person
  alias email mail_address
  delegate :config_password, :to => :domain

  after_save '@password_needs_confirmation = false'
  after_validation :crypt_password
  after_destroy 'Permission.find_with_period(:all, :conditions => {:user_id => "#{id}"}).each {|x| x.destroy}'

  validates_presence_of :login #, :on => :create
  validates_length_of :login, :within => 3..40 #, :on => :create
  validates_uniqueness_of :login #, :on => :create

  validates_presence_of :password, :if => :validate_password?
  validates_confirmation_of :password, :if => :validate_password?
  validates_length_of :password, { :minimum => 5, :if => :validate_password? }
  validates_length_of :password, { :maximum => 40, :if => :validate_password? }

  validates_associated :person

  belongs_to :person
  has_many :permissions, :dependent => :destroy
  has_one :portal
  has_many :default_lists, :dependent => :destroy
  has_one :default_matter, :dependent => :destroy

  after_create :initialize_permissions

  def initialize(attributes = nil)
    super
    @password_needs_confirmation = false
  end

  # 文字列として <tt>login</tt> を返す。
  def to_s
    login
  end

  # <em>login</em> と <em>pass</em> による認証を行う。
  # 成功した場合は <tt>nil</tt> でない値、失敗した場合は <tt>nil</tt> を返す。
  def self.authenticate(login, pass, update_failed_login_count=false)
    User.transaction do
      u = find( :first, :conditions => ["login = ? AND verified = ? AND deleted = ?", login, true, false])
      return nil if u.nil?
      return nil if u.lockout?
      ret = find( :first, :conditions => ["login = ? AND salted_password = ? AND verified = ?", login, salted_password(u.salt, hashed(pass)), true])
      if update_failed_login_count
        if ret
          if u.failed_login_count != 0
            u.failed_login_count = 0
            u.save!
          end
        else
          u.failed_login_count += 1
          u.save!
        end
      end
      return ret
    end
  end

  # <em>id</em> と <em>token</em> による認証を行う。
  # 成功した場合は <tt>nil</tt> でない値、失敗した場合は <tt>nil</tt> を返す。
  def self.authenticate_by_token(id, token)
    # Allow logins for deleted accounts, but only via this method (and
    # not the regular authenticate call)
    logger.info "Attempting authorization of #{id} with #{token}"
    u = find( :first, :conditions => ["id = ? AND security_token = ?", id, token])
    if u
      logger.info "Authenticated by token: #{u.inspect}"
    else
      logger.info "Not authenticated" if u.nil?
    end
    return nil if (u.nil? or u.token_expired?)
    u.update_attributes :verified => true, :token_expiry => Clock.now
    return u
  end

  # token が期限切れかどうかを判定する。
  def token_expired?
    self.security_token and self.token_expiry and (Clock.now >= self.token_expiry)
  end

  # token を生成する。
  def generate_security_token
    if self.security_token.nil? or self.token_expiry.nil? or (Clock.now.to_i + token_lifetime / 2) >= self.token_expiry.to_i
      token = new_security_token
      return token
    else
      return self.security_token
    end
  end

  # パスワードを <em>pass</em> に変更する。
  def change_password(pass, confirm = nil)
    self.password = pass
    self.password_confirmation = confirm.nil? ? pass : confirm
    @password_needs_confirmation = true
  end

  # token の生存時間を秒単位で返す。
  def token_lifetime
    UserSystem::CONFIG[:security_token_life_hours] * 60 * 60
  end

  # パスワードが更新された時刻を返す。
  def password_updated_on
    str = read_attribute("password_updated_on")
    if /\A(\d{4})(\d\d)(\d\d)\z/ =~ str
      return Date.new(*$~.captures.map(&:to_i))
    elsif str.nil?
      return nil
    else
      raise ArgumentError, "invalid password_updated_on #{str.inspect}"
    end
  end

  # パスワードの有効期限が切れていれば true、さもなくば false を返す。
  def password_expire?
    unless config_password
      raise ArgumentError, s_("rfw|User|invalid domain")
    end
    if config_password.enable_max_age?
      return Date.today >= password_updated_on + config_password.max_age
    end
    return false
  end

  # ログイン資格が失効していれば true、さもなくば false を返す。
  def lockout?
    return false unless config_password.enable_lockout?
    return failed_login_count >= config_password.lockout_threshold
  end

  # 現在のユーザーを返す。
  def self.current
    CacheEachRequest.current[:user]
  end

  # 現在のユーザーを代入する。
  def self.current=(u)
    if CacheEachRequest.current.key?(:user)
      raise "must not change user in a request"
    end
    CacheEachRequest.current[:user] = u
  end

  # 現在のユーザーIDを返す。
  def self.current_id
    if current
      return current.id
    else
      return nil
    end
  end

  # 現在のユーザーがシステム管理者かどうかを判定する。
  def self.admin?
    if current
      return current.admin?
    else
      return false
    end
  end

  # 現在のページごとの個数の既定値を返す。
  def self.list_default_per_page
    return current.list_default_per_page
  rescue
    return columns_hash["list_default_per_page"].default
  end

  # 現在の行ごとの個数の既定値を返す。
  def self.list_header_per_line
    return current.list_header_per_line
  rescue
    return columns_hash["list_header_per_line"].default
  end

  # 現在のメニュー履歴の最大値を返す。
  def self.menu_history_max
    return current.menu_history_max
  rescue
    return columns_hash["menu_history_max"].default
  end

  # ユーザの言語のコード。
  def lang
    code = GetText.locale.to_s[0,2]
    code ||= person.last_language
    code ||= Language.default_code
    return code
  end

  # ユーザーの言語の翻訳のクラス。
  def translation_class
    "PoTranslation#{lang.camelize}".constantize
  end

  protected

  attr_accessor :password, :password_confirmation

  # パスワードを検証する必要があるかどうかを返す。
  def validate_password?
    @password_needs_confirmation
  end

  # <em>str</em> をハッシュ化する。
  def self.hashed(str)
    return PasswordDigestClass.hexdigest("#{UserSystem::CONFIG[:password_hash_prefix]}--#{str}--")[0,128]
  end

  # パスワードを暗号化する。
  def crypt_password
    if @password_needs_confirmation
      write_attribute("salt", self.class.hashed("salt-#{Clock.now}.#{Clock.now.usec}"))
      write_attribute("salted_password", self.class.salted_password(salt, self.class.hashed(@password)))
      write_attribute("password_updated_on", Time.now.utc.strftime("%Y%m%d"))
    end
  end

  # 新しい token を返す。
  def new_security_token
    expiry = Time.at(Clock.now.to_i + token_lifetime)
    write_attribute('security_token', self.class.hashed(self.salted_password + Clock.now.to_i.to_s + rand.to_s))
    write_attribute('token_expiry', expiry)
    update_without_callbacks
    return self.security_token
  end

  # <em>salt</em> と <em>hashed_password</em> をハッシュ化する。
  def self.salted_password(salt, hashed_password)
    hashed(salt + hashed_password)
  end

  # <em>email</em> からユーザーを探す。
  def self.find_by_email(email)
    person = Person.find(:first, :conditions => {:mail_address => email}, :include => [:user])
    if person
      return person.user
    else
      return nil
    end
  end

  private

  def initialize_permissions
    return unless person
    person.grant_ons.each do |grant_on|
      Permission.create_with_grant_on_and_user_id_and_priority(grant_on, id, Membership::PRIORITY_PERSON)
    end
    {
     :group        => Membership::PRIORITY_GROUP,
     :organization => Membership::PRIORITY_ORGANIZATION,
     :company      => Membership::PRIORITY_COMPANY,
    }.each do |key, priority|
      c_member_class = "#{key.to_s.classify}Member".constantize
      c_member_class.find_with_period(:all, :conditions => {:person_id => person_id}).each do |member|
        member.__send__(key).grant_ons.each do |grant_on|
          Permission.create_with_grant_on_and_user_id_and_priority(grant_on, id, priority, member.inception, member.expiry)
        end
      end
    end
    person.domain.grant_ons.each do |grant_on|
      Permission.create_with_grant_on_and_user_id_and_priority(grant_on, id, Membership::PRIORITY_DOMAIN)
    end
  end
end
