#!/usr/bin/env ruby1.9

# - take a path of .rb as the argument,
# - accumulate comments for the annotation of form leading codes or method definitions, and
# - puts in the RD style.
# 
# Usage:
# $ ruby1.9 accumulate_comments.rb [-lmn] <path> > comment.log

require 'optparse'
require 'ripper'

def may_i_help_you?(tokens)
  tokens.first !~ /^##*\W*LEAVEMEALONE/
end

COMMENT_REGEXP = /^#([^{].+)/

def leading_comments(tokens)
  result = []
  tokens.each do |x|
    case x
    when COMMENT_REGEXP
      result << "  #{$1.lstrip}"
    else
      return result
    end
  end
  raise "unexpected to be reached"
end

def method_comments(tokens)
  result = []
  mark = false
  tokens.inject([]) do |seed, x|
    # match a comment
    case x
    when COMMENT_REGEXP
      seed | ["    #{$1.lstrip}"]
    when /^\s+$/
      seed
    when /^def$/
      mark = !seed.empty?
      seed
    when /^\w+$/
      if mark && !seed.empty?
        # push a stuff of comments
        seed.unshift "* #{x}"
        result << seed
        mark = false
      end
      []
    else
      []
    end
  end
  result
end

mode = {}

opt = OptionParser.new
opt.on('-l', '--leading') {|v| mode[:l] = v}
opt.on('-m', '--method') {|v| mode[:m] = v}
opt.on('-n', '--name') {|v| mode[:n] = v}

argv = opt.parse(ARGV)
path = argv[0]
tokens = Ripper.tokenize(File.read(path))
exit unless may_i_help_you?(tokens)
puts "=== #{File.basename(path)}" if mode[:n]
if mode[:l]
  leading_comments(tokens).each do |r|
    puts r
  end
  puts "\n" if mode[:m]
end
if mode[:m]
  method_comments(tokens).each do |r|
    r.each {|x| puts x}
    puts "\n"
  end
end
