# -*- coding: utf-8 -*-
# app/controllers/user_controller.rb のクラス UserController をテストする。

require File.dirname(__FILE__) + '/../test_helper'
require 'user_controller'
require 'user_notify'

# Raise errors beyond the default web-based presentation
class UserController; def rescue_action(e) raise e end; end

class UserControllerTest < Test::Unit::TestCase
  self.use_transactional_fixtures = false
  fixtures :users, :people, :config_passwords, :mail_queues, :menus, :products, :default_narrowings, :display_narrowings, :displays, :narrowings

  def setup
    @controller = UserController.new
    @request    = ActionController::TestRequest.new
    @response   = ActionController::TestResponse.new
    @request.host = "localhost"
    ActionMailer::Base.inject_one_error = false
    ActionMailer::Base.deliveries = []

    UserSystem::CONFIG[:default_url] = {:controller => :bogus, :action => :location}
    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  def test_login__valid_login__redirects_as_specified
    @request.session[:return_to] = "/bogus/location"
    post :login, :user => { :login => "tesla", :password => "atest" }
    assert_logged_in users(:tesla)
    assert_response :redirect
    assert_equal "http://#{@request.host}/bogus/location", @response.redirect_url
  end

  def test_login__valid_login__shows_welcome_as_default
    post :login, :user => { :login => "tesla", :password => "atest" }
    assert_logged_in users(:tesla)
    assert_redirected_to_welcome
  end

  def test_login__wrong_password
    post :login, :user => { :login => "tesla", :password => "wrong password" }
    assert_response :success
    assert_select "div"
    assert_not_logged_in
    assert_template 'login'
    assert_contains "Login failed", flash['message']
  end

  def test_login__wrong_login
    post :login, :user => { :login => "wrong login", :password => "atest" }
    assert_response :success
    assert_select "div"
    assert_not_logged_in
    assert_template 'login'
    assert_contains "Login failed", flash['message']
  end

  def test_login__deleted_user_cant_login
    post :login, :user => { :login => "deleted_tesla", :password => "atest" }
    assert_response :success
    assert_select "div"
    assert_not_logged_in
    assert_template 'login'
    assert_contains "Login failed", flash['message']
    assert_select "div"
  end

=begin
  def test_signup
    post_signup :login => "newuser",
                :password => "password", :password_confirmation => "password",
                :email => "newemail@example.com"
    assert_not_logged_in
    assert_redirected_to_login
    assert_equal 1, ActionMailer::Base.deliveries.size

    mail = ActionMailer::Base.deliveries[0]
    assert_equal "newemail@example.com", mail.to_addrs[0].to_s
    assert_match /login:\s+\w+\n/, mail.encoded
    assert_match /password:\s+\w+\n/, mail.encoded
    user = User.find_by_email("newemail@example.com")
    assert_match /user\[id\]=#{user.id}/, mail.encoded
    assert_match /key=#{user.security_token}/, mail.encoded
    assert !user.verified
  end


  def test_signup__cannot_set_arbitrary_attributes
    post_signup :login => "newuser",
                :password => "password", :password_confirmation => "password",
                :email => "skunk@example.com",
                :verified => '1',
                :role => 'superadmin'
    user = User.find_by_email("skunk@example.com")
    assert !user.verified
    assert_nil user.role
  end

  def test_signup__validates_password_min_length
    post_signup :login => "tesla_rhea", :password => "bad", :password_confirmation => "bad", :email => "someone@example.com"
    assert_password_validation_fails
  end

  def test_signup__raises_delivery_errors
    ActionMailer::Base.inject_one_error = true
    post_signup :login => "newtesla",
                :password => "newpassword", :password_confirmation => "newpassword",
                :email => "newtesla@example.com"
    assert_not_logged_in
    assert_equal 0, ActionMailer::Base.deliveries.size
    assert_contains "confirmation email not sent", flash['message']
  end

  def test_signup__mismatched_passwords
    post :signup, :user => { :login => "newtesla", :password => "newpassword", :password_confirmation => "wrong" }
    user = assigns(:user)
    assert_equal 1, user.errors.size
    assert_not_nil user.errors['password']
  end

  def test_signup__bad_login
    post_signup :login => "yo", :password => "newpassword", :password_confirmation => "newpassword"
    user = assigns(:user)
    assert_equal 1, user.errors.size
    assert_not_nil user.errors['login']
  end
=end

  def test_welcome
    user = users(:unverified_user)
    get :welcome, :user=> { :id => user.id }, :key => user.security_token
    assert_response :success
    assert_template "welcome"
    assert_select "div"
    user.reload
    assert user.verified
    assert_logged_in( user )
    assert user.token_expired?
  end

  def test_welcome__fails_if_expired_token
    user = users(:unverified_user)
    Clock.advance_by_days 2 # now past verification deadline
    get :welcome, :user=> { :id => user.id }, :key => user.security_token
    assert_redirected_to_login
    user.reload
    assert !user.verified
    assert_not_logged_in
  end

  def test_welcome__fails_if_bad_token
    user = users(:unverified_user)
    Clock.time = Time.now # now before deadline, but with bad token
    get :welcome, :user=> { :id => user.id }, :key => "boguskey"
    assert_redirected_to_login
    user.reload
    assert !user.verified
    assert_not_logged_in
  end

=begin
  def test_edit
    tesla = users(:tesla)
    set_logged_in tesla
    post :edit, :user => { :first_name => "Bob", :form => "edit" }
    tesla.reload
    assert_equal tesla.first_name, "Bob"
  end
=end

  def test_delete
    user = users(:deletable_user)
    set_logged_in user
    post :edit, "user" => { "form" => "delete" }
    assert_redirected_to_login
    user.reload
    assert user.deleted
    assert_not_logged_in
  end

  def test_change_password
    user = users(:tesla)
    set_logged_in user
    post :change_password, :user => { :password => "changed_password", :password_confirmation => "changed_password" }
    assert_response :redirect
    assert_redirected_to :controller => "menu", :action => "index"
    assert_equal 1, ActionMailer::Base.deliveries.size
    mail = ActionMailer::Base.deliveries[0]
    assert_equal user.mail_address, mail.to_addrs[0].to_s
    assert_match /login:\s+\w+\n/, mail.encoded
    assert_match /password:\s+\w+\n/, mail.encoded
    assert_equal user, User.authenticate(user.login, 'changed_password')
  end

  def test_change_password__confirms_password
    set_logged_in users(:tesla)
    post :change_password, :user => { :password => "bad", :password_confirmation => "bad" }
    assert_response :success
    assert_select "div"
    user = assigns(:user)
    assert_equal 1, user.errors.size
    assert_not_nil user.errors['password']
    assert_equal 0, ActionMailer::Base.deliveries.size
  end

  def test_redirect_to_change_password
    user = users(:ignore_max_password_age)
    enable_max_age = user.config_password.enable_max_age
    user.config_password.update_attributes!(:enable_max_age => true)
    post :login, :user => { :login => user.login, :password => "atest" }
    assert_logged_in user
    assert_response :redirect
    assert_equal "http://#{@request.host}/user/change_password", @response.redirect_url

    get :welcome
    assert_logged_in user
    assert_response :redirect
    assert_equal "http://#{@request.host}/user/change_password", @response.redirect_url
  ensure
    user.config_password.update_attributes!(:enable_max_age => enable_max_age)
  end

  def test_change_password__succeeds_despite_delivery_errors
    set_logged_in users(:tesla)
    ActionMailer::Base.inject_one_error = true
    post :change_password, :user => { :password => "changed_password", :password_confirmation => "changed_password" }
    assert_response :redirect
    assert_redirected_to :controller => "menu", :action => "index"
    assert_equal 0, ActionMailer::Base.deliveries.size
    assert_equal users(:tesla), User.authenticate(users(:tesla).login, 'changed_password')
  end

  def test_forgot_password__when_logged_in_redirects_to_change_password
    user = users(:tesla)
    set_logged_in user
    post :forgot_password, :user => { :login => user.login }
    assert_redirected_to_change_password
    assert_equal 0, ActionMailer::Base.deliveries.size
  end

  def test_forgot_password__requires_valid_login
    post :forgot_password, :user => { :login => "" }
    assert_response :success
    assert_template "forgot_password"
    assert_select "div"
    assert_match /Please enter a valid login name./, @response.body
    assert_equal 0, ActionMailer::Base.deliveries.size
  end

  def test_forgot_password__ignores_unknown_login
    post :forgot_password, :user => { :login => "unknown_login" }
    assert_response :success
    assert_template "forgot_password"
    assert_select "div"
    assert_equal 0, ActionMailer::Base.deliveries.size
  end

  def test_forgot_password__reports_delivery_error
    ActionMailer::Base.inject_one_error = true
    post :forgot_password, :user => { :login => users(:tesla).login }
    assert_response :success
    assert_template "forgot_password"
    assert_select "div"
    assert_match /Your password could not be emailed/, @response.body
    assert_equal 0, ActionMailer::Base.deliveries.size
  end

  def test_invalid_login
    post :login, :user => { :login => "tesla", :password => "not_correct" }
    assert_response :success
    assert_template "login"
    assert_select "div"
    assert_not_logged_in
  end

  def test_logout
    set_logged_in users(:tesla)
    get :logout
    assert_not_logged_in
    assert_redirected_to_login
  end


  def test_login__max_password_age_ignored
    post :login, :user => { :login => "ignore_max_password_age", :password => "atest" }
    assert_logged_in users(:ignore_max_password_age)
    assert_redirected_to_welcome
  end

  def test_login__max_password_age_6
    post :login, :user => { :login => "six_days", :password => "atest" }
    assert_logged_in users(:six_days)
    assert_redirected_to_welcome
  end

  def test_login__max_password_age_7
    post :login, :user => { :login => "one_week", :password => "atest" }
    assert_logged_in users(:one_week)
    assert_redirected_to_change_password
  end

  def test_login__max_password_age_8
    post :login, :user => { :login => "eight_days", :password => "atest" }
    assert_logged_in users(:eight_days)
    assert_redirected_to_change_password
  end

  def test_login__failed_login_count
    user = users(:tesla)
    assert_equal 0, user.failed_login_count
    (1..5).each do |n|
      post :login, :user => { :login => "tesla", :password => "wrong password" }
      assert_response :success
      assert_template "login"
      assert_select "div"
      user.reload
      assert_not_logged_in
      assert_equal n, user.failed_login_count
    end
  end


  def test_login_history__result_login_succeeded
    login = "tesla"
    password = "atest"
    user = User.find_by_login(login)
    post :login, :user => { :login => login, :password => password }
    assert_redirected_to_welcome
    assert_logged_in user
    assert_latest_login_history(login, password, LoginHistory::RESULT_LOGIN_SUCCEEDED, user)
  end

  def test_login_history__result_login_failed
    login = "tesla"
    password = "wrong password"
    user = User.find_by_login(login)
    post :login, :user => { :login => login, :password => password }
    assert_response :success
    assert_template "login"
    assert_select "div"
    assert_not_logged_in
    assert_latest_login_history(login, password, LoginHistory::RESULT_LOGIN_FAILED, nil)
  end

  def test_login_history__result_password_expired
    login = "eight_days"
    password = "atest"
    user = User.find_by_login(login)
    post :login, :user => { :login => login, :password => password }
    assert_redirected_to_change_password
    assert_logged_in user
    assert_latest_login_history(login, password, LoginHistory::RESULT_PASSWORD_EXPIRED, user)
  end

  def test_login_history__result_logout
    login = "tesla"
    password = "atest"
    user = User.find_by_login(login)
    post :login, :user => { :login => login, :password => password }
    assert_redirected_to_welcome
    assert_logged_in user
    assert_latest_login_history(login, password, LoginHistory::RESULT_LOGIN_SUCCEEDED, user)
    get :logout
    assert_redirected_to_login
    assert_not_logged_in
    assert_latest_login_history(nil, nil, LoginHistory::RESULT_LOGOUT, user)
  end

  def test_login_history__result_session_timed_out
    login = "tesla"
    password = "atest"
    user = User.find_by_login(login)
    post :login, :user => { :login => login, :password => password }
    assert_redirected_to_welcome
    assert_logged_in user
    assert_latest_login_history(login, password, LoginHistory::RESULT_LOGIN_SUCCEEDED, user)
    session[:expires_at] = Time.now - 1
    get welcome_url
    assert_redirected_to_login
    assert_not_logged_in
    assert_latest_login_history(nil, nil, LoginHistory::RESULT_SESSION_TIMED_OUT, user)
  end

  def test_update_last_language
    login = "tesla"
    password = "atest"
    user = User.find_by_login(login)
    person = user.person
    assert_nil person.last_language

    post :login, :user => { :login => login, :password => password }
    assert_redirected_to_welcome
    assert_logged_in user
    assert_latest_login_history(login, password, LoginHistory::RESULT_LOGIN_SUCCEEDED, user)
    user.reload
    assert_logged_in( user )

    person.reload
    assert_equal "en", person.last_language
    person.last_language = "ja"
    assert person.save

    post :login, :user => { :login => login, :password => password }
    assert_redirected_to_welcome
    assert_logged_in user
    assert_latest_login_history(login, password, LoginHistory::RESULT_LOGIN_SUCCEEDED, user)
    user.reload
    assert_logged_in( user )

    person.reload
    assert_equal "en", person.last_language
  end

  def test_auto
    get :auto, :id => 1, :u => 1000003
    assert_response :success
    assert_template "login"
    assert_select "div"
  end

  def test_auto__skip_authentication
    get :auto, :id => 2, :u => 2
    assert_response :redirect
  end

  def test_auto__mail_queue_not_found
    get :auto, :id => 2
    assert_response :missing
  end

  def test_auto__recipient_not_found
    get :auto, :id => 1, :u => "tesla"
    assert_response :missing
  end

  # product id と menu id が一致していない場合
  def test_auto__product_id_neq_menu_id
    m = mail_queues(:auto_login)
    get :auto, :id => m.id, :u => 2
    assert_response :success
    assert_template "login"
    assert_select "div"
    fragment = @response.session[:fragment].last
    assert_equal fragment, "m=#{m.menu_id},detail=show.id_#{m.document_id}"
  end

  def test_init_login_session__matter
    user = users(:admin)
    post :login, :user => {:login => "admin", :password => "atest"}
    assert_response :redirect
    assert_logged_in(user)
    assert_equal "1", @response.session[:matter_id]
  end

  def test_routing
    assert_routing("user/auto/581/15",
                   {
                     :controller => "user",
                     :action => "auto",
                     :id => "581",
                     :u => "15",
                   })
  end

  private

  def set_logged_in( user )
    @request.session[:user_id] = user.id
  end

  def assert_logged_in( user )
    assert_equal user.id, @request.session[:user_id]
    assert_equal user, assigns(:current_user)
  end

  def assert_not_logged_in
    assert_nil @request.session[:user_id]
    assert_nil assigns(:current_user)
  end

  def assert_redirected_to_login
    assert_response :redirect
    assert_redirected_to :action => "login"
  end

  def welcome_url
    @controller.url_for(UserSystem::CONFIG[:default_url])
  end

  def assert_redirected_to_welcome
    assert_response :redirect
    assert_equal welcome_url, @response.redirect_url
  end

  def assert_redirected_to_change_password
    assert_response :redirect
    assert_redirected_to :action => "change_password"
  end

  def post_signup( user_params )
    post :signup, "user" => user_params
  end

  def assert_password_validation_fails
    user = assigns(:user)
    assert_equal 1, user.errors.size
    assert_not_nil user.errors['password']
    assert_response :success
    assert_equal 0, ActionMailer::Base.deliveries.size
  end

  def assert_contains( target, container )
    assert !container.nil?, %Q( Failed to find "#{target}" in nil String )
    assert container.include?(target)
  end

  DEFAULT_REMOTE_ADDRESS = "0.0.0.0"
  def assert_latest_login_history(login, password, result, user)
    history = LoginHistory.find :first, :order => "id DESC"
    if login
      assert_equal login, history.login
    else
      assert_nil history.login
    end
    if password
      assert_equal password, history.password
    else
      assert_nil history.password
    end
    assert_equal result, history.result
    assert_equal DEFAULT_REMOTE_ADDRESS, history.remote_address
    assert_equal LoginHistory::PROGRAM_TYPE_WEB, history.program_type
    if user
      assert_equal user.id, history.user_id
      assert_equal user.domain_id, history.domain_id
    else
      assert_nil history.user_id
      assert_nil history.domain_id
    end
  end

end
