# -*- coding: utf-8 -*-
# app/models/search_condition_item.rb のクラス SearchConditionItem をテストする。

require 'test_helper'

class SearchConditionItemTest < ActiveSupport::TestCase
  fixtures :items, :search_condition_items

  # SearchConditionItem#extract で出力される SQL 文をテストする。
  def test_extract
    quoted_table_name = Domain.quoted_table_name
    i = SearchConditionItem.new do |i|
      i.item_id = 841
      i.condition_value = "abc  123 xyz "
    end
    i.condition_pattern = "none"
    assert !i.extract
    i.condition_pattern = "eq"
    v_abc = ItemProper.quote_value('abc')
    v_123 = ItemProper.quote_value('123')
    v_xyz = ItemProper.quote_value('xyz')
    assert_equal "(#{quoted_table_name}.name = #{v_abc} OR #{quoted_table_name}.name = #{v_123} OR #{quoted_table_name}.name = #{v_xyz})", i.extract
    i.condition_pattern = "neq"
    assert_equal "(#{quoted_table_name}.name <> #{v_abc} AND #{quoted_table_name}.name <> #{v_123} AND #{quoted_table_name}.name <> #{v_xyz})", i.extract
    i.condition_pattern = "head"
    v = ItemProper.quote_value('abc  123 xyz %')
    assert_equal "(#{quoted_table_name}.name LIKE #{v})", i.extract
    i.condition_pattern = "tail"
    v = ItemProper.quote_value('%abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.name LIKE #{v})", i.extract
    i.condition_pattern = "include"
    v = ItemProper.quote_value('%abc  123 xyz %')
    assert_equal "(#{quoted_table_name}.name LIKE #{v})", i.extract
    i.condition_pattern = "exclude"
    v = ItemProper.quote_value('%abc  123 xyz %')
    assert_equal "(#{quoted_table_name}.name IS NULL OR #{quoted_table_name}.name NOT LIKE #{v})", i.extract
    i.condition_pattern = "ge"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.name >= #{v})", i.extract
    i.condition_pattern = "le"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.name <= #{v})", i.extract
    i.condition_pattern = "gt"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.name > #{v})", i.extract
    i.condition_pattern = "lt"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.name < #{v})", i.extract
    i.condition_pattern = "null"
    assert_equal "(#{quoted_table_name}.name IS NULL OR #{quoted_table_name}.name = '')", i.extract
    i.condition_pattern = "not-null"
    assert_equal "(#{quoted_table_name}.name IS NOT NULL AND #{quoted_table_name}.name <> '')", i.extract
  end

  def test_extract_with_method_chain
    q_table_name = MatterReport.quoted_table_name
    d_table_name = Domain.quoted_table_name
    i = SearchConditionItem.new do |i|
      i.item_id = 99968
      i.condition_value = "abc  123 xyz "
    end
    i.item.update_attributes!(:method_chain => "Company#domain_id.Domain#code")
    i.condition_pattern = "none"
    assert !i.extract
    i.condition_pattern = "eq"
    c_table_name = Company.quoted_table_name
    v_abc = ItemProper.quote_value('abc')
    v_123 = ItemProper.quote_value('123')
    v_xyz = ItemProper.quote_value('xyz')
    left = "(#{q_table_name}.sales_person_id IN (SELECT id FROM #{c_table_name} WHERE #{c_table_name}.domain_id IN (SELECT id FROM #{d_table_name} WHERE "
    right = ")))"
    assert_equal "#{left}#{d_table_name}.code IN (#{v_abc},#{v_123},#{v_xyz})#{right}", i.extract
    i.condition_pattern = "neq"
    assert_equal "#{left}#{d_table_name}.code NOT IN (#{v_abc},#{v_123},#{v_xyz})#{right}", i.extract
    i.condition_pattern = "head"
    v = ItemProper.quote_value('abc  123 xyz %')
    assert_equal "#{left}(#{d_table_name}.code LIKE #{v})#{right}", i.extract
    i.condition_pattern = "tail"
    v = ItemProper.quote_value('%abc  123 xyz ')
    assert_equal "#{left}(#{d_table_name}.code LIKE #{v})#{right}", i.extract
    i.condition_pattern = "include"
    v = ItemProper.quote_value('%abc  123 xyz %')
    assert_equal "#{left}(#{d_table_name}.code LIKE #{v})#{right}", i.extract
    i.condition_pattern = "exclude"
    v = ItemProper.quote_value('%abc  123 xyz %')
    assert_equal "#{left}(#{d_table_name}.code IS NULL OR #{d_table_name}.code NOT LIKE #{v})#{right}", i.extract
    i.condition_pattern = "ge"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "#{left}(#{d_table_name}.code >= #{v})#{right}", i.extract
    i.condition_pattern = "le"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "#{left}(#{d_table_name}.code <= #{v})#{right}", i.extract
    i.condition_pattern = "gt"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "#{left}(#{d_table_name}.code > #{v})#{right}", i.extract
    i.condition_pattern = "lt"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "#{left}(#{d_table_name}.code < #{v})#{right}", i.extract
    i.condition_pattern = "null"
    assert_equal "#{left}(#{d_table_name}.code IS NULL OR #{d_table_name}.code = '')#{right}", i.extract
    i.condition_pattern = "not-null"
    assert_equal "#{left}(#{d_table_name}.code IS NOT NULL AND #{d_table_name}.code <> '')#{right}", i.extract
  end

  # proper かどうかを判定する。
  def test_proper?
    [:list8_1_name, :list8_1_code].each do |name|
      i = search_condition_items(name)
      assert i.proper?
      i.condition_value = ""
      assert !i.proper?
      i.condition_pattern = "null"
      assert i.proper?
      i.condition_pattern = "not-null"
      assert i.proper?
    end
  end
end
