/**************************************************
opengate server main

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
Modified Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include	"opengatesrv.h"

extern char ruleNumber4[WORDMAXLN];  /* ipfw rule number in string form  */
extern char ruleNumber6[WORDMAXLN];  /* ip6fw rule number in string form */
char language[WORDMAXLN];

char clientAddr4[ADDRMAXLN]="";  /* client addr (nnn.nnn.nnn.nnn) */
char clientAddr6[ADDRMAXLN]="";  /* client addr (nnnn:nnnn:xxxx::xxxx) 128bit */
extern struct clientAddr *pClientAddr;

char macAddr4[ADDRMAXLN]="?";    /* client MAC address (format for arp) */
char macAddr6[ADDRMAXLN]="?";    /* client MAC address (format for ndp) */

char userid[USERMAXLN];
char useridshort[USERMAXLN];/* userID before @ mark(cut off extraID) */
char extraId[USERMAXLN];
char userProperty[BUFFMAXLN];
time_t timeIn, timeOut;
int ipStatus;              /* flag for IPV4ONLY,IPV6ONLY or IPV46DUAL */
int connectionMode;   /* client connect mode */
char *mode[4]={"NONE","HTTP","TIME", "NONE"};

/* variable to measuring processing time (in msec); for debug and evaluation */
struct timeval timeBeginCgi, timeBeginWait, timeConnect, timeDisconnect;

char sessionId[SIDMAXLN]="";    /* session ID */
char cookie[SIDMAXLN]="";    /* cookie */
char redirectedUrl[BUFFMAXLN]="";  /* redirected URL (requested URL before fwd) */

void PutCloseMsg(time_t timeOut, time_t timeIn);
void SetProcessTitle(char *useridshort, char *clientAddr4, char * ruleNumber4, char *clientAddr6, char * ruleNumber6, int ipStatus);
void logConnectMode();
void closeExit(int signo);

/***************************************************/
/*  main routine called as cgi from Web server     */
/***************************************************/
int  main(int argc, char **argv)
{
  char password[PASSMAXLN]="";
  int port;
  int dummyfd[2];
  int pid;
  int parentpid;
  int duration=0;      /* requested usage duration */
  int durationEntered=0; /* the duration value is entered or not */
  int authResult=DENY;
  int authNum=1; /* present authserver number to check user */
  int cookieAuth=FALSE; /* Auth with HTTP-Cookie is passed */
  int isUidInEnv=FALSE; /* userid is included in environment (shibb/basic) */
  char closeTime[WORDMAXLN]; /* session closing time ('-'=not close) */
  char* proto=""; /* authentication protocol */

  /* drop root privilege */
  /* the root privilege is needed only at the timing of ipfw control */
  seteuid(getuid());

  /* if this process is executed in shell with '-v' option, show the directory excuting [make] */  
  if(argc>1 && strcmp(argv[1],"-v")==0){
    printf("makedir: %s\n", MAKEDIR);
    exit(0);
  }

  /* get time of the starting cgi(in msec) */
  gettimeofday(&timeBeginCgi, NULL) ;

  /* prepare config file */
  if(OpenConfFile()==-1) return 0;
 
  /* start syslog */
  errToSyslog(atoi(GetConfValue("Syslog/Enable")));
  openlog(GetConfValue("MainCgi"), LOG_PID, atoi(GetConfValue("Syslog/Facility")));

  /* initialize config */
  InitConf();

  /* get default language from the top of HtmlLangs list in conf file */
  sscanf(GetConfValue("HtmlLangs"), "%s", language);

  /* check referer (this function is Disabled)
  if(CheckReferer()==FALSE){
    PutClientRetry(language);
    return 0;
  } */

  /* get POST data from HTTP request */
  if(GetPostData(userid, password, clientAddr4, 
		 &duration, &durationEntered, language, redirectedUrl)==FALSE){
    PutClientRetry(language);
    return 0;
  }

  /* get userid from environment var(available only for shbboleth or httpbasic) */
  isUidInEnv=GetUserIdFromEnv(userid);

  /* try to authentication with cookie data, if cookie auth is enabled. */
  /* get http-cookie from client and get corresponding data from Db */
  if( (*GetConfValue("EnableCookieAuth")!='0') ){
    cookieAuth=GetCookieData(userid, clientAddr4,
			   &duration,&durationEntered,language,closeTime);
  }

  /* if already opened, exit */
  if(cookieAuth && closeTime[0]=='-'){
    PutClientMsg("Already opened. Please close this page and retry.");
    return 0;
  }

  /* split user@server to user and server */
  SplitId(userid, useridshort, extraId);

  /* setup pointer to ExtraSet in config file */
  SetupConfExtra(useridshort, extraId);

  /* trim duration values between zero to max value in conf */
  /* (when user enters very long duration, the system limits the duration) */
  duration = TrimDuration(duration, durationEntered);

  /* setup static variable value for SqLite3_busy_timeout from conf */
  SetupSqliteBusyTimeoutValue();

  /* get address of client from getenv. it might be IPv4 or IPv6. */
  GetClientAddr(clientAddr6);

  /* two addresses are acquired, one from POST data and another from GETENV */
  /* as only IPv4 is forwarded by ipfw, first access (saved in POST) might be IPv4 */
  /* check enable IP versions (dual, v4only, v6only) */
  if((ipStatus=CheckIpVersions(clientAddr4, clientAddr6))==0){
      PutClientMsg("Error: Please contact the administrator(1)");
      return 0;
  }

  /* get MAC address from arp and ndp */
  GetMacAddr(clientAddr4, macAddr4, clientAddr6, macAddr6,ipStatus);

  /* pass auth by cookie */
  if(cookieAuth) authResult=ACCEPT;

  /* if exist [userid] in environment variable (set by shibboleth/httpbasic) */
  /*  and the protocol setting is shibboleth/httpbasic, then pass auth */
  ResetAuthServerPointer();
  proto=GetConfValue("AuthServer/Protocol");
  if( isUidInEnv && 
      (strcmp(proto,"shibboleth")==0 || strcmp(proto,"httpbasic")==0) 
    ) authResult=ACCEPT;

  /* if not pass auth, check by auth servers */
  if(authResult!=ACCEPT){
    ResetAuthServerPointer();
    while(SelectNextAuthServer()){

      /* authenticate the user with authentication servers */
      authResult=AuthenticateUser(useridshort, password);

      /* if success, exit auth server checking loop */
      if(authResult==ACCEPT) break;
	 
      /* else put out error message and select next server */
      err_msg("DENY:(auth%d) user %s from %s at %s", 
	      authNum, userid, clientAddr4, macAddr4);
      authNum++;
    }
  }

  /* after checking servers, if not authenticate, send deny to the client and exit */
  if(authResult!=ACCEPT){
    PutClientDeny(clientAddr4, language);
    return 0;
  }

  /* clear password */
  bzero(password, PASSMAXLN);

  /* check nat insertion. if found, put info to log */
  /* at now, not denied */
  CheckNatInsertion(macAddr4, macAddr6, userid);

  /* get user property from user database (if you edit comm-userdb.c) */
  if(!GetUserProperty(userid, userProperty)){
    PutClientMsg("Error: You are denied.");
    err_msg("DENY: user %s from %s at %s (ill-property)",
	    userid, clientAddr4, macAddr4);
    return 0;
  }

  /* create sessionID(for hello check) and HTTP-Cookie(for auth) */
  CreateSessionId(sessionId);
  CreateCookie(cookie);

  /* set terminate signal handler */
  if(signalx(SIGTERM, closeExit)==SIG_ERR){
    PutClientMsg("Error: Please contact the administrator(2)");    
    return 0;
  }

  /* open firewall for the client */
  if(OpenClientGate(clientAddr4,macAddr4,clientAddr6,macAddr6,
		    userid,userProperty,ipStatus)==FALSE){
    PutClientMsg("Error: Close this browser and retry");    
    return 0;
  }
  timeIn=time(NULL);

  /* set (ruleNumber,userid,clientAddr) for process listing command */
  SetProcessTitle(userid,clientAddr4,ruleNumber4,clientAddr6,ruleNumber6,ipStatus);
  
  /* get temporary port for server-listen */
  /* each process uses independent server port */
  port=GetListenPort();
  if(port<0){
    err_msg("ERR at %s#%d: cannot get unused listen port",__FILE__,__LINE__);
    PutClientMsg("Error: Please contact the administrator(3)");
    closeExit(1);
  }

  /* get the process id */
  /* this is the parent process to create the child process for watching client */
  parentpid=getpid();

  /* fork (create child process) */
  if((pid=Fork())==-1){
    err_msg("ERR at %s#%d: fork error",__FILE__,__LINE__);
    PutClientMsg("Error: Please contact the administrator(4)");
    closeExit(1);
  }

  if(pid!=0){
    /***** parent process *****/
    /* send accept page with javascript */
    PutClientAccept(userid, sessionId, port, pid, clientAddr4, clientAddr6, 
    		    ipStatus, duration, durationEntered, language,
		    cookie, cookieAuth, redirectedUrl);

    /* detach from Web server */
    return 0;
  }

  /***** child process *****/

  /* detach from Web server */
  /* (stop the read/write waiting of web server) */
  /* detach stdin and stdout pipe connecting to Web server */
  /* detach stderr */
  Close(0);Close(1);Close(2);

  Pipe(dummyfd);      /* connect dummy pipe for stdin and out */

  /* write the opening information to database */
  /* this should be in child process as to save watch-process PID */
  PutSessionBeginToDb(cookie, userid, clientAddr4, clientAddr6, 
			macAddr4, ruleNumber4, ruleNumber6,
		      duration, durationEntered, cookieAuth, language);

  /* get time at the starting of client wait (in msec) */
  gettimeofday(&timeBeginWait, NULL) ;

  /* wait connection request from the client */
  /* if no connection, close gate after some duration is passed */
  /* or ipaddr for the macAddr4 is changed */
  connectionMode=WaitClientConnect(userid, userProperty, sessionId, 
				   clientAddr4, clientAddr6, duration, 
				   macAddr4, macAddr6, ipStatus, 
				   pClientAddr, language, port, parentpid, 
				   cookieAuth,redirectedUrl);

  /* get time at the client connection (in msec) */
  gettimeofday(&timeConnect, NULL) ;

  /* if connected with Ajax, watching the client termination */
  /* if not connect, above WaitClientConnect function exits after delay, and do nothing */
  if(connectionMode==HTTPCONNECT){

    /* wait until the http keep alive is closed */
    WaitHttpClose(pClientAddr, userid, userProperty, macAddr4, macAddr6, 
		  ipStatus, sessionId, port);
  }

  /* close firewall and exit */
  closeExit(1);

  return 0;
}

/*********************************************/
/* calculate connect duration and put it out */
/*********************************************/
void putCloseMsg(time_t timeOut, time_t timeIn)
{
  double time;
  int hour,min,sec;

  time=difftime(timeOut,timeIn);
  hour=time/60/60;
  min=(time-hour*60*60)/60;
  sec=(time-hour*60*60-min*60);
  // err_msg("STOP: user %s at %s ( %02d:%02d:%02d )", userid, macAddr4, hour,min,sec);
  return;
}


/*****************************/
/* At termination, call this */
/*****************************/
void closeExit(int signo)
{
  /* signal is disabled */
  signalx(SIGTERM, SIG_DFL);

  /* write closing information to database */
  PutSessionEndToDb(cookie, mode[connectionMode]);

  /* delete old session log record from database */
  DeleteOldSessionInDb();
  
  /* ignore redundant process */
  if(connectionMode!=DUPLICATED){

    /* save the connect mode */
    logConnectMode();

    /* close firewalls */
    while(pClientAddr!=NULL){

      if(pClientAddr->ipType==IPV4){

	CloseClientGate4(pClientAddr,userid,macAddr4);
      }else{

	CloseClientGate6(pClientAddr,userid,macAddr6);
	DeleteNdpEntry(pClientAddr->ipAddr);
      }
      pClientAddr = pClientAddr->next;
    }

    /* put out time */
    timeOut=time(NULL);
    PutCloseMsg(timeOut,timeIn);
  }

  if(debug>1) err_msg("DEBUG:terminated");

  exit(1);
}

/************************************/
/* set up process title string      */
/************************************/
void setProcessTitle(char *useridshort, char *clientAddr4, char * ruleNumber4, char *clientAddr6, char * ruleNumber6, int ipStatus)
{
  switch(ipStatus){
  case IPV46DUAL:
    setproctitle("%s(useIPv6),[%s(%s)],[%s(%s)]", useridshort, 
		 clientAddr4, ruleNumber4, clientAddr6, ruleNumber6);
    break;

  case IPV4ONLY:
    setproctitle("%s,[%s(%s)],",useridshort, clientAddr4, ruleNumber4);
    break;

  case IPV6ONLY:
    setproctitle("%s(useIPv6),,[%s(%s)]",useridshort,  clientAddr6, ruleNumber6);
    break;

  default:
    err_msg("ERR at %s#%d: abnormal IP versions %d",__FILE__,__LINE__,ipStatus);
  }
}


/*****************************************/
/* save connectMode and others to syslog */
/*****************************************/
void logConnectMode()
{
  long time1sec,time2sec,time3sec;
  long time1usec,time2usec,time3usec;
  int conMode=0;

  /* set value on failure */
  conMode=connectionMode;
  if(connectionMode<0 || connectionMode>2) conMode=NOCONNECT;
  if(conMode==NOCONNECT){
    gettimeofday(&timeBeginWait, NULL) ;
    gettimeofday(&timeConnect, NULL) ;
  }

  /* get time at the end of connection (in msec) */
  gettimeofday(&timeDisconnect, NULL) ;

  /* calc time difference (seconds(sec) and micro-seconds(usec)) */
  time1sec=(timeBeginWait.tv_sec - timeBeginCgi.tv_sec);
  time1usec=(timeBeginWait.tv_usec - timeBeginCgi.tv_usec);
  /* if microsec diff is minus, bolow from sec diff */
  if(time1usec<0){ 
    time1usec += 1000000; time1sec--;
  }
  time2sec=(timeConnect.tv_sec - timeBeginWait.tv_sec);
  time2usec=(timeConnect.tv_usec - timeBeginWait.tv_usec);
  if(time2usec<0){
    time2usec += 1000000; time2sec--;
  }
  time3sec=(timeDisconnect.tv_sec - timeConnect.tv_sec);
  time3usec=(timeDisconnect.tv_usec - timeConnect.tv_usec);
  if(time3usec<0){
    time3usec += 1000000; time3sec--;
  }

  if(debug>0) err_msg("INFO: user=%s watchmode=%s procsec=%ld.%06ld,%ld.%06ld,%ld.%06ld ipversion=%d ipaddr=%s,%s macaddr=%s ipfwrule=%s,%s useragent=%s",
		      userid, mode[conMode], 
		      time1sec,time1usec,
		      time2sec,time2usec,
		      time3sec,time3usec,
		      ipStatus,
		      clientAddr4,clientAddr6,macAddr4,
		      ruleNumber4, ruleNumber6,
		      getenv("HTTP_USER_AGENT"));
}


/*****************************/
/*****************************/
void PutCloseMsg(time_t timeOut, time_t timeIn)
{
  if(debug>1) err_msg("DEBUG:=>putCloseMsg( )");
  putCloseMsg(timeOut,timeIn);
  if(debug>1) err_msg("DEBUG:<=putCloseMsg( )");
}

void SetProcessTitle(char *useridshort, char *clientAddr4, char * ruleNumber4, char *clientAddr6, char * ruleNumber6, int ipStatus){
  if(debug>1) err_msg("DEBUG:=>setProcessTitle(%s,%s,%s,%s,%s,%d)",useridshort,
		    clientAddr4,ruleNumber4,clientAddr6,ruleNumber6, ipStatus);
  setProcessTitle(useridshort,clientAddr4,ruleNumber4,clientAddr6,ruleNumber6,ipStatus);
  if(debug>1) err_msg("DEBUG:<=setProcessTitle( )");
}
