/*
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2007 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

#include <stdio.h>
#include <stdlib.h>

#include <tss/platform.h>
#include <tss/tss_defines.h>
#include <tss/tss_typedef.h>
#include <tss/tss_structs.h>
#include <tss/tss_error.h>
#include <tss/tspi.h>

#include <getopt.h>

// Local TCSD
#define SERVER    NULL


/*

 TPM PCR Read

 Usage:
  tpm_pcrread -p index 
 
 */

int hex2bin(void *dest, const void *src, size_t n);
void printhex(char *str, unsigned char *buf, int len);

/* options */
const struct option long_option[] = {
	{"pcrindex", required_argument, NULL, 'p'},
	{"help", no_argument, NULL, 'h'},
	{0, 0, 0, 0}
};
const char short_option[] = "p:h";

void usage() {
	printf("Usage: tpm_pcrread [options]\n");
	printf("\t-h, --help\n");
	printf("\t\tDisplay command usage info.\n");

	printf("\t-p, --pcrindex NUMBER\n");
	printf("\t\tPCR to read to.  Default is none.  This option can be specified multiple times to choose more than one PCR.\n");
}

int main(int argc, char *argv[])
{
	TSS_RESULT result;
	TSS_HCONTEXT hContext;

	TSS_HTPM hTPM;

	BYTE *blob;
	UINT32 blobLength;
	BYTE pcr[24];		// TODO

	int so;
	int pcrindex;


	/* Connect to TCSD */

	result = Tspi_Context_Create(&hContext);
	if (result != TSS_SUCCESS) {
		printf("ERROR: Tspi_Context_Create failed rc=0x%x\n",
		       result);
		goto close;
	}

	result = Tspi_Context_Connect(hContext, SERVER);
	if (result != TSS_SUCCESS) {
		printf("ERROR: Tspi_Context_Connect failed rc=0x%x\n",
		       result);
		goto close;
	}


	/* Get TPM handles */
	result = Tspi_Context_GetTpmObject(hContext, &hTPM);
	if (result != TSS_SUCCESS) {
		printf("ERROR: Tspi_Context_GetTpmObject failed rc=0x%x\n",
		       result);
		goto close;
	}


	/* OK, now we parse the option args */

	while (1) {
		so = getopt_long(argc, argv, short_option, long_option, 0);
		if (so == -1)
			break;	// END

		switch (so) {

		case 'p':	/* PCR */
			pcrindex = atoi(optarg);

			result =
			    Tspi_TPM_PcrRead(hTPM, pcrindex, &blobLength,
					     &blob);
			if (result != TSS_SUCCESS) {
				printf("ERROR: failed rc=0x%x\n", result);
				goto free;
			}
			printf("pcr.%d=", pcrindex);
			printhex("", blob, blobLength);
			Tspi_Context_FreeMemory(hContext, blob);

			pcr[pcrindex] = 1;

			break;
		case 'h':	/* help */
			usage();
			goto close;
		default:
			usage();
			goto close;
		}
	}


      free:
	Tspi_Context_FreeMemory(hContext, NULL);

	/* Close TSS/TPM */
      close:
	Tspi_Context_Close(hContext);


	return result;
}
