/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.eventlog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.engine.TPM;
import com.ibm.trl.tcg.pts.integrity.IntegrityReport;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.HexTool;

/**
 * 
 * @author Seiji Munetoh
 * 
 */
public class IML {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	final static int MAX_SML_SIZE = 64000; // For BIOS&IMA

	// final static int MAX_SML_SIZE = 10240; // For BIOS
	// static byte _sml[] = null;//new byte[MAX_SML_SIZE];
	private EventSet[] _e;

	// private boolean _verbose = false;
	private int _eventNum;

	TPM _tpm;

	// private int dbIndex;

	/**
	 * 
	 */
	public IML() {
		// this.dbIndex = dbIndex;

		_e = null;
		_eventNum = 0;
		_tpm = new TPM();
		// _tpm.init(prop)
	}

	public IML(TPM tpm) {
		_e = null;
		_eventNum = 0;
		_tpm = tpm;
	}

	/**
	 * @param file
	 * @return
	 * @throws Exception
	 */
	public void loadBIOSIML(String file) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("load BIOS IML file " + file);
		}

		byte[] sml = _loadBIOSIML(file);
		_e = createEventFromBIOSIML(sml);

	}

	private byte[] _loadBIOSIML(String file) throws Exception {
		File f = new File(file);
		if (f.isFile() == false) {
			throw new Exception("File not found " + file);
		}

		byte[] sml = new byte[MAX_SML_SIZE];

		try {
			// Pass 1, check the length
			FileInputStream fis = new FileInputStream(file);
			int len = fis.read(sml);
			fis.close();

			// Pass 2, create byte[]
			fis = new FileInputStream(file);
			sml = new byte[len];
			len = fis.read(sml);

			if (log.isDebugEnabled()) {
				log.debug(" binary_bios_measurement size = " + len);
			}

		} catch (FileNotFoundException e) {
			log.error("Error, File not found. filename = " + file);
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return sml;
	}

	/**
	 * @param sml
	 * 
	 */
	// private void createEvent() {
	private EventSet[] createEventFromBIOSIML(byte[] sml) {
		if (log.isTraceEnabled()) {
			log.trace("createEventFromBIOSIML ");
		}

		int ptr = 0;

		int pcrIndex;
		int type;
		byte digest[];
		byte eventdata[];
		// int num=0;

		// int eventNum[];
		// eventNum = new int[24];
		EventSet[] e = new EventSet[24];
		for (int i = 0; i < 24; i++) {
			e[i] = new EventSet();
			e[i].setPcrIndex(i);
		}

		while (ptr < sml.length) {
			pcrIndex = UINT32(sml, ptr);
			ptr += 4;
			type = UINT32(sml, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(sml, ptr);
			ptr += 20;
			eventdata = EventData(sml, ptr);
			ptr += 4 + eventdata.length;

			if (pcrIndex < 24) {
				e[pcrIndex].addEvent(type, digest, eventdata);
			} else {
				log.error("malfunced Event at " + _eventNum + " ?");
			}

			if (log.isDebugEnabled()) {
				String str = "PCR[" + pcrIndex + "],Type[" + type + "],Length["
						+ eventdata.length + "],digest["
						+ HexTool.getHexString(digest) + "]";
				log.debug(str);
			}
			_tpm.extend(pcrIndex, digest); // 20071111 SM
			_eventNum++;
		}

		if (log.isTraceEnabled()) {
			log.trace("createEventFromBIOSIML - done");
		}
		return e;
	}

	/**
	 * @param smlFilename
	 * @throws Exception
	 */
	public void addEventFromBIOSIML(String smlFilename) throws Exception {
		if (smlFilename == null)
			return;

		byte[] sml = _loadBIOSIML(smlFilename);
		if (sml == null)
			return;

		int ptr = 0;

		int pcrIndex;
		int type;
		byte digest[];
		byte eventdata[];

		while (ptr < sml.length) {
			pcrIndex = UINT32(sml, ptr);
			ptr += 4;
			type = UINT32(sml, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(sml, ptr);
			ptr += 20;
			eventdata = EventData(sml, ptr);
			ptr += 4 + eventdata.length;

			_e[pcrIndex].addEvent(type, digest, eventdata);

			if (log.isDebugEnabled()) {
				String str = "PCR[" + pcrIndex + "],Type[" + type + "],Length["
						+ eventdata.length + "],digest["
						+ HexTool.getHexString(digest) + "]";
				log.debug(str);
			}
			_eventNum++;
		}
	}

	public void loadIntegrityReport(String filename) throws Exception {
		IntegrityReport ir = new IntegrityReport(filename);
		_e = ir.getEventSet();
		_eventNum = ir.getEventNum();
	}

	/**
	 * Event[pcrindex][eventindex]
	 * 
	 * @return
	 */
	public EventSet[] getEvents() {
		return _e;
	}

	/**
	 * Return Event list (whole pcr)
	 * 
	 * @return
	 */

	public Event[] getEventList() {
		Event[] event = new Event[_eventNum];

		int index = 0;
		for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
			for (int i = 0; i < _e[pcrindex].size; i++) {
				event[index] = _e[pcrindex].getEvent(i);
				if (log.isTraceEnabled()) {
					log.trace(index + " " + i + " " + event[index].toString());
				}
				index++;
			}
		}

		return event;
	}

	/**
	 * Get events of pcr[pcrindex]
	 * 
	 * @param pcrindex
	 * @return
	 */
	public Event[] getEventList(int pcrindex) {
		Event[] event = new Event[_e[pcrindex].size];

		for (int i = 0; i < _e[pcrindex].size; i++) {
			event[i] = _e[pcrindex].getEvent(i);
		}
		return event;
	}

	/**
	 * @param message
	 * @return
	 */
	public Event[] getEvents(byte[] message) {
		int ptr = 0;

		int pcrIndex;
		int type;
		byte digest[];
		// int eventLength;
		byte eventdata[];
		int num = 0;

		// 1st pass
		// TODO Verify PCRs at this time
		while (ptr < message.length) {
			pcrIndex = UINT32(message, ptr);
			ptr += 4;
			type = UINT32(message, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(message, ptr);
			ptr += 20;
			eventdata = EventData(message, ptr);
			ptr += 4 + eventdata.length;

			if (log.isTraceEnabled()) {
				log.trace("PCR    " + pcrIndex);
				log.trace("Type   " + type);
				log.trace("Length " + eventdata.length);
				log.trace(HexTool.getHexString(digest));
			}
			num++;
		}

		if (log.isDebugEnabled()) {
			log.debug("EventNum    " + num);
		}

		// 2nd pass
		Event e[] = new Event[num];
		ptr = 0;
		num = 0;
		while (ptr < message.length) {
			e[num] = new Event();
			ptr = e[num].set(message, ptr);
			// e[num].pcrIndex = UINT32(message,ptr);
			// ptr +=4;
			// e[num].eventType = UINT32(message,ptr);
			// ptr +=4;
			// e[num].digest = SHA1_DIGEST(message,ptr);
			// ptr +=20;
			// e[num].eventData = EventData(message,ptr);
			// ptr += 4 + e[num].eventData.length;
			num++;
		}

		return e;
	}

	/**
	 * Re order the events based on PCR index (same as TSS's getEvent Func)
	 * 
	 * @param message
	 * @return
	 */
	public Event[] getEvents2(byte[] message) {
		int ptr = 0;

		int pcrIndex;
		int type;
		byte digest[];
		// int eventLength;
		byte eventdata[];
		int num = 0;

		int pcrNum[];
		pcrNum = new int[24];

		// 1st pass
		// TODO Verify PCRs at this time
		while (ptr < message.length) {
			pcrIndex = UINT32(message, ptr);
			pcrNum[pcrIndex]++;
			ptr += 4;
			type = UINT32(message, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(message, ptr);
			ptr += 20;
			eventdata = EventData(message, ptr);
			ptr += 4 + eventdata.length;

			if (log.isTraceEnabled()) {
				log.trace("PCR[" + pcrIndex);
				log.trace("],Type[" + type);
				log.trace("],Length[" + eventdata.length + "],digest[");
				log.trace(HexTool.getHexString(digest));
			}
			num++;
		}

		int offset[];
		offset = new int[25];

		for (int i = 0; i < 24; i++) {
			offset[i + 1] = pcrNum[i] + offset[i];
			if (log.isTraceEnabled()) {
				log.trace("EventNum PCR[" + i + "]=   " + pcrNum[i]
						+ " offset=" + offset[i]);
			}
		}
		if (log.isTraceEnabled()) {
			log.trace("EventNum (Total)   " + num);
		}

		// 2nd pass
		// DEBUG
		// NG log.debug("SM DEBUG getEvents2 LIMIT NUM 100");
		// if (num>100) num=100;

		Event e[] = new Event[num];

		ptr = 0;
		while (ptr < message.length) {
			pcrIndex = UINT32(message, ptr);
			ptr += 4;
			type = UINT32(message, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(message, ptr);
			ptr += 20;
			eventdata = EventData(message, ptr);
			ptr += 4 + eventdata.length;

			num = offset[pcrIndex];
			e[num] = new Event();
			e[num].set(pcrIndex, type, digest, eventdata);
			// e[num].pcrIndex = pcrIndex;
			// e[num].eventType = type;
			// e[num].digest = digest;
			// e[num].eventData = eventdata;
			offset[pcrIndex]++;
		}

		return e;
	}

	/**
	 * @param eventdata
	 */
	// private static void printHex(byte[] data) {
	// for (int i=0;i<data.length;i++) {
	// int d = data[i];
	// if (d < 0) {
	// d += 256;
	// }
	// if (d < 16) {
	// System.out.print("0");
	// }
	// System.out.print(Integer.toString(d, 16));
	// }
	// }
	/**
	 * @param message
	 * @param ptr
	 * @return
	 */
	private static byte[] EventData(byte[] message, int ptr) {
		// 
		int len = UINT32(message, ptr);

		byte b[] = new byte[len];
		int i;
		for (i = 0; i < len; i++) {
			b[i] = message[ptr + 4 + i];
		}
		return b;
	}

	/**
	 * @param message
	 * @param ptr
	 * @return
	 */
	private static byte[] SHA1_DIGEST(byte[] message, int ptr) {
		byte b[] = new byte[20];
		int i;
		for (i = 0; i < 20; i++) {
			b[i] = message[ptr + i];
		}
		return b;
	}

	/**
	 * @param b
	 * @return
	 */
	public static int UINT32(byte b[], int ptr) {
		int i;

		i = b[ptr + 3] & 0xFF;
		i = i << 8;
		i += b[ptr + 2] & 0xFF;
		i = i << 8;
		i += b[ptr + 1] & 0xFF;
		i = i << 8;
		i += b[ptr] & 0xFF;

		return i;
	}

	/**
	 * @param e
	 * @return
	 */
	// public static byte[] getPCEventData(Event e) {
	// byte[] pcdata = EventData(e.eventData,4); // skip type
	// return pcdata;
	// }

	/**
	 * @param base64BiosSML
	 */
	public void setBase64BiosSML(String base64BiosSML) {
		byte[] sml = null;
		try {
			sml = Base64Tool.decode(base64BiosSML);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		_e = createEventFromBIOSIML(sml);
	}

	public String toString() {
		String str = "";
		str += "Total " + _e.length + " events";
		return str;
	}

	public void print() {
		if (log.isDebugEnabled()) {
			log.debug("print IML ");
		}

		String pcrUsage[] = {
				"CRTM, POST BIOS, and Embedded Option ROMs", // 0
				"Host Platform Configuration", 
				"Option ROM Code",
				"Option ROM Configuration and Data", 
				"IPL",
				"IPL Configuration and Data", 
				"State Transition",
				"Host Platform Manufacturer Control", 
				"(Operating System)", // 8
				"", // 9
				"(IMA, LKM&ELF Components)", // 10
				"", // 11
				"", // 12
				"", // 13
				"", // 14
				"", // 15
				"Debug", // 16
				"", // 17
				"", // 18
				"", // 19
				"", // 20
				"", // 21
				"", // 22
				"Application Support", // 23
		};

		for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
			if (_e[pcrindex].size > 0) {
				System.out.println("--- PCR[" + pcrindex + "] "
						+ pcrUsage[pcrindex] + " ---");
				Event[] elist = _e[pcrindex].getEventList();
				for (int i = 0; i < elist.length; i++) {
					System.out.println(elist[i].toString());
					if (log.isTraceEnabled()) {
						log.trace(elist[i].toString());
					}
				}
			}
		}

		_tpm.print();

		if (log.isDebugEnabled()) {
			log.debug("print IML - done");
		}

	}

	/**
	 * 
	 * Update/replace Event of pcrindex
	 * 
	 * @param pcrindex
	 * @param events
	 */
	public void updateEvent(int pcrindex, Event[] events) {
		_e[pcrindex].setEventList(events);

		// TODO Auto-generated method stub

	}

	static int MAX_PCRS_NUM = 24; // TODO PCR NUM = 24

	/**
	 * Usage
	 * 
	 */
	private static void usage() {
		System.out.println("Usage: iml [OPTIONS]");
		System.out.println(" OPTIONS");
		System.out
				.println(" --text --in filename                           print IML by text");
		System.out
				.println(" --text --in filename1  --in filename2 print IMLs by text");
	}

	/**
	 * Main
	 * 
	 * @param args
	 */
	static int MAX_FILE_COUNT = 2;

	public static void main(String[] args) throws Exception {
		int opType = 0;
		int fileCount = 0;
		String[] inputFilename = new String[MAX_FILE_COUNT];
		// String outputFilename = null;
		String pcrsFilename = null;

		for (int i = 0; i < args.length; ++i) {
			if ("--text".equals(args[i])) {
				opType = 1;
			} else if ("--in".equals(args[i])) {
				if (fileCount < MAX_FILE_COUNT) {
					inputFilename[fileCount] = args[++i];
					fileCount++;
				} else {
					System.err
							.println("ERROR Supported eventlog file number is  "
									+ MAX_FILE_COUNT);
					usage();
				}
			} else if ("--pcrs".equals(args[i])) {
				pcrsFilename = args[++i];
			} else {
				System.err.println("Unknown arg " + args[i]);
				usage();
				return;
			}
		}

		switch (opType) {
		case 1: // --text --in inputFilename
			TPM tpm = new TPM();
			for (int i = 0; i < MAX_FILE_COUNT; i++) {
				if (inputFilename[i] != null) {
					IML iml = new IML(tpm);

					iml.loadBIOSIML(inputFilename[i]);

					if (pcrsFilename != null) {
						// TODO
					}
					iml.print();

					/* for next IML */
					tpm = iml.getTPM();
				}
			}
			break;

		default:
			usage();
			break;

		}

	}

	private TPM getTPM() {
		return _tpm;
	}

	/**
	 * Get Digest (base64 string) by Event Type
	 * 
	 * @param i
	 * @return
	 * @throws Exception
	 */

	public String getDigestByType(long type) throws Exception {
		String b64 = null;
		for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
			if (_e[pcrindex].size > 0) {
				Event[] elist = _e[pcrindex].getEventList();
				for (int i = 0; i < elist.length; i++) {
					if (elist[i].getEventType() == type) {
						// HIT
						b64 = Base64Tool.encode(elist[i].getDigest());
						return b64;
					}
				}
			}
		}
		if (log.isWarnEnabled()) {
			log.warn("Event Type " + type + " not found...");
		}
		throw new Exception("Event Type " + type + " not found...");
		// return null;
	}

	public int size() {
		return _eventNum;
	}
}
