/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */

package com.ibm.trl.tcg.pts.vulnerability.rpm;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.oval.ReadOvalRedhat;
import com.ibm.trl.tcg.pts.vulnerability.tool.MakeTM3File;

/**
 * Make TM3 file for visualization of the vulnerabiltiy map for RPM packages.
 * The input data is from perl script in tools/package/rpm/rpm-tm3.pl
 * 
 * @author Megumi Nakamura
 */
public class MakeTM3OfRpmVulnerability {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// For visualize
	private MakeTM3File makeTM3;

	// DB
	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	private SqlMapClient sqlMapIidb = null;

	// Input file
	private Properties inputProp = new Properties();

	private InputStream inStream = null;

	// Counter
	private static int lineNum = 0;

	/**
	 * Main method to start to load the xmlfile, parse them, and store into the
	 * database.
	 * 
	 * @param args --dbindex, --input, --output, --dir
	 */
	public static void main(String[] args) {
		String inputFile = null;
		String outputFile = null;
		String outputDir = null;
		int dbIndex = 0;

		for (int i = 0; i < args.length; ++i) {
			if ("--dbindex".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
			} else if ("--input".equals(args[i]) || "-i".equals(args[i])) {
				inputFile = args[++i];
			} else if ("--output".equals(args[i]) || "-o".equals(args[i])) {
				outputFile = args[++i];
			} else if ("--dir".equals(args[i]) || "-d".equals(args[i])) {
				outputDir = args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		if (inputFile == null) {
			usage();
			return;
		}
		if (outputFile == null) {
			outputFile = inputFile + ".tm3";
		}

		try {
			MakeTM3OfRpmVulnerability checker = new MakeTM3OfRpmVulnerability(dbIndex);
			checker.run(inputFile, outputFile, outputDir);
			checker.finish();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Constructor.
	 * 
	 * @param dbIndex
	 *            ID for Integrity database
	 */
	public MakeTM3OfRpmVulnerability(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: maketm3rpm -i tm3-data.txt -o output.tm3");
		System.out.println("--dbindex: (required)");
		System.out.println("--input, -i: (required)");
		System.out.println("	The output from the following command -");
		System.out.println("	$ perl rpm-tm3.pl");
		System.out.println("  ex. -i tm3-data.txt");
		System.out.println("--output, -o:");
		System.out.println("	Output file");
		System.out.println("--dir, -d:");
		System.out.println("	Output directory");
		System.out.println(" ");
	}

	/**
	 * Running the all process of preparing the file, checking the version,
	 * writing down in the file.
	 * 
	 * @param inputFile
	 *            The file contains the output from the command of rpm -qa
	 * @param outputFile
	 *            The file contains the result
	 * @throws Exception
	 *             All exceptions
	 */
	public void run(String inputFile, String outputFile) throws Exception {
		run(inputFile, outputFile, null);
	}

	/**
	 * Running the all process of preparing the file, checking the version,
	 * writing down in the file.
	 * 
	 * @param inputFile
	 *            The file contains the output from the command of rpm -qa
	 * @param outputFile
	 *            The file contains the result
	 * @param outputDir
	 *            The directory of outputFile
	 * @throws Exception
	 *             All exception
	 */
	public void run(String inputFile, String outputFile, String outputDir)
			throws Exception {
		try {
			sqlMapVul.startTransaction();
			sqlMapIidb.startTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

		writeFileOpen(outputFile, outputDir);
		getRPMSfromFile(inputFile);
	}

	/**
	 * Close the file stream.
	 */
	public void finish() {
		writeFileClose();

		try {
			sqlMapVul.commitTransaction();
			sqlMapIidb.commitTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Read the file, and store each line into Vector.
	 * 
	 * @param inputFile
	 *            The file contains the output from the command of rpm -qa
	 */
	private void getRPMSfromFile(String inputFile) {
		try {
			inStream = new FileInputStream(new File(inputFile));
			inputProp.load(inStream);

			Packages pkg = null;

			int i = 1;
			String name = null;
			String version = null;
			String release = null;
			String buildDate = null;
			long buildTime = 0;
			String size = null;
			String majorGroup = null;
			String minorGroup = null;
			String requiredNum = null;
			String requiredPkgs = null;
			while (!inputProp.isEmpty()) {
				name = inputProp.getProperty("Name." + i);
				if (name == null) {
					break;
				} else {
					name = name.trim();
				}
				version = inputProp.getProperty("Version." + i).trim();
				release = inputProp.getProperty("Release." + i).trim();
				if (release != null) {
					version = version + "-" + release;
				}
				buildDate = inputProp.getProperty("Build_Date." + i).trim();
				size = inputProp.getProperty("Size." + i).trim();
				majorGroup = inputProp.getProperty("MajorGroup." + i).trim();
				minorGroup = inputProp.getProperty("MinorGroup." + i).trim();
				requiredNum = inputProp.getProperty("Required_Num." + i);
				if (requiredNum != null) {
					requiredNum = requiredNum.trim();
				} else {
					requiredNum = "0";
				}
				requiredPkgs = inputProp.getProperty("Required_Packages." + i);
				if (requiredPkgs != null) {
					requiredPkgs = requiredPkgs.trim();
				} else {
					requiredPkgs = "";
				}

				// insert num of required-package
				pkg = new Packages();
				pkg.setPackageName(name);
				pkg.setPackageVersion(version);
				pkg.setPackageSize(Integer.valueOf(size));
				pkg.setPackageDependencyNum(Integer.valueOf(requiredNum));
				sqlMapIidb.update("updatePackageByNameVersion", pkg);
				sqlMapIidb.commitTransaction();

				i++;

				List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
						"getPackageByNameVersion", pkg);
				if (pkgs != null) {
					if (pkgs.size() > 0) {
						for (Packages p : pkgs) {
							int vul = p.getPackageVulnerability();
							// TODO there is no need to change this status ?
							if (vul == ReadOvalRedhat.NO_VUL_DISTRO_NOT_MATCH) {
								vul = ReadOvalRedhat.NO_VUL_DISTRO_MATCH;
							} else if (vul == ReadOvalRedhat.VUL_DISTRO_NOT_MATCH) {
								vul = ReadOvalRedhat.VUL_DISTRO_MATCH;
							}
							makeTM3.setIntegrity(vul);
							makeTM3.setCvss(p.getPackageCvssMax());
						}
					}
				}

				buildTime = changeTime(buildDate);
				makeTM3.setBuild(buildTime);
				makeTM3.setLoc(size);
				makeTM3.setLayer("Runtime\t" + majorGroup + "\t" + minorGroup
						+ "\t" + name + "-" + version + "-" + release);
				makeTM3.setRequiredNum(requiredNum);
				makeTM3.setRequiredPkgs(requiredPkgs);

				writeFileLineCountup();
			}

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Get time(msec) from 1970/1/1 00:00.
	 * 
	 * @param source
	 *            Wed 30 Aug 2006 10:34:15 PM JST
	 * @return Time in long format
	 */
	private long changeTime(String source) {
		try {
			String[] s = source.split(" ");
			SimpleDateFormat dateFormat = new SimpleDateFormat();
			dateFormat.setLenient(true);
			Date dat = null;
			if (s.length == 7) { // Fri 31 Aug 2007 10:29:06 AM JST
				dateFormat.applyPattern("dd MM yyyy kk:mm:ss");
				dat = dateFormat.parse(s[1] + " " + changeMonth(s[2]) + " "
						+ s[3] + " " + changeAmPm(s[4], s[5]));
			} else if (s.length == 5) { // Fri Nov 17 03:14:35 2006
				dateFormat.applyPattern("dd MM yyyy kk:mm:ss");
				dat = dateFormat.parse(s[2] + " " + changeMonth(s[1]) + " "
						+ s[4] + " " + s[3]);
			}
			if (dat != null) {
				return dat.getTime();
			} else {
				return 0;
			}
		} catch (ParseException e) {
			e.printStackTrace();
		}
		return 0;
	}

	/**
	 * Change Time in 12 hours to in 24 hours.
	 * 
	 * @param time
	 *            Time in 12 hours
	 * @param ampm
	 *            AM or PM
	 * @return Time in 24 hours
	 */
	private String changeAmPm(String time, String ampm) {
		String[] times = time.split(":");
		int hour = Integer.valueOf(times[0]).intValue();
		if (ampm.equalsIgnoreCase("PM")) {
			hour += 12;
			times[0] = Integer.toString(hour);
		}
		return times[0] + ":" + times[1] + ":" + times[2];
	}

	/**
	 * Change month of three characters(Jan-Dec) to digit string(01-12).
	 * 
	 * @param month
	 *            The month of three characters(Jan-Dec)
	 * @return The month of digit string(01-12)
	 */
	private String changeMonth(String month) {
		if (month.equalsIgnoreCase("Apr")) {
			return "04";
		} else if (month.equalsIgnoreCase("May")) {
			return "05";
		} else if (month.equalsIgnoreCase("Jun")) {
			return "06";
		} else if (month.equalsIgnoreCase("Jul")) {
			return "07";
		} else if (month.equalsIgnoreCase("Aug")) {
			return "08";
		} else if (month.equalsIgnoreCase("Sep")) {
			return "09";
		} else if (month.equalsIgnoreCase("Oct")) {
			return "10";
		} else if (month.equalsIgnoreCase("Nov")) {
			return "11";
		} else if (month.equalsIgnoreCase("Dec")) {
			return "12";
		} else if (month.equalsIgnoreCase("Jan")) {
			return "01";
		} else if (month.equalsIgnoreCase("Feb")) {
			return "02";
		} else if (month.equalsIgnoreCase("Mar")) {
			return "03";
		}
		return null;
	}

	/**
	 * Setup the file to write.
	 * 
	 * @param outputFile
	 *            The file contains the result
	 * @param outputDir
	 *            The directory name for outputFile
	 * @throws Exception
	 *             All exceptions
	 */
	private void writeFileOpen(String outputFile, String outputDir)
			throws Exception {
		makeTM3 = new MakeTM3File(outputFile, outputDir);
	}

	/**
	 * Close the file.
	 */
	private void writeFileClose() {
		makeTM3.writeFileClose();
	}

	/**
	 * The number written in the file. Rpm order.
	 */
	private void writeFileLineCountup() {
		lineNum++;
		makeTM3.writeVariable();
	}
}
