/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */

package com.ibm.trl.tcg.pts.vulnerability.rpm;

import com.ibm.trl.tcg.pts.vulnerability.tool.PackageVersionTool;

/**
 * RPM version compare.
 * 
 * @author Seiji Munetoh, Megumi Nakamura
 */
public class RpmVersionTool extends PackageVersionTool {

	private static final int PIECE_NUM = 6;

	private static final int PIECE_NAME = 0;

	private static final int PIECE_VER = 1;

	private static final int PIECE_REL1 = 2;

	private static final int PIECE_REL2 = 3;

	private static final int PIECE_DIST = 4;

	private static final int PIECE_ARCH = 5;

	/**
	 * Parse one line of "rpm -qa" or rpm filename to name, version, release and
	 * arch strings. if release contains distro info, it returns rel2 and dist
	 * string. e.g. kernel-2.6.9-34.EL => kernel 2.6.9 34.EL (34 EL)
	 * 
	 * @param rpmName
	 *            The name of rpm, which contains name, version and release.
	 * @return String[6]{Name, Version, Release, Release2, Distribution,
	 *         Architecture}
	 */
	public static String[] parseRpmName(String rpmName) {
		String[] rpmInfo = new String[PIECE_NUM];
		rpmInfo[PIECE_NAME] = null; // Name
		rpmInfo[PIECE_VER] = null; // Version
		rpmInfo[PIECE_REL1] = null; // Release
		rpmInfo[PIECE_REL2] = null; // Release2
		rpmInfo[PIECE_DIST] = null; // Dist
		rpmInfo[PIECE_ARCH] = null; // Arch

		/* Get Package Name */
		String[] parts = patternCheck("-", rpmName);
		if (parts.length == PIECE_REL1 + 1) { // General pattern
			rpmInfo[PIECE_NAME] = parts[0];
		} else if (rpmName.startsWith("gpg-pubkey")) {
			// gpg-pubkey-db42a60e-37ea5438
			rpmInfo[PIECE_NAME] = "gpg-pubkey";
		} else if (patternCheck("-v[0-9]", rpmName).length > 1) {
			// -v3.0
			parts = patternCheck("-v[0-9]", rpmName);
			rpmInfo[PIECE_NAME] = parts[0];
		} else if (patternCheck("-alpha[0-9]", rpmName).length > 1) {
			// -alpha3.0
			parts = patternCheck("-alpha[0-9]", rpmName);
			rpmInfo[PIECE_NAME] = parts[0];
		}
		if (rpmInfo[PIECE_NAME] == null) { // General pattern
			parts = patternCheck("-[0-9]", rpmName);
			if (parts[0].equalsIgnoreCase("java") && parts.length > 2) {
				// java-1.4.2-gcj-compat-1.4.2.0-27jpp
				rpmInfo[PIECE_NAME] = rpmName.substring(0, parts[0].length()
						+ parts[1].length() + 2);
			} else if (parts[0].startsWith("fonts-ISO8859") && parts.length > 2) {
				// fonts-ISO8859-2-100dpi-1.0-13, fonts-ISO8859-2-1.0-13
				for (int i = 0; i < parts.length; i++) {
					if (parts[i].endsWith("dpi")) {
						rpmInfo[PIECE_NAME] = rpmName.substring(0, rpmName
								.indexOf("dpi")
								+ "dpi".length());
					}
				}
				if (rpmInfo[PIECE_NAME] == null) {
					String[] tmp = patternCheck("-", rpmName);
					for (int j = 0; j < tmp.length; j++) {
						if (tmp[j].equalsIgnoreCase("ISO8859")
								&& tmp[j + 1] != null) {
							rpmInfo[PIECE_NAME] = rpmName
									.substring(0, rpmName.indexOf("ISO8859-"
											+ tmp[j + 1])
											+ "ISO8859-".length()
											+ tmp[j + 1].length());
						}
					}
				}
			} else if (parts[0].startsWith("fonts-KOI8-R") && parts.length > 2) {
				// fonts-KOI8-R-100dpi-1.0-7, fonts-KOI8-R-1.0-7
				for (int i = 0; i < parts.length; i++) {
					if (parts[i].endsWith("dpi")) {
						rpmInfo[PIECE_NAME] = rpmName.substring(0, rpmName
								.indexOf("dpi")
								+ "dpi".length());
					}
				}
				if (rpmInfo[PIECE_NAME] == null) {
					rpmInfo[PIECE_NAME] = parts[0];
				}
			} else if (parts[0].startsWith("fonts-xorg-ISO8859")
			// fonts-xorg-ISO8859-14-100dpi-6.8.2-1.EL
					&& parts.length > 2) {
				for (int i = 0; i < parts.length; i++) {
					if (parts[i].endsWith("dpi")) {
						rpmInfo[PIECE_NAME] = rpmName.substring(0, rpmName
								.indexOf("dpi")
								+ "dpi".length());
					}
				}
				if (rpmInfo[PIECE_NAME] == null) {
					rpmInfo[PIECE_NAME] = parts[0];
				}
			} else if (parts[0].startsWith("fonts-xorg") && parts.length > 2) {
				// fonts-xorg-100dpi-6.8.2-1.EL, fonts-xorg-base-6.8.2-1.EL,
				// fonts-xorg-syriac-6.8.2-1.EL
				for (int i = 0; i < parts.length; i++) {
					if (parts[i].endsWith("dpi")) {
						rpmInfo[PIECE_NAME] = rpmName.substring(0, rpmName
								.indexOf("dpi")
								+ "dpi".length());
					}
				}
				if (rpmInfo[PIECE_NAME] == null) {
					rpmInfo[PIECE_NAME] = parts[0];
				}
			} else {
				rpmInfo[PIECE_NAME] = parts[0];
			}
		}

		/* Get Version */
		String[] tmp = getVersionRelease(rpmName.substring(rpmInfo[PIECE_NAME]
				.length() + 1));
		rpmInfo[PIECE_VER] = tmp[0]; // version

		/* Get Release */
		rpmInfo[PIECE_REL1] = tmp[1]; // release

		/* Get Arch */
		String arch = checkArch(rpmInfo[PIECE_REL1]);
		if (arch != null) {
			/* Get Arch */
			rpmInfo[PIECE_ARCH] = arch;
			/* Then, Trim Release */
			rpmInfo[PIECE_REL1] = rpmInfo[PIECE_REL1].substring(0,
					rpmInfo[PIECE_REL1].indexOf(arch) - 1);
		} else {
			/* no arch */
			rpmInfo[PIECE_ARCH] = null;
		}

		/*
		 * release release2 Dist 13.EL4([2]) => 13([3]) EL4([4])
		 */
		String dist = checkDistribution(rpmInfo[PIECE_REL1]);
		if (dist != null) {
			/* Get Release2 */
			if (rpmInfo[PIECE_REL1].indexOf(dist) != 0) { // 13.EL4 => 13 EL4
				rpmInfo[PIECE_REL2] = rpmInfo[PIECE_REL1].substring(0,
						rpmInfo[PIECE_REL1].indexOf(dist) - 1);
			} else { // EL4.1 => null EL4.1
				rpmInfo[PIECE_REL2] = null;
			}
			/* Get Dist */
			rpmInfo[PIECE_DIST] = rpmInfo[PIECE_REL1]
					.substring(rpmInfo[PIECE_REL1].indexOf(dist));

		} else {
			rpmInfo[PIECE_REL2] = null;
			rpmInfo[PIECE_DIST] = null;
		}

		return rpmInfo;
	}

	/**
	 * Compare two version numbers, A(vera), B(verb). 1 means A is larget than
	 * B. -1 means B is larger than A. 0 means A is the same as B.
	 * 
	 * @param vera
	 *            The version number and the release number
	 * @param verb
	 *            The version number and the release number
	 * @return 1(A&gt;B), -1(A&lt;B), 0(A==B)
	 */
	public static int compareTwoRpmVersion(String vera, String verb) {
		return compareTwoVersion(vera, verb);
	}

	/**
	 * Compare two version numbers, A(ver1a), B(ver2 + rel2). 1 means A is
	 * larget than B. -1 means B is larger than A. 0 means A is the same as B.
	 * 
	 * @param ver1a
	 *            The version number and the release number
	 * @param ver2
	 *            The version number, used with rel2
	 * @param rel2
	 *            The release number, used with ver2
	 * @return 1(A&gt;B), -1(A&lt;B), 0(A==B)
	 */
	public static int compareTwoRpmVersion(String ver1a, String ver2,
			String rel2) {
		if (ver1a.equals(ver2 + "-" + rel2)) {
			return EQUAL;
		}

		// parse versionInDB
		String[] tmp = getVersionRelease(ver1a);
		String ver1 = tmp[0];
		String rel1 = tmp[1];

		// check
		int rtn = compareFigureStrings(patternCheck("[-._]+", ver1),
				patternCheck("[-._]+", ver2));
		if (rtn == EQUAL) {
			rtn = compareFigureStrings(patternCheck("[-._]+", rel1),
					patternCheck("[-._]+", rel2));
		}
		return rtn;
	}

	/**
	 * Compare the version number of rpm with the number from oval. 1 means
	 * dbValue is larget than ovalValue. -1 means ovalValue is larger than
	 * dbValue. 0 means dbValue is the same as ovalValue.
	 * 
	 * @param dbValue
	 *            The version number stored in the database
	 * @param ovalValue
	 *            The version number came from oval
	 * @return 1(dbValue&gt;ovalValue), -1(dbValue&lt;ovalValue),
	 *         0(dbValue==ovalValue)
	 */
	public static int compareRpmOval(String dbValue, String ovalValue) {
		if (dbValue.equals(ovalValue)) {
			return EQUAL;
		}

		// parse db
		String[] dbStrings = patternCheck("[:-]", dbValue); // VERSION-RELEASE
		// etc.
		String[] dbVersionStrings = patternCheck("[._]", dbStrings[0]); // VERSION
		// ->
		// x.y.z
		// parse oval
		String[] ovalStrings = patternCheck("[:-]", ovalValue); // EPOCH:VERSION-RELEASE
		String[] ovalVersionStrings = patternCheck("[._]", ovalStrings[1]); // VERSION->x.y.z

		// compare version
		int rtn = compareFigureStrings(dbVersionStrings, ovalVersionStrings);
		// compare release
		if (rtn == EQUAL) {
			String[] ovalReleaseStrings = patternCheck("[._]", ovalStrings[2]); // RELEASE->a.b.c
			String[] dbReleaseStrings = null;
			if (dbStrings[1] != null) {
				dbReleaseStrings = patternCheck("[._]", dbStrings[1]); // RELEASE->a.b.c
			} else {
				dbReleaseStrings = new String[1];
				dbReleaseStrings[0] = "0";
			}
			rtn = compareFigureStrings(dbReleaseStrings, ovalReleaseStrings);
		}
		return rtn;
	}

	/**
	 * Judge whether the param string is architecture name.
	 * 
	 * @param rel
	 *            The String of the release number
	 * @return The architecture name extracted from the param
	 */
	private static String checkArch(String rel) {
		if (rel.indexOf("i386") != -1) {
			return "i386";
		}
		if (rel.indexOf("i486") != -1) {
			return "i486";
		}
		if (rel.indexOf("i586") != -1) {
			return "i586";
		}
		if (rel.indexOf("i686") != -1) {
			return "i686";
		}
		if (rel.indexOf("athron") != -1) {
			return "athron";
		}
		if (rel.indexOf("ppc") != -1) {
			return "ppc";
		}
		if (rel.indexOf("noarch") != -1) {
			return "noarch";
		}
		return null;
	}

}
