/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.tool;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.Debians;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.oval.ReadOvalRedhat;

/**
 * DB operations. list summary - fix vulnerability status - update obsolete -
 * delete vulnerablity infomation from iidb
 * 
 * @author Megumi Nakamura
 * 
 */
public class DbOperations {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private SqlMapClient sqlMapIidb = null;

	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	public static final int NO_VUL = ReadOvalRedhat.NO_VUL_DISTRO_MATCH; // 1

	public static final int VUL = ReadOvalRedhat.VUL_DISTRO_MATCH; // 3

	public static final int UNCLEAR = ReadOvalRedhat.CHECK_AGAIN; // 10

	private static final int CHANGED_MANUALLY = 10;

	public DbOperations() {
	}

	public void listVuldbSummary() {
		// TODO
	}

	/**
	 * List all Iidb Summary to stdout.
	 */
	public void listAllIidbSummary() {
		try {
			Properties prop = new Properties();
			InputStream ips = this.getClass().getClassLoader()
					.getResourceAsStream("sqlMapsConfig.properties");
			prop.load(ips);
			for (int i = 0; i < 8; i++) {
				String iidbUrl = prop.getProperty("url_iidb" + i);
				if (iidbUrl.length() > 0) {
					String[] urls = iidbUrl.split("/");
					String iidbName = urls[urls.length - 1];
					if (iidbName != null) {
						System.out.println("===================");
						System.out.println("Current Summary of " + iidbName);
						listIidbSummary(i);
					}
				}
			}
			System.out.println("===================");

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * List one Iidb Summary to stdout.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	public void listIidbSummary(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
		try {
			Object packageAllNum = sqlMapIidb.queryForObject("getPackageNum");
			Object digestAllNum = sqlMapIidb.queryForObject("getDigestNum");
			System.out.println("IIDB index:\t" + dbIndex);
			System.out.println("packages:\t" + packageAllNum);
			System.out.println("measuremnets:\t" + digestAllNum);

			// 0: unchecked, 1: safe, 3: vulnerable, 10: unclear
			int[] vulid = {ReadOvalRedhat.VUL_DISTRO_MATCH,
					ReadOvalRedhat.NO_VUL_DISTRO_MATCH,
					ReadOvalRedhat.CHECK_AGAIN, 0 };
			for (int i = 0; i < vulid.length; i++) {
				Object packageNum = sqlMapIidb.queryForObject(
						"getPackageNumByVulnerability", vulid[i]);
				Object digestNum = sqlMapIidb.queryForObject(
						"getDigestNumByVulnerability", vulid[i]);
				if (vulid[i] == 0) {
					System.out.println(" - unchecked:");
				} else if (vulid[i] == ReadOvalRedhat.NO_VUL_DISTRO_MATCH) {
					System.out.println(" - safe:");
				} else if (vulid[i] == ReadOvalRedhat.VUL_DISTRO_MATCH) {
					System.out.println(" - vulnerable:");
				} else if (vulid[i] == ReadOvalRedhat.CHECK_AGAIN) {
					System.out.println(" - unclear:");
				}
				System.out.println("\tpackage    \t" + packageNum);
				System.out.println("\tmeasurement\t" + digestNum);
			}
		} catch (SQLException e) {
			log.error(e.getMessage());
		}
	}

	/**
	 * Fix the vulnerability status(unclear) manually.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	public void fixUnclearPackage(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
		try {
			sqlMapIidb.startTransaction();

			BufferedReader bufReader = new BufferedReader(
					new InputStreamReader(System.in), 1);

			List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByVulnerability", ReadOvalRedhat.CHECK_AGAIN);
			System.out.println("Unclear packages num: " + pkgs.size());
			System.out.println("===================");
			int counter = 1;
			for (Packages p : pkgs) {
				System.out.println("Unclear package #" + counter++);
				System.out.println("Package name:\t" + p.getPackageName());
				System.out.println(" - version:\t" + p.getPackageVersion());

				String ovaldsa = p.getPackageOval();
				System.out.println("associated OVAL/DSA:\t" + ovaldsa);
				System.out.println(" - CVE:\t" + p.getPackageCve());

				if (ovaldsa != null) {
					if (ovaldsa.startsWith("DSA")) {
						Debians deb = (Debians) sqlMapVul.queryForObject(
								"getDebianByDsaid", ovaldsa);
						System.out.println(" - URL:\t" + deb.getDebianUrl());
						System.out.println(" - version(etch) :\t"
								+ deb.getDebianFixedEtch());
						System.out.println(" - version(sarge):\t"
								+ deb.getDebianFixedSarge());
						System.out.println(" - version(sid)  :\t"
								+ deb.getDebianFixedSid());

						Measurements mgmt = new Measurements();
						System.out
								.println("\nYour decision [VALID/INVALID/UNVERIFIED]?");
						String decision = bufReader.readLine();
						if (decision != null) {
							if (decision.equalsIgnoreCase("VALID")) {
								p
										.setPackageVulnerability(ReadOvalRedhat.NO_VUL_DISTRO_MATCH
												+ CHANGED_MANUALLY);
								mgmt
										.setDigestVulnerability(ReadOvalRedhat.NO_VUL_DISTRO_MATCH
												+ CHANGED_MANUALLY);
							} else if (decision.equalsIgnoreCase("INVALID")) {
								p
										.setPackageVulnerability(ReadOvalRedhat.VUL_DISTRO_MATCH
												+ CHANGED_MANUALLY);
								mgmt
										.setDigestVulnerability(ReadOvalRedhat.VUL_DISTRO_MATCH
												+ CHANGED_MANUALLY);
							} else if (decision.equalsIgnoreCase("UNVERIFIED")) {
								// TODO
							}
							sqlMapIidb.update("updatePackageVulnerability", p);
							sqlMapIidb.commitTransaction();

							mgmt.setDigestPackageId(p.getPackageId());
							sqlMapIidb.update(
									"updateDigestVulnerabilityByPackageId",
									mgmt);
							sqlMapIidb.commitTransaction();
						}

					} else if (ovaldsa.startsWith("oval")) {
						// TODO
					}
				}

				System.out.println("===================");
			}

		} catch (SQLException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Check and update obsolete packages.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	
	public void updateObsolete(int dbIndex) {
		updateObsolete(dbIndex, true);
	}
	
	/**
	 * Check and update obsolete packages.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 * @param clearMeasurementObsolete If true, skip to make obsolete clear before updating.
	 */
	public void updateObsolete(int dbIndex, boolean clearMeasurementObsolete) {
		try {
			sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
			sqlMapIidb.startTransaction();

			String version = null;
			String newestVersion = null;
			Packages newestPkg = null;

			List<Packages> pkgall = (List<Packages>) sqlMapIidb
					.queryForList("getPackageGroup");
			for (Packages ps : pkgall) {
				List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
						"getPackageByName", ps.getPackageName());
				if (pkgs.size() > 1) {
					for (Packages p : pkgs) {
						version = p.getPackageVersion();
						if (newestVersion != null) {
							int rtn = PackageVersionTool.compareTwoVersion(
									version, newestVersion);
							if (rtn == PackageVersionTool.GREATER) {
								newestVersion = version;
								newestPkg = p;
							}
						} else {
							newestVersion = version;
							newestPkg = p;
						}
					}

					// Update Packages obsolete = 1 when pkg != newest
					sqlMapIidb.update("updatePackageObsoleteByNameVersion",
							newestPkg);
					sqlMapIidb.commitTransaction();
					
					List<Packages> newPkgs = (List<Packages>) sqlMapIidb.queryForList(
							"getPackageByNameVersion", newestPkg);
					// Update Newest Package to obsolete = 0
					sqlMapIidb.update("updatePackageObsoleteZeroByNameVersion",
							newestPkg); 
					sqlMapIidb.commitTransaction();
					
				}
				newestVersion = null;
				newestPkg = null;

			}
			if(clearMeasurementObsolete){
				// Update All Measurements to obsolete = 0
				sqlMapIidb.update("updateDigestObsoleteZeroAll");
				sqlMapIidb.commitTransaction();
			}
			// Update Measurements where packages.obsolete = 1
			List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList("getPackageByObsolete");
			for (Packages p : pkgs) {
				sqlMapIidb.update("updateDigestObsoletePid", p.getPackageId());
				sqlMapIidb.commitTransaction();
			}

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Delete vulnerability information from pacakges, measurements.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	public void deleteVulInfo(int dbIndex) {
		deleteVulInfoFromPackages(dbIndex);
		deleteVulInfoFromMeasurements(dbIndex);
	}

	/**
	 * Delete vulnerability information from measurements.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	public void deleteVulInfoFromMeasurements(int dbIndex) {
		try {
			sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
			sqlMapIidb.startTransaction();

			sqlMapIidb.update("deleteDigestVulnerability");
			sqlMapIidb.commitTransaction();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Delete vulnerability information from pacakges.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	public void deleteVulInfoFromPackages(int dbIndex) {
		try {
			sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
			sqlMapIidb.startTransaction();

			sqlMapIidb.update("deletePackageVulnerability");
			sqlMapIidb.commitTransaction();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * List Vulnerable packages.
	 * 
	 * 20071114 Munetoh added
	 * 
	 * @param dbIndex
	 *            IIDB index
	 * @param flag
	 *            3 VUL, 10 GUESS, 13 VUL(manual)
	 * @throws Exception
	 *             All exceptions
	 */
	public void listVulnerablePackage(int dbIndex, int flag) throws Exception {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
		sqlMapIidb.startTransaction();

		List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
				"getPackageByVulnerability", flag);
		System.out.println("Vulnerable packages num: " + pkgs.size());
		System.out
				.println("--------------------------------------------------");
		int counter = 1;
		for (Packages p : pkgs) {
			System.out.println("Vulnerable package #   : " + counter++);
			System.out
					.println("Package name           : " + p.getPackageName());
			System.out.println(" - version             : "
					+ p.getPackageVersion());

			String ovaldsa = p.getPackageOval();
			System.out.println(" - associated OVAL/DSA : " + ovaldsa);
			System.out.println(" - CVE                 : " + p.getPackageCve());
			System.out.println(" - CVSS                : "
					+ p.getPackageCvssMin() + " - " + p.getPackageCvssMax());
			System.out
					.println(" - size                : " + p.getPackageSize());

			if (ovaldsa != null) {
				if (ovaldsa.startsWith("DSA")) {
					Debians deb = (Debians) sqlMapVul.queryForObject(
							"getDebianByDsaid", ovaldsa);
					System.out.println(" - URL                 : "
							+ deb.getDebianUrl());
					System.out.println(" - fixed version(etch) : "
							+ deb.getDebianFixedEtch());
					System.out.println(" - fixed version(sarge): "
							+ deb.getDebianFixedSarge());
					System.out.println(" - fixed version(sid)  : "
							+ deb.getDebianFixedSid());
				} else if (ovaldsa.startsWith("oval")) {
					// TODO
				}
			}

			System.out
					.println("--------------------------------------------------");
		}
		sqlMapIidb.endTransaction();
	}
	
	
	/**
	 * 
	 * @param args --list-all-db, --list-db, --fix-unclear, --update-obsolete, --delete-vul
	 */
	public static void main(String[] args) {
		int dbIndex = 0;

		DbOperations ope = new DbOperations();
		for (int i = 0; i < args.length; ++i) {
			if ("--list-all-db".equals(args[i])) {
				ope.listAllIidbSummary();
				return;
			} else if ("--list-db".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
				ope.listIidbSummary(dbIndex);
				return;
			} else if ("--fix-unclear".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
				ope.fixUnclearPackage(dbIndex);
				return;
			} else if ("--update-obsolete".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
				ope.updateObsolete(dbIndex);
				return;
			} else if ("--delete-vul".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
				ope.deleteVulInfo(dbIndex);
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
	}
	
	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: db [OPTIONS]");
		System.out.println(" --list-all-db");
		System.out.println(" --list-db [dbIndex]");
		System.out.println(" --fix-unclear [dbIndex]");
		System.out.println(" --update-obsolete [dbIndex]");
		System.out.println(" --delete-vul [dbIndex]");
		System.out.println(" ");
	}

}
