/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/ifm.c
 * \brief TCG IF-M protocol
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-04-01
 * cleanup 2011-01-22 SM
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <unistd.h>
#include <fcntl.h>

#ifdef  HAVE_SENDFILE
#include <sys/sendfile.h>
#endif

#include <openpts.h>

// TODO
#define MAX_TLV_MESSAGE_LENGTH 5120000


// DEBUG
// 2011-02-24 SM make check => pass
// #undef HAVE_SENDFILE

#ifndef HAVE_SENDFILE
#define SENDFILE_BUF_SIZE 1024
// http://linux.die.net/man/2/sendfile
// sendfile - transfer data between file descriptors
ssize_t my_sendfile(int out_fd, int in_fd, off_t *offset, size_t count) {
    char buf[SENDFILE_BUF_SIZE];
    size_t read_size;
    size_t write_size;
    ssize_t sum = 0;

    DEBUG("my_sendfile() ############################\n");

    // offset == NULL
    do {
        /* set read size */
        if ((count - sum) > SENDFILE_BUF_SIZE) {
            read_size = SENDFILE_BUF_SIZE;
        } else {
            read_size = count - sum;
        }

        /* read */
        read_size = read(in_fd, buf, read_size);
        if (read_size < 0) {
            // sum = -1;
            break;
        }

        /* write */
        write_size = write(out_fd, buf, read_size);

        if (write_size < 0) {
            ERROR("\n");
            sum = -1;
            break;
        }
        if (write_size !=  read_size) {
            ERROR("\n");
            sum = -1;
            break;
        }

        sum += write_size;
    } while (sum < (ssize_t) count);

    return sum;
}
#define sendfile my_sendfile
#endif  // !HAVE_SENDFILE

/**
 * read IF-M PTS message (standalone)
 *
 * This just fill the PTS_IF_M_Attribute structure.
 * The received packet is parsed by in ptscd.c
 */
// PTS_IF_M_Attribute *readPtsTlvFromSock(OPENPTS_CONTEXT *ctx, int sock) {
PTS_IF_M_Attribute *readPtsTlvFromSock(int sock) {
    int rc;
    BYTE head[12];
    int ptr;
    int rest;
    PTS_Byte * read_msg = NULL;  // TODO(munetoh)
    PTS_IF_M_Attribute *read_tlv;

    DEBUG_CAL("readPtsTlvFromSock - start\n");

    memset(head, 0, 12);

    /* malloc TLV for read */
    read_tlv = (PTS_IF_M_Attribute *)malloc(sizeof(PTS_IF_M_Attribute));
    if (read_tlv == NULL) {
        ERROR("no memory");
        return NULL;
    }
    memset(read_tlv, 0, sizeof(PTS_IF_M_Attribute));

    /* read header */
    rc = read(sock, head, 12);
    if (rc == 0) {
        DEBUG("sock read fail. probably end of the handshake\n");
        free(read_tlv);
        return NULL;
    }

    // TODO(munetoh) nltoh?
    read_tlv->flags  = head[0];
    read_tlv->vid[0] = head[1];
    read_tlv->vid[1] = head[2];
    read_tlv->vid[2] = head[3];
    read_tlv->type = (head[4] << 24) | (head[5] << 16) |
                     (head[6] << 8) | head[7];
    read_tlv->length = (head[8] << 24) | (head[9] << 16) |
                       (head[10] << 8) | head[11];

    /* check the length */
    if (read_tlv->length > MAX_TLV_MESSAGE_LENGTH) {
        ERROR("read_tlv->length = %d > %d\n", read_tlv->length, MAX_TLV_MESSAGE_LENGTH);
        return NULL;
    }

    /* read msg body */
    if (read_tlv->length > 0) {
        read_msg = (PTS_Byte *)malloc(read_tlv->length +1);
        if (read_msg == NULL) {
            ERROR("ERROR malloc %d\n", read_tlv->length +1);
            return NULL;
        } else {
            ptr = 0;
            rest = read_tlv->length;
            while (1) {
                rc = read(sock, &read_msg[ptr], rest);
                if (rc ==0) {
                    break;
                }
                ptr += rc;
                rest -= rc;

                if (rest < 0) {
                    break;
                }
            }
        }
        read_msg[read_tlv->length] = 0;
        read_tlv->value = read_msg;
    } else {
        read_tlv->value = NULL;
    }

    /* */

    DEBUG_IFM("IF-M read,  type=0x%08x, length=%d\n",
        read_tlv->type, read_tlv->length);

    DEBUG_CAL("readPtsTlvFromSock - done\n");

    return read_tlv;
}

/**
 * free PTS_IF_M_Attribute
 */
void freePtsTlv(PTS_IF_M_Attribute *tlv) {
    if (tlv == NULL) {
        return;
    }

    /* free*/
    if (tlv->value != NULL) {
        free(tlv->value);
    }
    free(tlv);
}




/**
 * write IF-M PTS message (socket, standalone)
 *
 * we are using sendfile() here and send the data steb by step. 
 * but IF-M of IMC/IMV version need to create whole blob to send.
 *
 * Retrun
 *  length of write data
 *  -1 ERROR
 */
int writePtsTlvToSock(OPENPTS_CONTEXT *ctx, int sock, int type) {
    int rc;
    PTS_IF_M_Attribute *write_tlv;
    int num = 2;
    BYTE head[12];
    BYTE buf[64];

    // TODO MAX_RM_NUM > 0
    int i;
    int fsize[MAX_RM_NUM];
    int fd[MAX_RM_NUM];
    int count[MAX_RM_NUM];
    struct stat st[MAX_RM_NUM];

    OPENPTS_CONFIG *conf;

    /* check */
    if (ctx == NULL) {
        ERROR("ctx is NULL\n");
        return -1;
    }
    conf = ctx->conf;
    if (conf == NULL) {
        ERROR("conf is NULL\n");
        return -1;
    }
    if (conf->uuid == NULL) {
        ERROR("writePtsTlvToSock() uuid is NULL\n");
        return -1;
    }

    /* init */
    for (i = 0; i < MAX_RM_NUM; i++) {
        fd[i] = -1;
    }

    DEBUG_CAL("writePtsTlvToSock - start\n");

    /* malloc TLV */
    write_tlv = (PTS_IF_M_Attribute *)malloc(sizeof(PTS_IF_M_Attribute));
    if (write_tlv == NULL) {
        ERROR("no memory");
        return -1;
    }

    /* check the TLV type */
    switch (type) {
    /* Collector <-> Verifier */
    case OPENPTS_CAPABILITIES:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        // UUID
        write_tlv->length = sizeof(OPENPTS_IF_M_Capability);  // 16 * 2;  // collector UUID + RM UUID
        break;

    /* Verifier -> Collector, ( */
    /* DH: Initiator -> Respondor */
    case DH_NONCE_PARAMETERS_REQUEST:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 4;
        // DEBUG("DH_NONCE_PARAMETERS_REQUEST write_tlv->length=%d\n",write_tlv->length);
        break;
    case DH_NONCE_FINISH:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length =
            4 +
            ctx->nonce->initiator_nonce_length +
            ctx->nonce->pubkey_length;
        // DEBUG("DH_NONCE_FINISH write_tlv->length=%d\n",write_tlv->length);
        break;
    /* RIMM */
    case REQUEST_RIMM_SET:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    /* NEW RIMM */
    case REQUEST_NEW_RIMM_SET:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    /* IR */
    case REQUEST_INTEGRITY_REPORT:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    /* VR */
    case VERIFICATION_RESULT:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;

    /* Collector -> Verifier */
    /* DH: IRespondor -> Initiator */
    case DH_NONCE_PARAMETORS_RESPONSE:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length =
            4 + 4 +
            ctx->nonce->respondor_nonce_length +
            ctx->nonce->pubkey_length;
        // DEBUG("DH_NONCE_PARAMETORS_RESPONSE write_tlv->length=%d\n",write_tlv->length);
        break;
    /* RIMM */
    case RIMM_SET:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;

        /* body - two RMs (runtime and platform)  */

        /* open Platform RM */
        write_tlv->length = 4;
        num = 0;
        for (i = 0; i < conf->rm_num; i++) {
            /* open */
            fd[i] = open(ctx->conf->rm_filename[i], O_RDONLY);
            if (fd[i] < 0) {
                // 20101124 SM must be a fullpath for Daemon
                ERROR("Error RM file, %s not found\n", ctx->conf->rm_filename[i]);
                goto error;
            }
            stat(ctx->conf->rm_filename[i], &st[i]);
            fsize[i] = st[i].st_size;
            write_tlv->length += 4 + fsize[i];
            /* close */
            close(fd[i]);
            fd[i] = -1;
            num++;
        }
        break;
    /* NEW RIMM */
    case NEW_RIMM_SET:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;

        /* body - two RMs (runtime and platform)  */

        /* open Platform RM */
        write_tlv->length = 16 + 4;  // UUID + num
        num = 0;
        for (i = 0; i < conf->newrm_num; i++) {
            fd[i] = open(ctx->conf->newrm_filename[i], O_RDONLY);
            if (fd[i] < 0) {
                // 20101124 SM must be a fullpath for Daemon
                ERROR("Error RM file, %s not found\n", ctx->conf->newrm_filename[i]);
                goto error;
            }
            stat(ctx->conf->newrm_filename[i], &st[i]);
            fsize[i] = st[i].st_size;
            write_tlv->length += 4 + fsize[i];
            /* close */
            close(fd[i]);
            fd[i] = -1;
            num++;
        }
        break;
    /* IR */
    case INTEGRITY_REPORT:
        write_tlv->flags = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type = type;

        // DEBUG_IFM("IR file=%s\n", ctx->conf->ir_filename);

        /* generate new IR */
        rc = genIr(ctx);
        if (rc != PTS_SUCCESS) {
            ERROR("writePtsTlvToSock - gen IR failed\n");
            write_tlv->length = 0;
            break;
        }

        /* send new IR */

        /* body */
        // TODO fd[0]?
        fd[0] = open(ctx->conf->ir_filename, O_RDONLY);  // TODO(munetoh)
        if (fd[0] < 0) {
            ERROR("Error %s not found\n", ctx->conf->ir_filename);
            goto error;
        }
        stat(ctx->conf->ir_filename, &st[0]);
        fsize[0] = st[0].st_size;
        write_tlv->length = fsize[0];
        /* close */
        close(fd[0]);
        fd[0] = -1;
        break;
    /* OpenPTS */
#ifdef CONFIG_AIDE
    /* send AIDE DATABASE */
    case REQUEST_AIDE_DATABASE:
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case AIDE_DATABASE:
        write_tlv->flags = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type = type;

        /* body */
        if (ctx->conf->aide_database_filename == NULL) {
            // Test
            DEBUG("writePtsTlvToSock - Error AIDE DB file is not configured\n");
            write_tlv->length = 0;
        } else {
            fd[0] = open(ctx->conf->aide_database_filename, O_RDONLY);
            if (fd[0] < 0) {
                /* erorr */
                ERROR("writePtsTlvToSock - Error AIDE DB file, %s not found\n", ctx->conf->aide_database_filename);
                write_tlv->length = 0;
            } else {
                /* OK */
                stat(ctx->conf->aide_database_filename, &st[0]);
                fsize[0] = st[0].st_size;
                write_tlv->length = fsize[0];
                /* close */
                close(fd[0]);
                fd[0] = -1;
            }
        }
        break;
#endif
    /* send TPM_PUBKEY */
    case REQUEST_TPM_PUBKEY:
        // DEBUG("REQUEST_TPM_PUBKEY\n");
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case TPM_PUBKEY:
        // DEBUG("TPM_PUBKEY len %d\n", ctx->conf->pubkey_length);
        write_tlv->flags = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type = type;
        if (ctx->conf->pubkey_length > 0) {
            // DEBUG("TPM_PUBKEY len %d\n", ctx->conf->pubkey_length);
            if (ctx->conf->pubkey != NULL) {
                write_tlv->length = ctx->conf->pubkey_length;
            } else {
                ERROR("writePtsTlvToSock - PUBKEY blob is missing\n");
                write_tlv->length = 0;
            }
        } else {
            /* missing */
            ERROR("writePtsTlvToSock - PUBKEY blob is missing\n");
            write_tlv->length = 0;
        }
        break;
    default:
        // BAT type
        ERROR("BAD IFM message TYPE 0x%08x\n", type);
        goto error;  // return -1;
    }

    /* Send message */
    // TODO(munetoh) htonl?

    /* head */
    head[0] = write_tlv->flags;
    head[1] = write_tlv->vid[0];
    head[2] = write_tlv->vid[1];
    head[3] = write_tlv->vid[2];
    head[4] = (write_tlv->type & 0xFF000000) >> 24;
    head[5] = (write_tlv->type & 0xFF0000) >> 16;
    head[6] = (write_tlv->type & 0xFF00) >> 8;
    head[7] = (write_tlv->type & 0xFF);
    head[8] = (write_tlv->length & 0xFF000000) >> 24;
    head[9] = (write_tlv->length & 0xFF0000) >> 16;
    head[10] = (write_tlv->length & 0xFF00) >> 8;
    head[11] = (write_tlv->length & 0xFF);
    rc = write(sock, head, 12);

    /* value */
    if (write_tlv->length > 0) {
        /* RIMM - sendfile */
        if (type == RIMM_SET) {  // 2 files
            /* RM msg */
            // NUM
            buf[0] = (num & 0xFF000000) >> 24;
            buf[1] = (num & 0xFF0000) >> 16;
            buf[2] = (num & 0xFF00) >> 8;
            buf[3] = (num & 0xFF);
            rc = write(sock, buf, 4);

            for (i = 0; i< conf->rm_num; i++) {
                // LEN1
                buf[0] = (fsize[i] & 0xFF000000) >> 24;
                buf[1] = (fsize[i] & 0xFF0000) >> 16;
                buf[2] = (fsize[i] & 0xFF00) >> 8;
                buf[3] = (fsize[i] & 0xFF);
                rc = write(sock, buf, 4);

                // BODY1
                fd[i] = open(ctx->conf->rm_filename[i], O_RDONLY);
                count[i] = sendfile(sock, fd[i], NULL, st[i].st_size);
                /* close */
                close(fd[i]);
                fd[i] = -1;
                DEBUG_IFM("RM[%d] len = %d\n", i, count[i]);
            }
        } else if (type == NEW_RIMM_SET) {  /* New RIMM - sendfile */
            /* RM msg */
            // UUID
            memcpy(buf, ctx->conf->newrm_uuid->uuid, 16);  // TODO uuid change
            // NUM
            buf[16 + 0] = (num & 0xFF000000) >> 24;
            buf[16 + 1] = (num & 0xFF0000) >> 16;
            buf[16 + 2] = (num & 0xFF00) >> 8;
            buf[16 + 3] = (num & 0xFF);
            rc = write(sock, buf, 16 + 4);

            // TODO num == conf->rm_num
            for (i = 0; i< conf->newrm_num; i++) {
                // LEN1
                buf[0] = (fsize[i] & 0xFF000000) >> 24;
                buf[1] = (fsize[i] & 0xFF0000) >> 16;
                buf[2] = (fsize[i] & 0xFF00) >> 8;
                buf[3] = (fsize[i] & 0xFF);
                rc = write(sock, buf, 4);

                // BODY1
                fd[i] = open(ctx->conf->newrm_filename[i], O_RDONLY);
                count[i] = sendfile(sock, fd[i], NULL, st[i].st_size);
                /* close */
                close(fd[i]);
                fd[i] = -1;
                DEBUG_IFM("RM[%d] len = %d\n", i, count[i]);
            }
        } else if (type == INTEGRITY_REPORT) {  /* IR - sendfile */
            // BODY1
            fd[0] = open(ctx->conf->ir_filename, O_RDONLY);
            count[0] = sendfile(sock, fd[0], NULL, st[0].st_size);
            /* close */
            close(fd[0]);
            fd[0] = -1;
            DEBUG_IFM("IR len = %d\n", count[0]);
        /* CAP */
        } else if (type == OPENPTS_CAPABILITIES) {  // UUID
            /* versions */
            rc = write(sock, &ctx->conf->pts_flag, 4);
            rc = write(sock, &ctx->conf->tpm_version, 4);
            rc = write(sock, &ctx->conf->tss_version, 4);
            rc = write(sock, &ctx->conf->pts_version, 4);

            /* send Platform UUID, ctx->uuid */
            rc = write(sock, ctx->conf->uuid->uuid, 16);

            /* send RM UUID */
            if (ctx->conf->rm_uuid == NULL) {
                // TODO  verifier does not have Rm UUID. just send Verifier's UUID
                DEBUG("writePtsTlvToSock() RM uuid is NULL, => send platform UUID\n");
                rc = write(sock, ctx->conf->uuid->uuid, 16);
            } else if (ctx->conf->rm_uuid->uuid == NULL) {
                // TODO verifier?
                DEBUG("writePtsTlvToSock() RM uuid is NULL, => send platform UUID, file = %s\n",
                    ctx->conf->rm_uuid->filename);
                rc = write(sock, ctx->conf->uuid->uuid, 16);
            } else {
                rc = write(sock, ctx->conf->rm_uuid->uuid, 16);
            }
        } else if (type == DH_NONCE_PARAMETERS_REQUEST) {  /* ini -> res */
            /* DH Nonce */
            /* setup */
            buf[0] = ctx->nonce->req->reserved;
            buf[1] = ctx->nonce->req->min_nonce_len;
            buf[2] = (BYTE)(ctx->nonce->req->dh_group_set >> 8);
            buf[3] = (BYTE)(ctx->nonce->req->dh_group_set & 0xff);
            /* send */
            rc = write(sock, buf, 4);
        } else if (type == DH_NONCE_PARAMETORS_RESPONSE) { /* res -> ini */
            /* setup */
            buf[0] = ctx->nonce->res->reserved[0];
            buf[1] = ctx->nonce->res->reserved[1];
            buf[2] = ctx->nonce->res->reserved[2];
            buf[3] = ctx->nonce->res->nonce_length;
            buf[4] = (BYTE)(ctx->nonce->res->selected_dh_group >> 8);
            buf[5] = (BYTE)(ctx->nonce->res->selected_dh_group & 0xff);
            buf[6] = (BYTE)(ctx->nonce->res->hash_alg_set >> 8);
            buf[7] = (BYTE)(ctx->nonce->res->hash_alg_set & 0xff);
            /* send */
            rc = write(sock, buf, 8);

            /* send dh_respondor_nonce */
            // DEBUG("ctx->nonce->respondor_nonce_length = %d\n",ctx->nonce->respondor_nonce_length);
            // printHex("\t\trespondor nonce", ctx->nonce->respondor_nonce, ctx->nonce->respondor_nonce_length, "\n");
            rc = write(sock, ctx->nonce->respondor_nonce, ctx->nonce->respondor_nonce_length);

            /* send dh_respondor_public */
            rc = write(sock, ctx->nonce->pubkey, ctx->nonce->pubkey_length);
        } else if (type == DH_NONCE_FINISH) {  /* ini -> res */
            buf[0] = ctx->nonce->fin->reserved = 0;
            buf[1] = ctx->nonce->fin->nonce_length = ctx->nonce->initiator_nonce_length;
            buf[2] = (BYTE)(ctx->nonce->fin->selected_hash_alg >> 8);
            buf[3] = (BYTE)(ctx->nonce->fin->selected_hash_alg & 0xff);
            /* send */
            rc = write(sock, buf, 4);
            /* send dh_initiator_pubkey */
            rc = write(sock, ctx->nonce->pubkey, ctx->nonce->pubkey_length);
            /* send dh_initiator_nonce */
            rc = write(sock, ctx->nonce->initiator_nonce, ctx->nonce->initiator_nonce_length);
        }
#ifdef CONFIG_AIDE
        else if (type == AIDE_DATABASE) {  // 1 file
            // BODY1
            fd[0] = open(ctx->conf->aide_database_filename, O_RDONLY);
            count[0] = sendfile(sock, fd[0], NULL, st[0].st_size);
            /* close */
            close(fd[0]);
            fd[0] = -1;
            DEBUG_IFM("Send AIDE DB %s\n", ctx->conf->aide_database_filename);
            DEBUG_IFM("AIDE DATABASE len = %d\n", count[0]);
        }
#endif
        else if (type == TPM_PUBKEY) {  //
            rc = write(
                    sock,
                    ctx->conf->pubkey,
                    ctx->conf->pubkey_length);
        } else {
            /* normal msg */
            rc = write(sock, write_tlv->value, write_tlv->length);
        }
    }

    DEBUG_IFM("IF-M write, type=0x%08x, length=%d\n",
        write_tlv->type, write_tlv->length);
    DEBUG_CAL("writePtsTlvToSock - done\n");

    /* done */
    rc = write_tlv->length;

    /* close */
    for (i = 0; i < MAX_RM_NUM; i++) {
        if (fd[i] >= 0) close(fd[i]);
    }
    free(write_tlv);
    return rc;

  error:
    ERROR("writePtsTlvToSock()\n");
    /* close */
    for (i = 0; i < MAX_RM_NUM; i++) {
        if (fd[i] >= 0) close(fd[i]);
    }

    free(write_tlv);
    return -1;
}


/* TNC */

/**
 * get IF-M PTS message (TNC)
 * return *msg
 * TODO use RC core
 */
char* getPtsTlvMessage(OPENPTS_CONTEXT *ctx, int type, int *len) {
    PTS_IF_M_Attribute *write_tlv;
    int i;
    OPENPTS_CONFIG *conf;

    DEBUG("writePtsTlvToSock - start\n");

    /* check */
    if (ctx == NULL) {
        ERROR("ctx is NULL\n");
        return NULL;
    }
    conf = ctx->conf;
    if (conf == NULL) {
        ERROR("conf is NULL\n");
        return NULL;
    }

    /* TLV */
    *len = 0;


    switch (type) {
    /* Verifier -> Collector */
    case DH_NONCE_PARAMETERS_REQUEST:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case DH_NONCE_FINISH:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case REQUEST_RIMM_SET:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case REQUEST_INTEGRITY_REPORT:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case VERIFICATION_RESULT:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;

    /* Collector -> Verifier */
    case OPENPTS_CAPABILITIES:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case DH_NONCE_PARAMETORS_RESPONSE:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = type;
        write_tlv->length = 0;
        break;
    case RIMM_SET:
        {
            int rc;
            int vlen;
            int num = 2;  // TODO set RM num from conf?
            BYTE * buf;
            int ptr;

            int fsize[MAX_RM_NUM];
            int fd[MAX_RM_NUM];
            struct stat st[MAX_RM_NUM];


            DEBUG_IFM("create TEMPLATE_RIMM_SET_METADATA\n");

            /* body - RMs  */
            vlen = 4;
            for (i = 0; i< conf->rm_num; i++) {
                fd[i] = open(ctx->conf->rm_filename[i], O_RDONLY);
                if (fd[i] < 0) {
                    printf("Error RM file, '%s' not found\n", ctx->conf->rm_filename[i]);
                    return NULL;
                }
                stat(ctx->conf->rm_filename[i], &st[i]);
                fsize[i] = st[i].st_size;
                vlen += 4 + fsize[i];
            }

            DEBUG_IFM("size of TEMPLATE_RIMM_SET_METADATA is %d\n", vlen);

            /* malloc msg buffer */
            if ((buf = malloc(12 + vlen)) == NULL) {
                ERROR("no memory size=%d\n", 12 + vlen);
                return NULL;
            }

            write_tlv = (PTS_IF_M_Attribute *)buf;
            write_tlv->flags  = 0;
            write_tlv->vid[0] = 0;
            write_tlv->vid[1] = 0;
            write_tlv->vid[2] = 0;
            write_tlv->type   = type;
            write_tlv->length = vlen;

            /* payload */

            DEBUG_IFM("RIMMs %d\n", num);

            ptr = 12;
            buf[ptr + 0] = (num & 0xFF000000) >> 24;
            buf[ptr + 1] = (num & 0xFF0000) >> 16;
            buf[ptr + 2] = (num & 0xFF00) >> 8;
            buf[ptr + 3] = (num & 0xFF);
            ptr += 4;

            for (i = 0; i < conf->rm_num; i++) {
                DEBUG_IFM("RIMM %d size %d\n", i, fsize[i]);

                buf[ptr + 0] = (fsize[i] & 0xFF000000) >> 24;
                buf[ptr + 1] = (fsize[i] & 0xFF0000) >> 16;
                buf[ptr + 2] = (fsize[i] & 0xFF00) >> 8;
                buf[ptr + 3] = (fsize[i] & 0xFF);
                ptr += 4;

                // BODY1
                while (1) {
                    rc = read(fd[i], &buf[ptr], fsize[i]);
                    if (rc == 0) {
                         break;
                    }
                    ptr += rc;
                    fsize[i] -= rc;
                }
                close(fd[i]);
                fd[i] = -1;
            }
            DEBUG_IFM("create TEMPLATE_RIMM_SET_METADATA ... done\n");
        }
        break;
    case INTEGRITY_REPORT:
        {
            int rc;
            int vlen;
            BYTE * buf;
            int ptr;
            int fsize1 = 0;
            int fd1 = 0;
            struct stat st1;

            DEBUG_IFM("create INTEGRITY_REPORT\n");

            if (ctx->conf->iml_mode == 1) {
                /* get IML via securityfs */
                /* reset FSM */
                /* read BIOS IML */
                rc = getBiosImlFile(ctx, ctx->conf->bios_iml_filename, 1);  // TODO endian?
                if (rc != PTS_SUCCESS) {
                    ERROR("load IML %s was failed\n", ctx->conf->bios_iml_filename);
                }
                /* read Runtime IML */
                rc = getImaImlFile(
                        ctx,
                        ctx->conf->runtime_iml_filename,
                        ctx->conf->runtime_iml_type, 1);  // TODO endian?
                /* read PCRS */
                rc = getPcrBySysfsFile(ctx, ctx->conf->pcrs_filename);
                /* TPM Quote */
                TODO("add TPM_quote\n");
                /* save IR */
                rc = writeIr(ctx, ctx->conf->ir_filename);
            } else {
                TODO("get IML/PCR via TSS is not ready\n");
            }

            DEBUG_IFM("send INTEGRITY_REPORT\n");

            /* body - IR  */
            fd1 = open(ctx->conf->ir_filename, O_RDONLY);
            if (fd1 < 0) {
                printf("Error platform RM file, '%s' not found\n", ctx->conf->ir_filename);
                return NULL;
            }
            stat(ctx->conf->ir_filename, &st1);
            fsize1 = st1.st_size;

            vlen = 4 + fsize1;

            DEBUG_IFM("size of INTEGRITY_REPORT is %d\n", vlen);

            /* malloc msg buffer */
            if ((buf = malloc(12 + vlen)) == NULL) {
                ERROR("no memory size=%d\n", 12 + vlen);
                return NULL;
            }

            write_tlv = (PTS_IF_M_Attribute *)buf;
            write_tlv->flags  = 0;
            write_tlv->vid[0] = 0;
            write_tlv->vid[1] = 0;
            write_tlv->vid[2] = 0;
            write_tlv->type   = type;
            write_tlv->length = vlen;

            /* payload */
            ptr = 12;

            // LEN1
            DEBUG_IFM("IR size %d\n", fsize1);

            buf[ptr + 0] = (fsize1 & 0xFF000000) >> 24;
            buf[ptr + 1] = (fsize1 & 0xFF0000) >> 16;
            buf[ptr + 2] = (fsize1 & 0xFF00) >> 8;
            buf[ptr + 3] = (fsize1 & 0xFF);
            ptr += 4;

            // BODY1
            while (1) {
                rc = read(fd1, &buf[ptr], fsize1);
                if (rc == 0) {
                     break;
                }
                ptr += rc;
                fsize1 -= rc;
            }
            close(fd1);

            DEBUG_IFM("create INTEGRITY_REPORT ... done\n");
        }
        break;

    default:
        // BAT type
        ERROR("BAD TYPE %x\n", type);
        return NULL;
    }

    DEBUG_IFM("IF-M message, type=0x%x, length=%d\n",
        write_tlv->type, write_tlv->length);
    DEBUG("writePtsTlvToSock - done\n");

    if (write_tlv->length == 0) {
        /* just send TLV headder */
        *len = 12;
        return (char*)  write_tlv;
    } else if (write_tlv->length > 0) {
        *len = 12 + write_tlv->length;
        return (char*)  write_tlv;
    } else {
        ERROR("internal error\n");
        return NULL;
    }
}
