/*
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2007,2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/iml2text.c
 * \brief Convert binary IML file to plaintext
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-08-25
 * cleanup 2011-01-22 SM
 *
 * show eventlog (get though TSS)
 *
 *   ./src/iml2text
 *
 * show eventlog (binary eventlog file)
 *
 *   ./src/iml2text -i tests/data/ThinkpadX200_Fedora12/binary_bios_measurements
 *
 * show BE(big-endian) event on LE host
 *
 *   ./src/iml2text -E -i tests/data/XXX/example_event_log
 *
 */

/*
 * References:
 * [1] BIOS https://www.trustedcomputinggroup.org/specs/PCClient/
 *     Ref PC Spec v1.2, p74
 *       UINT32   pcrIndex
 *       UINT32   eventType
 *       BYTE[20] digest
 *       UINT32   eventDataSize
 *       BYTE[]   event
 *
 * [2] GRUB-IMA, see BIOS format
 * [3] Linux-IMA
 * [4] LIM/IMA
 *
 *     boot aggregate = sha1(PCR[0],PCR[1],,,PCR[7])
 *
 *     /sys/kernel/security/ima/binary_runtime_measurements
 *          UINT32 pcr
 *          BYTE   template_hash[20]
 *          UNIT32 name_len
 *          BYTE   name[name_len]
 *          BYTE   digest[20]
 *          UNIT32 filename_len
 *          CHAR   filename[filename_len]
 *
 *       ima_data
 *          BYTE   digest[20]
 *          CHAR   filename[filename_len]
 *
 *       template_hash = SHA1(ima_data) = SHA1(digest + filename)
 *
 *     Through TSS (TBD)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdint.h>


#ifndef __STDC_FORMAT_MACROS
#define __STDC_FORMAT_MACROS
#include <inttypes.h>
#endif


//#include <tss/platform.h>
//#include <tss/tss_defines.h>
//#include <tss/tss_typedef.h>
//#include <tss/tss_structs.h>
//#include <tss/tss_error.h>
//#include <tss/tspi.h>

#include <openssl/sha.h>

#include <openpts.h>

// Local TCSD
#define SERVER    NULL


// PCR
BYTE pcr[24][20];

int verbose = 0;

/*

 TPM PCR Read

 Usage:
  tpm_pcrread -p index

 */

int hex2bin(void *dest, const void *src, size_t n);
void printhex(char *str, unsigned char *buf, int len);

#if 0
struct biosEvent {
    UINT32   pcrIndex;
    UINT32   eventType;
    BYTE     digest[20];
    UINT32   eventDataSize;
    // BYTE[]   event
};
#endif


#define BUFSIZE 256
// State
int pcr4_grub = 0;
int pcr5_grub = 0;

#define IMA32 32
int ima_mode = 0;

/* EFI */
typedef UINT64 EFI_PHISICAL_ADDRESS;
typedef UINT64 UINTN;
typedef wchar_t CHAR16;
typedef BYTE UINT8;
typedef BYTE INT8;
typedef void VOID;
typedef BYTE EFI_DEVICE_PATH;  // TODO
typedef struct {
    UINT32 Data1;
    UINT16 Data2;
    UINT16 Data3;
    UINT8  Data4[8];
} EFI_GUID;


typedef struct tdEFI_PLATFORM_FIRMWARE_BLOB {
    EFI_PHISICAL_ADDRESS    BlobBase;
    UINTN                   BlobLength;
} EFI_PLATFORM_FIRMWARE_BLOB;

typedef struct tdEFI_IMAGE_LOAD_EVENT {
    EFI_PHISICAL_ADDRESS    ImageLocationInMemory;
    UINTN                   ImageLengthInMemory;
    UINTN                   ImageLinkTimeAddress;
    UINTN                   LengthOfDevicePath;
    EFI_DEVICE_PATH         DevicePath[1];
} EFI_IMAGE_LOAD_EVENT;


typedef struct tdEFI_CONFIGULATION_TABLE {
    EFI_GUID VendorGuid;
    VOID    *VentorTable;
} EFI_CONFIGULATION_TABLE;

typedef struct tdEFI_HANDOFF_TABLE_POINTERS {
    UINTN                   NumberOfTables;
    EFI_CONFIGULATION_TABLE TableEntry[1];
} EFI_HANDOFF_TABLE_POINTERS;


typedef struct tdEFI_VARIABLE_DATA {
    EFI_GUID    ValiableName;
    UINTN       UnicodeNameLength;
    UINTN       VariableDataLength;
    CHAR16      UnicodeName[1];
    INT8        VariableData[1];
} EFI_VARIABLE_DATA;


void fprintGuid(FILE *fp, EFI_GUID guid) {
    fprintf(fp, "GUID=%08x-", guid.Data1);
    fprintf(fp, "%04x-", guid.Data2);
    fprintf(fp, "%04x-", guid.Data3);
    fprintf(fp, "%02x-", guid.Data4[0]);
    fprintf(fp, "%02x-", guid.Data4[1]);
    fprintf(fp, "%02x-", guid.Data4[2]);
    fprintf(fp, "%02x-", guid.Data4[3]);
    fprintf(fp, "%02x-", guid.Data4[4]);
    fprintf(fp, "%02x-", guid.Data4[5]);
    fprintf(fp, "%02x-", guid.Data4[6]);
    fprintf(fp, "%02x",  guid.Data4[7]);
}

void fprintBin(FILE* fp, BYTE* data, UINT32 len) {
    int i;
    for (i = 0; i < (int)len; i++) {
        fputc(data[i], fp);
    }
}

void fprintHex(FILE* fp, BYTE* data, UINT32 len) {
    int i;
    for (i = 0; i < (int)len; i++) {
        fprintf(fp, "%02x", data[i]);
    }
}

// 61dfe48bca93d211aa0d00e098032b8c 0900000000000000 1a00000000000000
// 4200 6f00 6f00 7400 4f00 7200 6400 6500 7200 00000100020003000400050006000700080009000a000b000c00]
void fprintUnicode(FILE *fp, wchar_t * name, int len) {
    int i;
    char *u8;
    u8 = (char *) name;
    for (i = 0; i< len*2;i+=2) {
        fprintf(fp, "%c", u8[i]);
    }
}

/**
 *
 *
 * TODO(munetoh) move to iml.c and be common 
 */
void fprintEventData(
        FILE* fp,
        BYTE* data,  // event
        UINT32 len,  // event length
        UINT32 pcrindex,
        UINT32 type) {
    char buf[BUFSIZE];
    char b64buf[BUFSIZE];
    int i;

    if (len < BUFSIZE) {
        memcpy(buf, data, len);
        buf[len]=0;  // terminate
    } else {
        memcpy(buf, data, BUFSIZE);
        buf[BUFSIZE-1] = 0;  // terminate
    }


    if (pcrindex == 10) {  // Linux-IMA
        if (type == 2) {
            fprintf(fp, "[IMA-LKM:%s] ", buf);
        } else if (type == 1) {
            fprintf(fp, "[IMA-EXE:%s] ", buf);
        } else if (type == 0) {
            // fprintf(fp, "[IMA:%s] ", buf);
            if (ima_mode == 32) {
/*
RHEL6 - Kernel 2.6.32
--------------------------------------------------------------------------------
0a 00 00 00
98 0a 38 ef 63 42 a5 d6 37 cf 96 47 b5 34 45 ac 13 98 d5 c7
03 00 00 00
69 6d 61 
0c d0 a1 73 28 b3 e0 93 a0 51 15 c1 44 23 eb 62 45 df 3b 32 
0e 00 00 00 
62 6f 6f 74 5f 61 67 67 72 65 67 61 74 65 | boot_aggregate|
--------------------------------------------------------------------------------
0a 00 00 00
c8 7f 4d ea 27 e3 3e 3c 6b 88 71 d7 fb bf ed e2 0f f1 78 7a 
03 00 00 00
69 6d 61                                  |.ima
ac 63 ec 16  2b 60 31 3a 88 96 e4 1a 57 0c 64 da bf 3b 16 ec
05 00 00 00 
2f 69 6e 69 74                              |./init|
--------------------------------------------------------------------------------

IML->TSS(Fix the format)->
EventData
  BYTE[20] digest
  BYTE[len-20] filename
*/
                fprintf(fp, "[IMA:sha1(");
                fprintBin(fp, &data[20], len-20);
                fprintf(fp, ")=");
                fprintHex(fp, data, 20);
                fprintf(fp, "] ");

            } else {
                fprintf(fp, "[IMA:(TBD)] ");
            }
        } else if ((type & 0xFFFF) == 4) {
            fprintf(fp, "[IMA-USR,0x%04x:%s] ", (type >> 16), buf);
        } else {
            fprintf(fp, "[???:%s] ", buf);
        }
    } else if (pcrindex <= 8) {  // BIOS + Grub
        switch (type) {
        case 0:
            fprintf(fp, "[BIOS:EV_PREBOOT_CERT(EV_CODE_CERT)]");
            break;
        case 1:
            fprintf(fp, "[BIOS:EV_POST_CODE(EV_CODE_NOCERT)]");
            break;
        case 2:
            fprintf(fp, "[BIOS:EV_UNUSED(EV_XML_CONFIG)]");
            break;
        case 3:
            fprintf(fp, "[BIOS:EV_NO_ACTION]");
            break;
        case 4:
            if ((pcr4_grub > 1) && (pcrindex == 4)) {
                fprintf(fp, "[GRUB:EV_SEPARATOR, %s]", buf);
            } else if ((pcr5_grub > 0) && (pcrindex == 5)) {
                fprintf(fp, "[GRUB:EV_SEPARATOR, %s]", buf);
            } else if (pcrindex == 8) {
                fprintf(fp, "[GRUB:EV_SEPARATOR, %s]", buf);
            } else if (len == 4) {  // V1.2
                fprintf(fp, "[BIOS:EV_SEPARATOR, %02x%02x%02x%02x]",
                        (unsigned char) buf[0],
                        (unsigned char) buf[1],
                        (unsigned char) buf[2],
                        (unsigned char) buf[3]);
            } else {
                fprintf(fp, "[BIOS:EV_SEPARATOR, %s]", buf);
            }
            break;
        case 5:
            if ((pcr5_grub > 0) && (pcrindex == 5)) {
                fprintf(fp, "[GRUB:EV_ACTION, %s]", buf);
            } else {
                fprintf(fp, "[BIOS:EV_ACTION, %s]", buf);
            }
            break;
        case 6:
            if ((pcr4_grub > 1) && (pcrindex == 4)) {
                fprintf(fp, "[GRUB: measure MBR again]");
            } else {
                fprintf(fp, "[BIOS:EV_EVENT_TAG(EV_PLATFORM_SPECIFIC)]");
            }
            break;
        case 7:
            fprintf(fp, "[BIOS:EV_S_CRTM_CONTENTS]");
            break;
        case 8:
            fprintf(fp, "[BIOS:EV_S_CRTM_VERSION]");
            break;
        case 9:
            fprintf(fp, "[BIOS:EV_CPU_MICROCODE]");
            break;
        case 0x0a:
            fprintf(fp, "[BIOS:EV_PLATFORM_CONFIG_FLAG)]");
            break;
        case 0x0b:
            fprintf(fp, "[BIOS:EV_TABLE_OF_CONTENTS)]");
            break;
        case 0x0c:
            fprintf(fp, "[BIOS:EV_COMPACT_HASH]");
            break;
        case 0x0d:
            if (pcr4_grub == 0) {
                // BIOS
                fprintf(fp, "[BIOS:EV_IPL]");
                pcr4_grub = 1;
            } else if (pcr4_grub == 1) {
                // GRUB
                fprintf(fp, "[GRUB:EV_IPL, Stage1(MBR)]");
                pcr4_grub = 2;
            } else if (pcr4_grub == 2) {
                // GRUB
                fprintf(fp, "[GRUB:EV_IPL, Stage1.5]");
                pcr4_grub = 3;
            } else if (pcr4_grub == 3) {
                // GRUB
                fprintf(fp, "[GRUB:EV_IPL, Stage1.5(filesystem)]");
                pcr4_grub = 4;
            } else {
                // GRUB
                fprintf(fp, "[GRUB:EV_IPL]");
            }
            break;
        case 0x0e:
            if (pcr5_grub == 0) {
                fprintf(fp, "[BIOS:EV_IPL_PERTITION_DATA]");
                pcr5_grub = 1;
            } else {
                fprintf(fp, "[GRUB:grub.conf]");
            }
            break;
        case 0x0f:
            fprintf(fp, "[BIOS:EV_NOHOST_CODE)]");
            break;
        case 0x10:
            fprintf(fp, "[BIOS:EV_NOHOST_CONFIG]");
            break;
        case 0x11:
            fprintf(fp, "[BIOS:EV_NOHOST_INFO]");
            break;
        case 0x12:
            fprintf(fp, "[BIOS:EV_SPECIFICATION_IDENTIFIER 0x");
            for (i = 0; i < (int)len; i++) {
                fprintf(fp, "%02x", (BYTE)buf[i]);
            }
            fprintf(fp, "]");
            break;

        case 0x80000001:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_VARIABLE_DRIVER_CONFIG,");
            {
                EFI_VARIABLE_DATA *d = (EFI_VARIABLE_DATA *) buf;
                fprintGuid(fp, d->ValiableName);
                fprintf(fp, ",Name=");
                fprintUnicode(fp, d->UnicodeName, d->UnicodeNameLength);
                fprintf(fp, ",Valiable[0x%" PRIx64 "]", (UINT64)d->VariableDataLength);
            }
            fprintf(fp, "]");
            break;
        case 0x80000002:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_VARIABLE_BOOT,");
            {
                EFI_VARIABLE_DATA *d = (EFI_VARIABLE_DATA *) buf;
                fprintGuid(fp, d->ValiableName);
                fprintf(fp, ",Name=");
                fprintUnicode(fp, d->UnicodeName, d->UnicodeNameLength);
                fprintf(fp, ",Valiable[0x%" PRIx64 "]", (UINT64)d->VariableDataLength);
            }
            fprintf(fp, "]");
            break;
        case 0x80000003:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_BOOT_SERVICES_APPLICATION,");
            {
                EFI_IMAGE_LOAD_EVENT *e = (EFI_IMAGE_LOAD_EVENT *) buf;
                fprintf(fp, "base=0x%" PRIx64 ",", (UINT64)e->ImageLocationInMemory);
                fprintf(fp, "len=0x%" PRIx64 ",", (UINT64)e->ImageLengthInMemory);
                fprintf(fp, "len=0x%" PRIx64 ",", (UINT64)e->ImageLinkTimeAddress);
                fprintf(fp, "len=0x%" PRIx64 "]", (UINT64)e->LengthOfDevicePath);
            }
            break;
        case 0x80000004:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_BOOT_SERVICES_DRIVER,");
            {
                EFI_IMAGE_LOAD_EVENT *e = (EFI_IMAGE_LOAD_EVENT *) buf;
                fprintf(fp, "base=0x%" PRIx64 ",", (UINT64)e->ImageLocationInMemory);
                fprintf(fp, "len=0x%" PRIx64 ",", (UINT64)e->ImageLengthInMemory);
                fprintf(fp, "len=0x%" PRIx64 ",", (UINT64)e->ImageLinkTimeAddress);
                fprintf(fp, "len=0x%" PRIx64 "]", (UINT64)e->LengthOfDevicePath);
            }
            break;
        case 0x80000005:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_RUNTIME_SERVICES_DRIVER,");
            {
                EFI_IMAGE_LOAD_EVENT *e = (EFI_IMAGE_LOAD_EVENT *) buf;
                fprintf(fp, "base=0x%" PRIx64 ",", (UINT64)e->ImageLocationInMemory);
                fprintf(fp, "len=0x%" PRIx64 ",", (UINT64)e->ImageLengthInMemory);
                fprintf(fp, "len=0x%" PRIx64 ",", (UINT64)e->ImageLinkTimeAddress);
                fprintf(fp, "len=0x%" PRIx64 "]", (UINT64)e->LengthOfDevicePath);
            }
            break;
        case 0x80000006:  // EFI TODO
            fprintf(fp, "[BIOS:EV_EFI_GPT_EVENT len=%d,", len);
            for (i = 0; i < (int)len; i++) {
                fprintf(fp, "%02x", (BYTE)buf[i]);
            }
            fprintf(fp, "]");
            break;
        case 0x80000007:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_ACTION, %s]", buf);
            break;
        case 0x80000008:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_PLATFORM_FIRMWARE_BLOB,");
            {
                EFI_PLATFORM_FIRMWARE_BLOB *blob = (EFI_PLATFORM_FIRMWARE_BLOB *)buf;
                fprintf(fp, "base=0x%" PRIx64 ",", (UINT64)blob->BlobBase);
                fprintf(fp, "len=0x%" PRIx64 "]", (UINT64)blob->BlobLength);
            }
            break;
        case 0x80000009:  // EFI
            fprintf(fp, "[BIOS:EV_EFI_HANDOFF_TABLE,");
            {
                EFI_HANDOFF_TABLE_POINTERS *p = (EFI_HANDOFF_TABLE_POINTERS *)buf;

                fprintf(fp, "num=0x%" PRIx64 ",", (UINT64)p->NumberOfTables);
                if (p->NumberOfTables > 0) {
                    EFI_CONFIGULATION_TABLE *t = &p->TableEntry[0];
                    fprintGuid(fp, t->VendorGuid);
                    fprintf(fp, "]");
                }
            }
            // 0100000000000000 312d9deb882dd3119a160090273fc14d 00a06b7f 00000000
            break;

            // GRUB
        case 0x1005:
            fprintf(fp, "[GRUB:ACTION, %s]", buf);
            break;
        case 0x1105:
            fprintf(fp, "[GRUB:KERNEL_OPT %s]", buf);
            break;
        case 0x1205:
            fprintf(fp, "[GRUB:KERNEL %s]", buf);
            break;
        case 0x1305:
            fprintf(fp, "[GRUB:INITRD %s]", buf);
            break;
        case 0x1405:
            fprintf(fp, "[GRUB:MODULE %s]", buf);
            break;

        default:
            fprintf(fp, "[Unknown BIOS Event:size=%d] ", len);
            break;
        }
    } else {
        switch (type) {
            /* OpenPTS*/
        case EV_COLLECTOR_START:
            fprintf(fp, "[OpenPTS:EV_COLLECTOR_START[%d]", len);
            if (verbose) {
                fprintf(fp, "=0x");
                for (i = 0; i < (int)len; i++) {
                    fprintf(fp, "%02x", (BYTE)buf[i]);
                }
                encodeBase64(
                    (unsigned char *)b64buf,
                    (unsigned char *)buf, len);
                fprintf(fp, ", base64(%s)", b64buf);

            }
            fprintf(fp, "]");

            break;
        case EV_FILE_SCAN:
            fprintf(fp, "[OpenPTS:EV_FILE_SCAN]");
            break;
        default:
            if ((verbose) && (len < 64)) {
                fprintf(fp, "[Unknown Event[%d]=0x", len);
                for (i = 0; i < (int)len; i++) {
                    fprintf(fp, "%02x", (BYTE)buf[i]);
                }
                encodeBase64(
                    (unsigned char *)b64buf,
                    (unsigned char *)buf, len);
                fprintf(fp, ", base64(%s)", b64buf);
                fprintf(fp, "]");
            } else {
                fprintf(fp, "[Unknown Event:size=%d] ", len);
            }
            break;
        }
    }
}



// Verify
void extend(int index, BYTE* digest) {
    SHA_CTX ctx;

    SHA1_Init(&ctx);
    SHA1_Update(&ctx, &pcr[index][0], 20);
    SHA1_Update(&ctx, digest, 20);
    SHA1_Final(&pcr[index][0], &ctx);
}


void ima_boot_aggregate(BYTE* digest) {
    SHA_CTX ctx;
    BYTE buf1[20];
    BYTE buf2[256];  // note) this is FIXED size
    int i;
    char *filename = "boot_aggregate";

    SHA1_Init(&ctx);
    for (i = 0; i < 8; i++) {
        SHA1_Update(&ctx, &pcr[i][0], 20);
    }
    SHA1_Final(buf1, &ctx);

    memset(buf2, 0, 256);
    memcpy(buf2, filename, 14);

    // template
    SHA1_Init(&ctx);
    SHA1_Update(&ctx, buf1, 20);
    SHA1_Update(&ctx, buf2, 256);
    SHA1_Final(digest, &ctx);
}

#if 0
// move to misc.c
UINT32 b2l(UINT32 in) {
    UINT32 out;

    out = in & 0xff;
    in = in >> 8;
    out = out << 8;
    out += in & 0xff;
    in = in >> 8;
    out = out << 8;
    out += in & 0xff;
    in = in >> 8;
    out = out << 8;
    out += in & 0xff;

    return out;
}
#endif

/**
 * get IML from file (TCG binary format)
 *
 * Step 1. read event -> EVENT_WRAPPER chain
 * Step 2. EVENT_WRAPPER -> TSS_PCR_EVENT
 * Step 3. free  EVENT_WRAPPER
 *
 */
TSS_RESULT getEventLog(char *filename, int endian, int aligned, UINT32 *event_num, TSS_PCR_EVENT **pcr_events) {
    int rc = 0;
    int i = 0;
    size_t size;
    FILE *fp;
    UINT32 pcrIndex;
    UINT32 eventType;
    UINT32 eventLength;
    UINT32 aligned_length;

    /**/
    TSS_PCR_EVENT *event = NULL;
    OPENPTS_PCR_EVENT_WRAPPER *ew = NULL;
    OPENPTS_PCR_EVENT_WRAPPER *ew_start = NULL;
    OPENPTS_PCR_EVENT_WRAPPER *ew_last = NULL;

    DEBUG("getEventLog() - %s, ensian = %d, aligned = %d\n", filename, endian, aligned);

    /* check */
    if (filename == NULL) {
        ERROR("filename is NULL\n");  // TODO(munetoh)
        return TSS_E_INTERNAL_ERROR;
    }

    /* open file */
    if ((fp = fopen(filename, "rb")) == NULL) {
        ERROR("%s missing", filename);
        return TSS_E_INTERNAL_ERROR;
    }

    /* Read IML, add to Snapshot */
    while (1) {
        DEBUG("--- event %d ------------\n", i);
        /* PCR index */
        size = fread(&pcrIndex, 1, 4, fp);
        if (size != 4)
            break;

        /* Event type */
        size = fread(&eventType, 1, 4, fp);
        if (size != 4)
            break;

        /* alloc new event */
        event = (TSS_PCR_EVENT *) malloc(sizeof(TSS_PCR_EVENT));   // TODO memory leak
        if (event == NULL) {
            printf("no memory\n");
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }
        memset((void*)event, 0, sizeof(TSS_PCR_EVENT));

        // event->versionInfo = 0;  // TODO(munetoh)
        if (endian == 0) {
            event->ulPcrIndex = pcrIndex;
            event->eventType = eventType;
        } else {
            /* Big endian */
            event->ulPcrIndex = b2l(pcrIndex);
            event->eventType = b2l(eventType);
        }

        DEBUG("\tpcr index = 0x%x\n", event->ulPcrIndex);
        DEBUG("\tevent type = 0x%x\n", event->eventType);

        // DEBUG("%x %x\n", event->ulPcrIndex, event->eventType);


        /* Digest */
        event->ulPcrValueLength = SHA1_DIGEST_SIZE;
        if ((event->rgbPcrValue = (BYTE *) malloc(SHA1_DIGEST_SIZE)) == NULL) {
            ERROR("no memory");
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }

        size = fread(event->rgbPcrValue, 1, SHA1_DIGEST_SIZE, fp);
        if (size != SHA1_DIGEST_SIZE) {  // TODO(munetoh) SHA1 only
            ERROR("SHA1 only");
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }

        if (verbose == DEBUG_FLAG) {
            DEBUG("digest");
            printHex("\t\t\t\t", event->rgbPcrValue, 20, "\n");
        }

        /* EventData len */
        size = fread(&eventLength, 1, 4, fp);
        if (size != 4) {
            ERROR("fread NG\n");
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }

        if (endian == 0) {
            event->ulEventLength = eventLength;
        } else {
            event->ulEventLength = b2l(eventLength);
        }

        /* adjust read data length */
        aligned_length = event->ulEventLength;
        if (aligned == 4) {
            if ((event->ulEventLength & 0x03) != 0) {
                aligned_length = (event->ulEventLength & 0xFFFFFFFC) + 0x04;
            }
        }

        DEBUG("\tevent size = 0x%x (%d)\n", event->ulEventLength, event->ulEventLength);

        /* EventData  */
        if ((event->rgbEvent = malloc(aligned_length)) == NULL) {
            ERROR("no memory");
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }
        // DEBUG("\n");
        size = fread(event->rgbEvent, 1, aligned_length, fp);
        if (size != aligned_length) {
            ERROR("fread NG, size = %d != %d \n", (unsigned int) size, (unsigned int) event->ulEventLength);
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }

        if (verbose == DEBUG_FLAG) {
            DEBUG("");
            printHex("\tevent data", event->rgbEvent, event->ulEventLength, "\n");
        }


        // DEBUG("\n");
        /* create wrapper */
        ew = (OPENPTS_PCR_EVENT_WRAPPER *)malloc(sizeof(OPENPTS_PCR_EVENT_WRAPPER));
        if (ew == NULL) {
            ERROR("no memory\n");
            rc =  TSS_E_INTERNAL_ERROR;
            goto close;
        }
        memset(ew, 0, sizeof(OPENPTS_PCR_EVENT_WRAPPER));

        // DEBUG("\n");
        ew->event = event;
        event = NULL;

        if (i == 0) {
            /* 1st EW */
            ew_start = ew;
            ew_last = ew;
        } else {
            ew_last->next_all = ew;
            ew_last = ew;
        }
        ew = NULL;

        // DEBUG("Next");
        i++;
    }



    *event_num = i;

    /* generate TSS_PCR_EVENT** */

    /* maloc */
    *pcr_events = calloc(*event_num, sizeof(TSS_PCR_EVENT));
    if ((*pcr_events) == NULL) {
        ERROR("no memory\n");
        rc = TSS_E_INTERNAL_ERROR;
        goto close;
    }

    ew = ew_start;
    for (i = 0; i < (int)(*event_num); i++) {
        event = ew->event;
        memcpy(&((*pcr_events)[i]), event, sizeof(TSS_PCR_EVENT));

        /* free */
        // if (event->rgbEvent != NULL) free(event->rgbEvent);
        // if (event->rgbPcrValue != NULL) free(event->rgbPcrValue);
        // free(event);

        ew_last = ew;
        ew = ew->next_all;
        free(ew_last);
    }
    ew = NULL;
    event = NULL;

    // TODO free ew

    rc = TSS_SUCCESS;

  close:
    if (fclose(fp) == EOF) {
        // TODO(munetoh) SYSLOG ERROR?
        rc = TSS_E_INTERNAL_ERROR;
    }

// free:
    // TODO(munetoh) we needs free all. but ok at this time:-P
#if 0
    if (event != NULL) {
        if (event->rgbPcrValue != NULL) {
            free(event->rgbPcrValue);
        }
        if (event->rgbEvent != NULL) {
            free(event->rgbEvent);
        }

        free(event);  // last one
    }
    if (ew  != NULL) {
        free(ew);  // TODO memory leak
    }
#endif
    return rc;  // TSS_E_INTERNAL_ERROR;
}

/**
 * Usage
 */
void usage(void) {
    fprintf(stderr, "OpenPTS command\n\n");
    fprintf(stderr, "Usage: iml2text [options]\n\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "  -i filename           Set binary eventlog file (at securityfs)\n");
    fprintf(stderr, "  -p pcr_index          Select pcr (TSS)\n");
    fprintf(stderr, "  -I mode               Select IMA's log format (Kernel 2.6.32:32)\n");
    fprintf(stderr, "  -E                    Enable endian conversion (BE->LE or LE->BE)\n");
    fprintf(stderr, "  -h                    Show this help message\n");
    fprintf(stderr, "\n");
}

/**
 * main
 */
int main(int argc, char *argv[]) {
    TSS_RESULT result = 0;
    TSS_HCONTEXT hContext;

    TSS_HTPM hTPM;

    UINT32 ulEventNumber = 0;
    TSS_PCR_EVENT * PcrEvents;
    BYTE digest[20];
    BYTE zero[20];
    BYTE fox[20];
    BYTE boot_aggregate[20];
    int i, j;
    char *outfilename = NULL;
    int firmware = 1;
    int kernel = 1;
    int pcrindex = -1;
    int verify = 0;
    int gmode = 0;  // 0:TSS,1,file
    char *filename = NULL;
    FILE *fp = stdout;
    int c;
    int endian = 0;  // 0:normal 1:convert
    int aligned = 0;

    //
    memset(zero, 0, 20);
    memset(fox, 0xff, 20);
    memset(boot_aggregate, 0xff, 20);
    BYTE *blob;
    UINT32 blobLength;


    /* Args */
    while ((c = getopt(argc, argv, "i:p:I:EAvh")) != EOF) {
        switch (c) {
        case 'i':
            filename = optarg;
            gmode = 1;  // file
            // DEBUG("%s\n", filename);
            break;
        case 'p':
            pcrindex = atoi(optarg);
            break;
        case 'I':
            ima_mode = atoi(optarg);
            break;
        case 'E':  /* Enable Endian Conversion */
            DEBUG("enable endian conversion\n");
            endian = 1;
            break;
        case 'A':  /*  four byte aligned event data */
            aligned = 4;
            break;
        case 'v':  /* DEBUG */
            verbose = DEBUG_FLAG;
            break;
        case 'h':
            usage();
            return 0;
        default:
            fprintf(stderr, "bad option '%c'\n", c);
            usage();
            return -1;
        }
    }
    argc -= optind;
    argv += optind;



    if (gmode == 0) {
        /* TSS */
        /* Connect to TCSD */

        result = Tspi_Context_Create(&hContext);
        if (result != TSS_SUCCESS) {
            printf("ERROR: Tspi_Context_Create failed rc=0x%x\n",
                   result);
            goto close;
        }

        result = Tspi_Context_Connect(hContext, SERVER);
        if (result != TSS_SUCCESS) {
            printf("ERROR: Tspi_Context_Connect failed rc=0x%x\n",
                   result);
            goto close;
        }


        /* Get TPM handles */
        result = Tspi_Context_GetTpmObject(hContext, &hTPM);
        if (result != TSS_SUCCESS) {
            printf("ERROR: Tspi_Context_GetTpmObject failed rc=0x%x\n",
                   result);
            goto close;
        }

        /* Get Log */

        result = Tspi_TPM_GetEventLog(
                    hTPM,
                    &ulEventNumber,
                    &PcrEvents);
        if (result != TSS_SUCCESS) {  // ERROR
            printf("ERROR: Tspi_TPM_GetEventLog failed rc=0x%x\n",
                   result);
            goto close;
        }
    } else {
        // FILE
        result = getEventLog(filename, endian, aligned, &ulEventNumber, &PcrEvents);
        if (result != TSS_SUCCESS) {  // ERROR
            ERROR("getEventLog failed rc=0x%x\n",
                   result);
            goto close;
        }
    }


    /* print all */
    if (outfilename != NULL) {
        if ((fp = fopen(outfilename, "w")) == NULL) {
            ERROR("File open failed, %s \n",
                   outfilename);
            result = -1;  // TODO
            goto free;
        }
    }

    fprintf(fp, " Idx PCR       Type    Digest                                EventData\n");
    fprintf(fp, "-----------------------------------------------------------------------\n");
    for (i = 0; i < (int)ulEventNumber; i++) {
        if ((firmware == 0) && (PcrEvents[i].ulPcrIndex <= 8)) {
            // SKIP
        } else if ((kernel == 0) && (PcrEvents[i].ulPcrIndex == 10)) {
            // SKIP
        } else {
            if ((pcrindex < 0) || (pcrindex == (int)PcrEvents[i].ulPcrIndex)) {
                fprintf(fp, "%4d ", i);
                fprintf(fp, "%3d ", PcrEvents[i].ulPcrIndex);
                fprintf(fp, "0x%08x ", PcrEvents[i].eventType);
                for (j = 0; j < (int)PcrEvents[i].ulPcrValueLength; j++)
                    fprintf(fp, "%02x", PcrEvents[i].rgbPcrValue[j]);
                fprintf(fp, " ");

                /* event Data */
                fprintEventData(
                    fp,
                    PcrEvents[i].rgbEvent,
                    PcrEvents[i].ulEventLength,
                    PcrEvents[i].ulPcrIndex,
                    PcrEvents[i].eventType);

                fprintf(fp, "\n");

                if (verify) {
                    if (PcrEvents[i].ulPcrIndex == 10) {  // IMA log
                        if (memcmp(PcrEvents[i].rgbPcrValue, zero, 20) == 0) {  // zero
                            extend(PcrEvents[i].ulPcrIndex, fox);
                        } else {
                            extend(PcrEvents[i].ulPcrIndex, PcrEvents[i].rgbPcrValue);
                        }
                        // TODO get boot aggregate
                        if (memcmp(PcrEvents[i].rgbEvent, "boot_aggregate", 14) == 0) {
                            memcpy(boot_aggregate, PcrEvents[i].rgbPcrValue, 20);
                        }

                    } else {
                        extend(PcrEvents[i].ulPcrIndex, PcrEvents[i].rgbPcrValue);
                    }
                }
            }
        }
    }
    if (verify) {
        fprintf(fp, "\n");
        fprintf(fp, "Verify IML :-)\n");
        fprintf(fp, "\n");
        fprintf(fp, "\tcalculated pcr values\t\t\t\tactual pcr values\n");
        fprintf(fp, "------------------------------------------------------------------------------------------------------\n");
        for (i = 0; i < 24; i++) {
            fprintf(fp, "pcr.%d=\t", i);
            // my calc
            for (j = 0; j < 20; j++) {
                fprintf(fp, "%02x", pcr[i][j]);
            }
            fprintf(fp, "\t");
            // actual
            Tspi_TPM_PcrRead(hTPM, i, &blobLength, &blob);

            for (j = 0; j < 20; j++) {
                fprintf(fp, "%02x", blob[j]);
            }
            fprintf(fp, "\n");
        }
        fprintf(fp, "\n");
        fprintf(fp, "\n");
        ima_boot_aggregate(digest);
        fprintf(fp, "IMA boot aggregate:\n");
        fprintf(fp, "\tcalculated boot_aggregate\t\t\t\tactual boot_aggregate\n");
        fprintf(fp, "------------------------------------------------------------------------------------------------------\n");
        for (j = 0; j < 20; j++) {
            fprintf(fp, "%02x", digest[j]);
        }
        fprintf(fp, "\t");
        for (j = 0; j < 20; j++) {
            fprintf(fp, "%02x", boot_aggregate[j]);
        }
        fprintf(fp, "\n");
    }

  free:
    if (gmode == 0) {
        result = Tspi_Context_FreeMemory(hContext, (BYTE *)PcrEvents);
        Tspi_Context_FreeMemory(hContext, NULL);
    } else {
    }
    /* Close TSS/TPM */

  close:
    if (gmode == 0) {
        Tspi_Context_Close(hContext);
    }

    return result;
}
