/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/uuid.c
 * \brief UUID wrapper (Generic part, OPENPTS_UUID)
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-11-29
 * cleanup 2011-01-20,21 SM
 *
 * Linux uses libuuid
 *
 * Program  UUID   Description        When            
 * ---------------------------------------------------------------------------------------------------
 * ptscd    CID    Colelctor ID       System install     => /var/lib/openpts/uuid (UUID of sign key)
 *          RM     RM ID              RM Gen xid in RM,  path /var/lib/openpts/$UUID/rm_files
 *          RunID  ID of this daemon  Daemon start       => /var/lib/openpts/run_uuid
 * openpts  VID    Verifier ID        1st run            => uuid=XXX in 'HOME/.openpts/openpts.conf' file
 *          RM                                           => 'HOME/.openpts/hostname/$UUID/rm_files
 *
 * Unit Test: check_uuid.c
 *
 */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>

// DIR
#include <unistd.h>
#include <dirent.h>

#include <openpts.h>

#define SEP_LINE "------------------------------------------------------------------------------------"


/******************************/
/* OPENPTS_UUID               */
/******************************/

/**
 * Create new OPENPTS_UUID, no contents
 */
OPENPTS_UUID *newOpenptsUuid() {
    OPENPTS_UUID *uuid;

    uuid = malloc(sizeof(OPENPTS_UUID));  // BYTE[16]
    if (uuid == NULL) {
        ERROR("no memory\n");
        return NULL;
    }
    memset(uuid, 0, sizeof(OPENPTS_UUID));

    return uuid;
}

/**
 * init UUID from file 
 */
OPENPTS_UUID *newOpenptsUuidFromFile(char * filename) {
    OPENPTS_UUID *uuid;
    int rc;

    uuid = newOpenptsUuid();
    // TODO check

    uuid->filename = smalloc(filename);
    rc = readOpenptsUuidFile(uuid);
    // TODO check

    return uuid;
}

/**
 * free OPENPTS_UUID
 */
void freeOpenptsUuid(OPENPTS_UUID *uuid) {
    /* check */
    if (uuid == NULL) {
        ERROR("null input\n");
        return;
    }

    if (uuid->filename != NULL) {
        free(uuid->filename);
    }
    if (uuid->uuid  != NULL) {
        free(uuid->uuid);
    }
    if (uuid->str  != NULL) {
        free(uuid->str);
    }
    if (uuid->time  != NULL) {
        free(uuid->time);
    }

    free(uuid);
}

/**
 *
 */
int genOpenptsUuid(OPENPTS_UUID *uuid) {
    /* check */
    if (uuid == NULL) {
        ERROR("\n");
        return PTS_INTERNAL_ERROR;
    }

    DEBUG("genOpenptsUuid()\n");

    /* free */
    if (uuid->uuid != NULL) free(uuid->uuid);
    if (uuid->str != NULL) free(uuid->str);
    if (uuid->time != NULL) free(uuid->time);

    /* set */
    uuid->uuid = newUuid();
    uuid->str  = getStringOfUuid(uuid->uuid);
    uuid->time = getDateTimeOfUuid(uuid->uuid);
    // TODO check

    if (uuid->status == OPENPTS_UUID_EMPTY) {
        uuid->status = OPENPTS_UUID_UUID_ONLY;
    } else if (uuid->status == OPENPTS_UUID_FILENAME_ONLY) {
        // TODO("genOpenptsUuid() %s filled\n", uuid->str);
        uuid->status = OPENPTS_UUID_FILLED;
    } else if (uuid->status == OPENPTS_UUID_FILLED) {
        uuid->status = OPENPTS_UUID_CHANGED;
    } else if (uuid->status == OPENPTS_UUID_CHANGED) {
        uuid->status = OPENPTS_UUID_CHANGED;
    } else if (uuid->status == OPENPTS_UUID_UUID_ONLY) {
        uuid->status = OPENPTS_UUID_UUID_ONLY;
    } else {
        ERROR("genOpenptsUuid() - bad status\n");
    }
    return PTS_SUCCESS;
}

/**
 * read UUID from file(uuid->filename), and fill OPENPTS_UUID
 */
int readOpenptsUuidFile(OPENPTS_UUID *uuid) {
    int rc = PTS_SUCCESS;
    FILE *fp;
    char line[BUF_SIZE];
    int i;



    /* check */
    if (uuid == NULL) {
        ERROR("\n");
        return PTS_INTERNAL_ERROR;
    }
    if (uuid->filename == NULL) {
        ERROR("\n");
        return PTS_INTERNAL_ERROR;
    }

    DEBUG("readOpenptsUuidFile()      : %s\n", uuid->filename);

    // TODO check UUID status?

    /* free */
    if (uuid->uuid != NULL) free(uuid->uuid);
    if (uuid->str != NULL) free(uuid->str);
    if (uuid->time != NULL) free(uuid->time);

    /* open */
    if ((fp = fopen(uuid->filename, "r")) == NULL) {
        // DEBUG("readUuidFile - UUID File %s open was failed\n", filename);
        return PTS_DENIED;  // TODO
    }

    /* init buf */
    memset(line, 0, BUF_SIZE);

    /* read */
    if (fgets(line, BUF_SIZE, fp) != NULL) {
        /* trim \n */
        /* remove LR at the end otherwise getUuidFromString() go bad */
        for (i = 0; i < BUF_SIZE; i++) {
            if (line[i] == 0x0a) {
                /* hit */
                line[i] = 0;
            }
        }
        /* parse */
        uuid->uuid = getUuidFromString(line);
        if (uuid->uuid  == NULL) {
            ERROR("readUuidFile() - UUID is NULL, file %s\n", uuid->filename);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
        uuid->str = getStringOfUuid(uuid->uuid);
        if (uuid->str == NULL) {
            ERROR("readUuidFile() - STR UUID is NULL, file %s\n", uuid->filename);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
        uuid->time = getDateTimeOfUuid(uuid->uuid);
        if (uuid->time == NULL) {
            ERROR("readUuidFile() - TIME UUID is NULL, file %s\n", uuid->filename);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
        uuid->status = OPENPTS_UUID_FILLED;
    } else {
        ERROR("readOpenptsUuidFile() - read UUID fail\n");
    }

 close:
    fclose(fp);
    return rc;
}

/**
 *
 */
int writeOpenptsUuidFile(OPENPTS_UUID *uuid, int overwrite) {
    FILE *fp;
    struct stat st;

    /* check */
    if (uuid == NULL) {
        ERROR("writeOpenptsUuidFile() - uuid == NULL\n");
        return PTS_INTERNAL_ERROR;
    }
    if ((uuid->status != OPENPTS_UUID_FILLED) && (uuid->status != OPENPTS_UUID_CHANGED)) {
        ERROR("writeOpenptsUuidFile() - uuid->status = %d\n", uuid->status);
        return PTS_INTERNAL_ERROR;
    }
    if (uuid->filename == NULL) {
        ERROR("writeOpenptsUuidFile() - uuid->filename == NULL\n");
        return PTS_INTERNAL_ERROR;
    }
    if (uuid->str == NULL) {
        ERROR("writeOpenptsUuidFile() - uuid->str == NULL\n");
        return PTS_INTERNAL_ERROR;
    }

    // TODO("writeOpenptsUuidFile() %s -> %s\n", uuid->str, uuid->filename);


    /* check the file */
    if (stat(uuid->filename, &st) == 0) {
        if (overwrite != 1) {
            /* keep existing file   */
            DEBUG("writeOpenptsUuidFile() - UUID File %s exist, return PTS_DENIED\n", uuid->filename);
            return PTS_DENIED;
        }
    }

    /* open */
    if ((fp = fopen(uuid->filename, "w")) == NULL) {
        ERROR("UUID File %s open was failed\n", uuid->filename);
        return PTS_INTERNAL_ERROR;
    }

    fprintf(fp, "%s", uuid->str);

    fclose(fp);

    return PTS_SUCCESS;
}


/**
 * Generate UUID and Write UUID file 
 *
 * Return
 *  PTS_SUCCESS  ok
 *  PTS_DENIED   if UUID file already exist
 *  PTS_INTERNAL_ERROR  else 
 */
int writeUuidFile(char *str_uuid, char *filename, int overwrite) {
    FILE *fp;
    struct stat st;

    // DEBUG("writeUuidFile() %s -> %s\n", str_uuid, filename);

    /* check */
    if (stat(filename, &st) == 0) {
        if (overwrite != 1) {
            /* keep existing file   */
            ERROR("UUID File %s exist\n", filename);
            return PTS_DENIED;
        }
    }

    /* open */
    if ((fp = fopen(filename, "w")) == NULL) {
        ERROR("UUID File %s open was failed\n", filename);
        return PTS_INTERNAL_ERROR;
    }

    fprintf(fp, "%s", str_uuid);

    fclose(fp);

    return PTS_SUCCESS;
}


/**
 * TODO deprecated use readOpenptsUuidFile
 * Read UUID file
 * 
 * Return
 *  PTS_SUCCESS        ok
 *  PTS_DENIED         missing uuiod file
 *  PTS_INTERNAL_ERROR
 *
 */
int readUuidFile(char *filename, char **str_uuid, PTS_UUID **uuid) {
    int rc = PTS_SUCCESS;
    FILE *fp;
    char line[BUF_SIZE];
    int i;

    // DEBUG("Read UUID from file %s\n",filename);

    /* open */
    if ((fp = fopen(filename, "r")) == NULL) {
        // DEBUG("readUuidFile - UUID File %s open was failed\n", filename);
        return PTS_DENIED;  // TODO
    }

    /* init buf */
    memset(line, 0, BUF_SIZE);

    /* read */
    if (fgets(line, BUF_SIZE, fp) != NULL) {
        /* trim \n */
        /* remove LR at the end otherwise getUuidFromString() go bad */
        for (i = 0; i < BUF_SIZE; i++) {
            if (line[i] == 0x0a) {
                /* hit */
                line[i] = 0;
            }
        }
        /* parse */
        *uuid = getUuidFromString(line);
        if (*uuid == NULL) {
            ERROR("readUuidFile() - UUID is NULL, file %s\n", filename);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
        *str_uuid = getStringOfUuid(*uuid);
        if (*str_uuid == NULL) {
            ERROR("readUuidFile() - STR UUID is NULL, file %s\n", filename);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    } else {
        ERROR("readUuidFile - read UUID fail\n");
    }

 close:
    fclose(fp);
    return rc;
}


/**
 * selectUuidDir
 *
 * select UUID dir, e.g. 12877946-0682-11e0-b442-001f160c9c28
 */
static int selectUuidDir(const struct dirent *entry) {
    int len;

    /* skip . .. dirs */
    if (0 == strcmp(".", entry->d_name)) return 0;
    if (0 == strcmp("..", entry->d_name)) return 0;

    /* skip bad dir name - by length */
    len = strlen(entry->d_name);
    // TODO ("UUID dirname len = %d, %s\n",len, entry->d_name);
    if (len != 36) return 0;

    // TODO not enough?, add test cases for the bad dir name

    /* Dir HIT */
    // TODO check the format
    if (entry->d_type == DT_DIR) return 1;

    return 0;
}

/**
 *  time1 > time2    1
 *  time1 <= time2  = 0
 */
int cmpDateTime(PTS_DateTime *time1, PTS_DateTime *time2) {
    uint64_t t1 = 0;
    uint64_t t2 = 0;

    t1 += time1->year;
    t1 = t1 << 16;
    t1 += time1->mon;
    t1 = t1 << 8;
    t1 += time1->mday;
    t1 = t1 << 8;
    t1 += time1->hour;
    t1 = t1 << 8;
    t1 += time1->min;
    t1 = t1 << 8;
    t1 += time1->sec;

    t2 += time2->year;
    t2 = t2 << 16;
    t2 += time2->mon;
    t2 = t2 << 8;
    t2 += time2->mday;
    t2 = t2 << 8;
    t2 += time2->hour;
    t2 = t2 << 8;
    t2 += time2->min;
    t2 = t2 << 8;
    t2 += time2->sec;

    if (t1 > t2) {
        // printf("%lX > %lX\n", t1, t2);
        return 1;
    }

    return 0;
}


void printRmList(OPENPTS_CONFIG *conf, char *indent) {
    int cnt;
    PTS_DateTime *time;
    int state;
    OPENPTS_RMSET *rmset;
    char * str_uuid;
    int num = 0;

    num = conf->rmsets->rmset_num;

    printf("%s  ID              UUID                        date(UTC)                status\n", indent);
    printf("%s %s\n", indent, SEP_LINE);

    /* Print  */
    for (cnt = 0; cnt < num; cnt++) {
        rmset = &conf->rmsets->rmset[cnt];

        str_uuid = rmset->str_uuid;
        time = rmset->time;
        state = rmset->state;

        printf("%s %3d %s %04d-%02d-%02d-%02d:%02d:%02d",
            indent,
            cnt,
            str_uuid,
            time->year + 1900,
            time->mon + 1,
            time->mday,
            time->hour,
            time->min,
            time->sec);

        if (state == OPENPTS_RM_STATE_OLD) {
            printf(" OLD\n");
        } else if (state == OPENPTS_RM_STATE_NOW) {
            printf(" NOW\n");
        } else if (state == OPENPTS_RM_STATE_NEW) {  // TODO def name is not clear
            printf(" NEW (for next boot)\n");
        } else if (state == OPENPTS_RM_STATE_TRASH) {
            printf(" RENEWED (-R to purge)\n");
        } else {
            printf(" state=UNKNOWN\n");
        }
    }
    printf("%s %s\n", indent, SEP_LINE);
}

/**
 * list/get RM
 *
 * CONFDIR/UUID/rmX.xml
 *   conf->rm_num
 *   conf->rm_uuid[]
 *   conf->rm_date[]
 *
 *   conf->current_rm_uuid
 *   conf->next_rm_uuid
 */
int getRmList(OPENPTS_CONFIG *conf, char * config_dir) {
    int cnt = 0;
    int dir_num;
    struct dirent **dir_list;
    int i, j;

    char         *tmp_str_uuid;
    PTS_UUID     *tmp_uuid;
    PTS_DateTime *tmp_time;
    int           tmp_state;
    char         *tmp_dir;

    OPENPTS_RMSET *rmset;
    OPENPTS_RMSET *rmset1;
    OPENPTS_RMSET *rmset2;

    // printf("Show RMs by UUID\n");
    // printf("config dir                  : %s\n", config_dir);

    /* move to config dir */
    if ((chdir(conf->config_dir)) != 0) {
        fprintf(stderr, "Accessing config directory %s\n", conf->config_dir);
        return PTS_INTERNAL_ERROR;
    }

    /* scan dirs */
    dir_num = scandir(".", &dir_list, &selectUuidDir, NULL);
    if ( dir_num == -1 ) {
        fprintf(stderr, "no target data\n");
        return PTS_INTERNAL_ERROR;
    }

    /* malloc */
    // TODO alloc 1 more RMSET for update
    conf->rmsets = (OPENPTS_RMSETS *) malloc(sizeof(OPENPTS_RMSETS) + sizeof(OPENPTS_RMSET) * (dir_num + 1) );
    if (conf->rmsets == NULL) {
        ERROR("no memory");
        return PTS_INTERNAL_ERROR;
    }
    conf->rmsets->rmset_num = dir_num;

    /* Set */
    for (cnt = 0; cnt < dir_num; cnt++) {
        rmset = &conf->rmsets->rmset[cnt];
        if (rmset == NULL) {
            ERROR("no memory cnt=%d\n", cnt);
            return PTS_INTERNAL_ERROR;
        }
        rmset->str_uuid = smalloc(dir_list[cnt]->d_name);
        rmset->uuid = getUuidFromString(dir_list[cnt]->d_name);
        rmset->time = getDateTimeOfUuid(rmset->uuid);
        rmset->state = OPENPTS_RM_STATE_UNKNOWN;
        rmset->dir = getFullpathName(conf->config_dir, rmset->str_uuid);

        /* check state */
        if (conf->rm_uuid->str != NULL) {
            /* check new RM 1st */
            if (conf->newrm_uuid != NULL) {
                if (conf->newrm_uuid->str != NULL) {
                    if (strcmp(conf->newrm_uuid->str, rmset->str_uuid) == 0) {
                        rmset->state = OPENPTS_RM_STATE_NEW;
                    }
                }
            }

            /* check current RM */
            if (strcmp(conf->rm_uuid->str, rmset->str_uuid) == 0) {
                /* overrite if newrm = rm */
                // TODO ("HIT %s\n", conf->str_rm_uuid);
                rmset->state = OPENPTS_RM_STATE_NOW;
            }
        }

        free(dir_list[cnt]);
    }
    free(dir_list);

    /* sort (bub) */
    for (i = 0; i< dir_num - 1; i++) {
        for (j = dir_num - 1; j > i; j--) {
            // printf("i=%d, j=%d\n",i,j);
            rmset1 = &conf->rmsets->rmset[j-1];
            rmset2 = &conf->rmsets->rmset[j];
            if (cmpDateTime(rmset1->time, rmset2->time) > 0) {
                // printf("%d <-> %d\n", j-1, j);

                tmp_str_uuid = rmset2->str_uuid;
                tmp_uuid     = rmset2->uuid;
                tmp_time     = rmset2->time;
                tmp_state    = rmset2->state;
                tmp_dir      = rmset2->dir;

                rmset2->str_uuid = rmset1->str_uuid;
                rmset2->uuid     = rmset1->uuid;
                rmset2->time     = rmset1->time;
                rmset2->state    = rmset1->state;
                rmset2->dir      = rmset1->dir;

                rmset1->str_uuid = tmp_str_uuid;
                rmset1->uuid     = tmp_uuid;
                rmset1->time     = tmp_time;
                rmset1->state    = tmp_state;
                rmset1->dir      = tmp_dir;
            }
        }
        //  printRmList(conf);
    }

    /* set current_id */
    conf->rmsets->current_id = 0;
    for (i = 0; i< dir_num; i++) {
        rmset = &conf->rmsets->rmset[i];
        if (rmset->state == OPENPTS_RM_STATE_NOW) {
            conf->rmsets->current_id = i;
        }
    }

    /* set old flag  id < current_id */
    for (i = 0; i< conf->rmsets->current_id; i++) {
        rmset = &conf->rmsets->rmset[i];
        rmset->state = OPENPTS_RM_STATE_OLD;
    }

    /* set update_id */
    conf->rmsets->update_id = 9999;  // TODO
    for (i = conf->rmsets->current_id+1; i< dir_num; i++) {
        rmset = &conf->rmsets->rmset[i];
        if (rmset->state == OPENPTS_RM_STATE_NEW) {
            conf->rmsets->update_id = i;
        }
    }

    /* set trash  flag  id < current_id */
    for (i = conf->rmsets->current_id + 1; i< dir_num && i < conf->rmsets->update_id; i++) {
        rmset = &conf->rmsets->rmset[i];
        rmset->state = OPENPTS_RM_STATE_TRASH;
    }


    return PTS_SUCCESS;
}

/**
 * rm -r RM_dir
 */
int rmRmsetDir(char * dir) {
    int rc;
    char buf[BUF_SIZE];

    // DEBUG("rm -r %s\n", dir);
    snprintf(buf, BUF_SIZE, "rm -r %s\n", dir);
    rc = system(buf);
    if (rc < 0) {
        DEBUG("rmRmsetDir() - system failed, %s\n", buf);
        return PTS_OS_ERROR;
    }

    return PTS_SUCCESS;
}

/**
 * Purge RMs 
 */
int purgeRenewedRm(OPENPTS_CONFIG *conf) {
    int cnt;
    int state;
    char * str_uuid;
    int num = 0;
    PTS_DateTime *time;
    OPENPTS_RMSET *rmset;
    int rc;
    int rc2 = PTS_SUCCESS;

    num = conf->rmsets->rmset_num;

    /* scan  */
    for (cnt = 0; cnt < num; cnt++) {
        rmset = &conf->rmsets->rmset[cnt];
        str_uuid = rmset->str_uuid;
        time = rmset->time;
        state = rmset->state;

        if (state == OPENPTS_RM_STATE_TRASH) {
            printf("  purge %s\n", rmset->str_uuid);
            rc = rmRmsetDir(rmset->dir);
            if (rc != PTS_SUCCESS) {
                rc2 = PTS_OS_ERROR;
            }
        }
    }
    return rc2;
}

/* for verifier */

/**
 * list/get target list at HOME/.openpts
 *
 */
int getTargetList(OPENPTS_CONFIG *conf, char * config_dir) {
    int cnt = 0;
    int dir_num;
    struct dirent **dir_list;
    OPENPTS_TARGET *target;
    OPENPTS_CONFIG *target_conf;

    DEBUG("getTargetList()            : %s\n", config_dir);

    /* check */
    if (conf->target_list != NULL) {
        ERROR("conf->target_list exist\n");
    }

    /* move to config dir */
    if ((chdir(conf->config_dir)) != 0) {
        fprintf(stderr, "Accessing config directory %s\n", conf->config_dir);
        return PTS_INTERNAL_ERROR;
    }

    /* scan dirs */
    dir_num = scandir(".", &dir_list, &selectUuidDir, NULL);
    if ( dir_num == -1 ) {
        fprintf(stderr, "no target data\n");
        return PTS_INTERNAL_ERROR;
    }

    /* malloc */
    conf->target_list = newTargetList(dir_num + 1);
    if (conf->target_list == NULL) {
        ERROR("no memory");
        return PTS_INTERNAL_ERROR;
    }

    /* Set */
    for (cnt = 0; cnt < dir_num; cnt++) {
        target = &conf->target_list->target[cnt];
        if (target == NULL) {
            ERROR("no memory cnt=%d\n", cnt);
            return PTS_INTERNAL_ERROR;
        }
        /* init */
        target->str_uuid = smalloc(dir_list[cnt]->d_name);
        target->uuid = getUuidFromString(dir_list[cnt]->d_name);
        target->time = getDateTimeOfUuid(target->uuid);
        target->dir = getFullpathName(conf->config_dir, target->str_uuid);
        target->target_conf_filename = getFullpathName(target->dir, "target.conf");

        DEBUG("target conf[%3d]           : %s\n", cnt, target->target_conf_filename);

        /* read target config */
        target_conf = (void *)newPtsConfig();
        if (target_conf  == NULL) {
            printf("no memory\n");
            return PTS_INTERNAL_ERROR;  // TODO
        }
        readTargetConf(target_conf, target->target_conf_filename);

        /* collector UUID is dir name */
        // TODO cleanup
        target_conf->uuid = newOpenptsUuid();
        target_conf->uuid->uuid = malloc(16);
        memcpy(target_conf->uuid->uuid, target->uuid, 16);
        target_conf->uuid->str = getStringOfUuid(target_conf->uuid->uuid);
        target_conf->uuid->time = getDateTimeOfUuid(target_conf->uuid->uuid);

        target->target_conf = (void *)target_conf;

        free(dir_list[cnt]);
    }
    free(dir_list);

    return PTS_SUCCESS;
}

/**
 *  get the target in target_list which hostname is conf->hostname (given)
 *
 * TODO change the arg
 */
char *getTargetConfDir(OPENPTS_CONFIG *conf) {
    char *dir = NULL;
    int cnt;
    OPENPTS_TARGET *target;
    OPENPTS_CONFIG *target_conf;
    int num = 0;

    num = conf->target_list->target_num;

    /* Print  */
    for (cnt = 0; cnt < num; cnt++) {
        target = &conf->target_list->target[cnt];
        target_conf = (OPENPTS_CONFIG *) target->target_conf;

        if (target_conf->hostname == NULL) {
            DEBUG("hostname is missing in %s\n", target->target_conf_filename);
        } else {
            if (!strcmp(conf->hostname, target_conf->hostname)) {
                /* HIT */
                dir = smalloc(target->dir);
                return dir;
            }
        }
    }

    return dir;
}

/**
 *  get the target in target_list which hostname is conf->hostname (given)
 *
 * TODO change the arg
 */
OPENPTS_TARGET *getTargetCollector(OPENPTS_CONFIG *conf) {
    int cnt;
    OPENPTS_TARGET *target;
    OPENPTS_CONFIG *target_conf;
    int num = 0;

    num = conf->target_list->target_num;

    /* loop  */
    for (cnt = 0; cnt < num; cnt++) {
        target = &conf->target_list->target[cnt];
        target_conf = (OPENPTS_CONFIG *) target->target_conf;

        if (target_conf != NULL) {
            if (target_conf->hostname == NULL) {
                DEBUG("hostname is missing in %s\n", target->target_conf_filename);
            } else {
                if (!strcmp(conf->hostname, target_conf->hostname)) {
                    /* HIT */
                    return target;
                }
            }
        } else {
            /* miss -> skip */
        }
    }

    return NULL;
}

void printTargetList(OPENPTS_CONFIG *conf, char *indent) {
    int cnt;
    PTS_DateTime *time;
    int state;
    OPENPTS_TARGET *target;
    OPENPTS_CONFIG *target_conf;
    char * str_uuid = NULL;
    int num = 0;

    num = conf->target_list->target_num;

    printf("%s  ID  UUID                                 "
           "date(UTC)          port port(ssh)  (username@)hostname\n",
        indent);
    printf("%s %s\n", indent, SEP_LINE);

    /* Print  */
    for (cnt = 0; cnt < num; cnt++) {
        target = &conf->target_list->target[cnt];
        target_conf = (OPENPTS_CONFIG *) target->target_conf;

        time = target->time;
        state = target->state;

        if (target_conf != NULL) {
            if (target_conf->uuid != NULL) {
                if (target_conf->uuid->str != NULL) {
                    str_uuid = target_conf->uuid->str;
                }
            }
            if (target_conf->ssh_port > 0) {
                printf("%s %3d %s %04d-%02d-%02d-%02d:%02d:%02d %4d %4d     %s@%s\n",
                    indent,
                    cnt,
                    str_uuid,
                    time->year + 1900,
                    time->mon + 1,
                    time->mday,
                    time->hour,
                    time->min,
                    time->sec,
                    target_conf->port,
                    target_conf->ssh_port,
                    target_conf->ssh_username,
                    target_conf->hostname);
            } else {
                printf("%s %3d %s %04d-%02d-%02d-%02d:%02d:%02d %4d  --      %s\n",
                    indent,
                    cnt,
                    str_uuid,
                    time->year + 1900,
                    time->mon + 1,
                    time->mday,
                    time->hour,
                    time->min,
                    time->sec,
                    target_conf->port,
                    target_conf->hostname);
            }
        } else {
            printf("--\n");
        }
    }
    printf("%s%s\n", indent, SEP_LINE);
}
