/** @ingroup backend
 * @file  backendCommon.c
 * @brief definition of functions used by backend and Bidi object.
 *
 * This file defines some common functinos which is used by any backend programs.
 *
 * @date $Date: 2004/10/22 08:32:53 $
 * @version $Revision: 1.1.1.1 $
 *
 * Copyright (C) 2004 by Turbolinux,Inc.
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <string.h>
#include <signal.h>
#include <stdarg.h>
#include <regex.h>
#include <unistd.h>
#include <locale.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "backendCommon.h"

/**
 * Function for debug print.
 *
 * @param msg is print message.
 * @return None.
 */
void debugPrint(char *msg , ...)
{
  va_list ap;
  char buf[1024];

  va_start(ap,msg);
  vsnprintf(buf,1024,msg,ap);
  fprintf(stderr,"DEBUG: %s:%s\n",PACKAGE,buf);
  va_end(ap);
}

/**
 * Function for get program's current locale+UTF-8.
 * Especially ,in case 'ja' Japanese enviroment return the "ja_JP.UTF-8".
 *
 * @return current LC_MESSAGES locale.
 */
char *getLang(void)
{
  char *lang=NULL;
  char *ptr_lang;
  char *ptr_tmp;
  char tmp[64];
  char bidi_lang[64];
  
#ifdef LC_MESSAGES
  lang = setlocale(LC_MESSAGES,"");
  if( !lang || !strcmp(lang,"C") || !strcmp(lang,"POSIX") ){
    lang = setlocale(LC_MESSAGES,NULL);
  }
#else
  lang = setlocale(LC_ALL,"");
  if( !lang || !strcmp(lang,"C") || !strcmp(lang,"POSIX") ){
    lang = setlocale(LC_ALL,NULL);
  }
#endif /* LC_MESSAGES */

  ptr_tmp=tmp;
  for(ptr_lang=lang;*ptr_lang;ptr_lang++){
    if(*ptr_lang == '.') break;
    *ptr_tmp = *ptr_lang;
    ptr_tmp++;
  }
  *ptr_tmp='\0';

  DEBUGPRINT(("%s:locale=%s",__func__,lang));
  if(strcmp(lang,"ja") == 0){ // especially ,if locale is 'ja' it's assume ja_JP...
    snprintf(tmp,sizeof(tmp),"ja_JP");
  }
  if(!lang || !strcmp(lang,"C") || !strcmp(lang,"POSIX")){
    snprintf(bidi_lang,sizeof(bidi_lang),"%s",lang);
  }else{
    snprintf(bidi_lang,sizeof(bidi_lang),"%s.UTF-8",tmp);
  }
  DEBUGPRINT(("%s:return=%s",__func__,bidi_lang));

  return strdup(bidi_lang);
}

/**
 * Get the value from specific entry of the specified buffer.
 *
 * @param buf Pointer to the buffer for reading PPD data.
 * @param entry_name.
 * @return Pointer to the getting data.
 */
static char *fetchPPDValue(const char *buf,const char *entry_name)
{
  regex_t preg;
  const int nmatch=3;
  char *regex="^\\*(.*):[ \t]+\"([[:alnum:]]+)\"";
  regmatch_t pmatch[nmatch];
  char ret[BUFSIZ];
  
  if(regcomp(&preg,regex,REG_EXTENDED|REG_NEWLINE)){
    return NULL;
  }

  if( regexec(&preg,buf,nmatch,pmatch,0) != REG_NOMATCH ) {
    if(pmatch){
      sprintf(ret,"%.*s",
	      pmatch[2].rm_eo-pmatch[2].rm_so,&buf[pmatch[2].rm_so]);
    }
  }
  regfree(&preg);
  return strdup(ret);
}

/**
 * Get the value from PPD file.
 *
 * @param ppdname filename of PPD file.
 * @param obj Pointer to the BidiWrapperObj.
 * @retval -1 Fail.
 * @retval  0 Success.
 */
int fetchPPDEntry(const char *ppdname,BidiWrapperObj *obj)
{
  FILE *fp;
  char buf[BUFSIZ];
  
  if(!ppdname) return -1;

  if( (fp=fopen(ppdname,"r")) == NULL){
    fclose(fp);
    return -1;
  }

  while(fgets(buf,sizeof(buf),fp)!=(char*)NULL){
    if(strstr(buf,PPD_PLUGIN)!=NULL){
      obj->bidiModuleName = fetchPPDValue(buf,PPD_PLUGIN);
    }else if(strstr(buf,PPD_READ_INTERVAL)!=NULL){
      obj->bidiReadInterval = atoi(fetchPPDValue(buf,PPD_READ_INTERVAL));
    }
  }
  fclose(fp);

  if(obj->bidiModuleName==NULL){
    return -1;
  }
  if(obj->bidiReadInterval<0){
    return -1;
  }

  DEBUGPRINT(("bidiModule=%s",obj->bidiModuleName));
  DEBUGPRINT(("bidiReadInterval=%d",obj->bidiReadInterval));

  return 0;
}

/**
 * Function for make a directory recursively.
 *
 * @param path Pointer to directory path string.
 * @retval < 0 Fail
 * @retval 0 Success
 */
static int mkDirP(char *path)
{
  char *copy, *p;
  mode_t oldmask=umask(0);
  p = copy = (char *)strdup(path);
  do {
    p = strchr (p + 1, '/');
    if (p)
      *p = '\0';
    if (access (copy, F_OK) == -1) {
      if (mkdir (copy, 0755) == -1) {
	fprintf(stderr,"INFO: %s:mkdir %s failed: %s\n", __func__,path, strerror(errno));
	return -1;
      }
    }
    if (p)
      *p = '/';
  } while (p);
  umask(oldmask);
  return 0;
}

/**
 * Function for write status raw data which is not processed by XML parser.
 *
 * @param bidiobj      Pointer to BidiWrapperObject.
 * @param printer_name Pointer to file name string to save.
 * @retval < 0 Fail
 * @retval 0 Success
 */
int writeXMLdata(BidiWrapperObj *bidiobj,const char *printer_name)
{
  FILE *fp;
  int namelen=256;
  char fname[namelen];
  
  if(!printer_name) return -1;
  if(strlen(printer_name) == 0) return -1;
  if( mkDirP(XML_DIR) < 0 ){
    return -1;
  }

  snprintf(fname,namelen,"%s/%s",XML_DIR,printer_name);
  fp=fopen(fname, "w");
  if(fputs(bidiobj->XMLinfo,fp) == EOF){
    DEBUGPRINT(("Failed to output XML file"));
    return -1;
  }
  fflush(fp);
  fclose(fp);
  return 0;
}

/**
 * Function for convert XML status data into IPP data.
 *
 * @param bidiobj      Pointer to BidiWrapperObject.
 * @return Pointer to infoCUPS object.
 */
infoCUPS *parseXML(BidiWrapperObj *bidiobj)
{
  Entry *entry;
  infoCUPS *info;

  if (parseParserFromBuff(bidiobj->XMLParser,bidiobj->XMLinfo,bidiobj->infoSize) < 0){
    DEBUGPRINT(("Failed to parse"));
    return NULL;
  }
  if ( (entry = fetchEntryFromParser(bidiobj->XMLParser)) == NULL ){
    DEBUGPRINT(("Failed to fetch entry from parser"));
    return NULL;
  }
  if( (info = makeIPPInfo(entry)) == NULL ){
    DEBUGPRINT(("Failed to parse entry"));
    return NULL;
  }
  return info;
}
