/**
 * @file  backendCommon.h
 * @brief definition of functions used USB and parallel backend.
 * 
 * @date $Date: 2004/10/22 08:32:53 $
 * @version $Revision: 1.1.1.1 $
 *
 * Copyright (C) 2004 by Turbolinux,Inc.
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef BACKEND_COMMON_H
#define BACKEND_COMMON_H

#include <bidiPlugin.h>
#include <bidiDbg.h>
#include <bidiEntry.h>
#include <bidiParser.h>
#include <bidiEntryCUPS.h>

#define PPD_PLUGIN "*opbidiPlugin"              /**< Bidi-Plugin module name defined in PPD.*/
#define PPD_READ_INTERVAL "*opbidiReadInterval" /**< Bidi-Plugin interval time defined in PPD.*/

#define BIDI_TIMEOUT 3 /**< Second of timeout.*/ 
#define XML_DIR "/var/log/opbidi/printer/" /**< directory name where raw XML data is saved .*/ 
#define STATE_BUF 1024 /**< size of status buffer. */
#define STATE_MESG_SIZE 1024 /**< size of message buffer. */

#ifdef __cplusplus

extern "C" {
#endif /*__cplusplus */

  void debugPrint(char *msg ,...);
#ifdef DEBUG
#define DEBUGPRINT(msg) debugPrint msg /**< function for debug print. */ 
#else // no DEBUG
#define DEBUGPRINT(msg) /**< function for debug print. */ 
#endif // DEBUG

  static int bidiTimeout=0; /**< global variable for Bidi Timeout check.*/
  static int bidiCancel=0;  /**< global variable for Bidi Cancel check.*/

  /**
   * Wrapper structure containing bidiC,bidi module name,XML parser...
   * @struct BidiWrapperObj.
   */
  typedef struct{
    BidiC *bidiC;         /**< Pointer to the real Bidi object. */
    char *bidiLang;       /**< Pointer to the locale string. */
    int bidiReadFD;       /**< File descriptor of Bidi object for read. */
    char *bidiModuleName; /**< Pointer to the string of Bidi module name. */
    int bidiReadInterval; /**< Number of interval for read status. */

    int bidiCapJob;       /**< Whether Bidi object supports bidiStartJob(),bidiEndJob(),bidiCancelJob(). */
    int bidiDoJob;        /**< Whether it is under doing Job. */
    int bidiEprogress;    /**< Whether it is under eprogress. */
    int bidiReadStatus;   /**< Whether it is under read status from bidi object. */
    int bidiJobID;        /**< Job ID. */

    char *XMLinfo;        /**< Pointer to the read status from bidi object. */
    size_t infoSize;      /**< Number of bytes of the size for status. */
    Parser *XMLParser;    /**< Pointer to the XML parser for parse bidi status. */
  }BidiWrapperObj;

  // Prototypes
  void signal_handler(int);
  void initSignal(void);
  int fetchPPDEntry(const char*,BidiWrapperObj*);
  char *getLang(void);
  infoCUPS *parseXML(BidiWrapperObj *);
  int writeXMLdata(BidiWrapperObj *,const char *);

  BidiWrapperObj *newBidiObj(void);
  BidiWrapperObj *initBidi(const char*,int,const char *);
  void *deleteBidiObj(BidiWrapperObj*);
  void deleteBidi(BidiWrapperObj*);
  int startBidiJob(BidiWrapperObj*,int);
  int endBidiJob(BidiWrapperObj*);
  int endBidiRead(BidiWrapperObj*);
  int startBidiRead(BidiWrapperObj*,int);
  int readBidiStatus(BidiWrapperObj*);

#ifdef __cplusplus
}
#endif /*__cplusplus */

#endif /* BACKEND_COMMON_H */
