/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation;                            * 
 *                                                                         *
 ***************************************************************************/

/**
 * @ingroup bidiformat
 * @file bidiParser.c
 * @brief Functions for Parser object.
 * @date $Date: 2004/10/22 08:32:53 $
 * @version $Revision: 1.1.1.1 $
   copyright            : (C) 2004 by Turbolinux,Inc.
*/

#include <string.h>

#include "bidiEntry.h"
#include "bidiParser.h"

/**
  * constructor that creates an empty Parser object.
  * @return Pointer to Parser object.
  */
Parser *newParser(void)
{
  Parser *self = xmlMalloc(sizeof(Parser));
  memset(self, '\0', sizeof(Parser));
  return self;
}

/**
  * destructor that delete Parser object.
  * @param self Pointer to Parser object.
  */
void deleteParser(Parser *self)
{
  if (self->reader != NULL) {
    xmlFreeTextReader(self->reader);
    xmlCleanupParser();
  }
  xmlFree(self);
}

/**
  * parse from file which is written in Bi-di format.
  *
  * @param self Pointer to Parser object.
  * @param uri Pointer to parse uri.
  * @retval -1 Fail.
  * @retval  0 Success.
  */
int parseParserFromFile(Parser *self, const char *uri)
{
  self->reader = xmlNewTextReaderFilename(uri);
  if (self->reader != NULL) {
    self->state = STATE_START;
  }
  return (self->reader == NULL) ? -1 : 0;
}

/**
  * parse from file descriptor which is written in Bi-di format.
  *
  * @param self Pointer to Parser object.
  * @param fd File descriptor of parse file.
  * @retval -1 Fail.
  * @retval  0 Success.
  */
int parseParserFromFd(Parser *self, int fd)
{
  xmlParserInputBufferPtr ptr;
  ptr = xmlParserInputBufferCreateFd(fd,XML_CHAR_ENCODING_UTF8);
  if ( ptr != NULL ){
    self->reader = xmlNewTextReader(ptr,NULL);
    if (self->reader != NULL) {
      self->state = STATE_START;
    }
  }
  return (self->reader == NULL) ? -1 : 0;
}

/**
  * parse from memeory which is written in Bi-di format.
  *
  * @param self Pointer to Parser object.
  * @param buffer Pointer to parse buffer.
  * @param size Number in bytes of parse buffer.
  * @retval -1 Fail.
  * @retval  0 Success.
  */
int parseParserFromBuff(Parser *self, const char *buffer, int size)
{
  xmlParserInputBufferPtr ptr=NULL;
  ptr = xmlParserInputBufferCreateMem(buffer,size,XML_CHAR_ENCODING_UTF8);
  if ( ptr != NULL ){
    self->reader = xmlNewTextReader(ptr,NULL);
    if (self->reader != NULL) {
      self->state = STATE_START;
    }
  }
  return (self->reader == NULL) ? -1 : 0;
}

/**
  * parse XML data and set data to Entry object.
  *
  * @param parser Pointer to Parser object.
  * @param entry Pointer to Entry object.
  * @retval -1 Fail.
  * @retval  0 Success.
  */
static int parseNode(Parser *parser, Entry *entry)
{
  xmlElementType node_type;
  xmlChar *element_name;
  node_type = xmlTextReaderNodeType(parser->reader);
  element_name = xmlTextReaderName(parser->reader);
  
  switch(node_type){
  case XML_READER_TYPE_ELEMENT: /* in case start of tag */
    if ( element_name == NULL){
      return -1;
    }
    //    else if (xmlStrcmp(element_name, "Printer-MIB") == 0){
    //      parser->state = STATE_PRINTER_MIB;
    //    }
    break;
  case XML_READER_TYPE_END_ELEMENT: /* in case end of tag */
    if ( parser->state == STATE_START){
      return 1;
    }
    //    else if (xmlStrcmp(element_name, "Printer-MIB") == 0){
    //      parser->state = STATE_START;
    //    }
    break;
  case XML_READER_TYPE_TEXT: /* in case text value */
    break;
  }
  
  parseNodePrtLocalization(parser,entry);
  parseNodePrtAlert(parser,entry);
  parseNodePrtMarkerSupplies(parser,entry);
  
  return 0;
}

/**
  * Function for debug parser(not in use)
  *
  */
static void xmlTextReaderErrorCallback( void *arg, const char *msg, int severity, xmlTextReaderLocatorPtr locator)
{
  char *buf=NULL;
  size_t msg_size;
  
  msg_size = sizeof(int) + sizeof(char) + sizeof(int) + sizeof(char) + strlen(msg) + 1;
  buf = (char *)malloc(msg_size);
  if(buf){
    snprintf(buf,msg_size,"%d:%d:%s",severity,xmlTextReaderLocatorLineNumber(locator),msg);
    strncpy((char *)arg,buf,msg_size);
  }
}

/**
  * Function for fetch entry object.
  *
  * @param parser Pointer to Parser object.
  * @retval NULL Fail
  * @retval *Entry Success
  */
Entry *fetchEntryFromParser(Parser *parser)
{
  Entry *entry = NULL;
  int rc = 1;
  
  char errmsg[BUFSIZ];
  memset(errmsg,'\0',BUFSIZ);
  xmlTextReaderSetErrorHandler(parser->reader, (xmlTextReaderErrorFunc)xmlTextReaderErrorCallback, errmsg);
  
  do {
    rc = xmlTextReaderRead(parser->reader);
    if (rc != 1){
      entry = NULL;
      break; /* error or EOF */
    }
    if (entry == NULL){
      entry = newEntry();
    }
    rc = parseNode(parser, entry);
  } while (rc != 1);
  
  if (rc != 1) {
    if (entry != NULL) {
      deleteEntry(entry);
    }
    return NULL;
  }
  return entry;
}
