/**
 * @file  bidiPlugin.c
 * @brief Impletation of Bi-di Plug-in caller API
 *
 * @date 2004/02/13
 * @version 1.0.0
 *
 * Copyright (c) 2002-2004, BBR Co.Ltd., All rights reserved.
 */
/*
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#define _GNU_SOURCE     /* For type "sighandler_t" */

#include <stdio.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <dlfcn.h>
#include <setjmp.h>
#include <signal.h>
#include "bidiPlugin.h"
#include "bidiDbg.h"

#define MAX_PATH_LEN 128  /**< Maximum byte length of the module path name. */

/* Declaration of struct includes fds of pipesfor work*/
struct _pipeFDs {
    int dataWrite;
    int dataRead;
    int cmdWrite;
    int cmdRead;
};

/** Variable of the stack context */
static jmp_buf env;

/**
 * Signal handler for longjmp.
 *
 * @return None.
 */
static void sigExit(void)
{
    longjmp(env, 1);
}

/**
 * Get int value from 4 bytes chars by Big Endian rule.
 *
 * @param buf Big Endian롼󤷤 intɽ4Хʸ
 * @return Ѵint
 */
static int getInt(unsigned char *buf)
{
    int val;
    val = *buf++;
    val = (val << 8) + *buf++;
    val = (val << 8) + *buf++;
    val = (val << 8) + *buf;
    return val;
}

/**
 * Set 4 bytes chars from the int value by Big Endian rule.
 *
 * @param val int value.
 * @param buf Pointer to the 4 bytes chars.
 * @return None.
 */
static void setInt(unsigned int val, char *buf)
{
    *buf++ = (char)(val >> 24);
    *buf++ = (char)(val >> 16);
    *buf++ = (char)(val >> 8);
    *buf = (char)val;
}

/**
 * Read the data from the fd as possible.
 *
 * @param fd   File descriptor.
 * @param buf  Pointer to the buffer for reading data.
 * @param size Number of bytes of the read data buffer.
 * @retval >0  Number of bytes of the read data.
 * @retval 0   When no data remains.
 * @retval -1  When error.
 */
static int fullRead(int fd, char *buf, int size)
{
    int len = 0;
    int p = 0;
    while (p < size) {
        len = read(fd, buf + p, size - p);
        if (len == 0 || len == -1) {
            _dbgPrint(sys_errlist[errno]);
            return len;
        }
        p += len;
    }
    return size;
}

/**
 * Write the data to the fd as posible.
 *
 * @param fd   File descriptor.
 * @param buf  Pointer to the buffer for writing data.
 * @param size Number of bytes of the write data buffer.
 * @retval >=0 Number of bytes of the write data.
 * @retval -1  When error.
 */
static int fullWrite(int fd, char *buf, int size)
{
    int len = 0;
    int p = 0;
    while (p < size) {
        len = write(fd, buf + p, size - p);
        if (len == -1) {
            _dbgPrint(sys_errlist[errno]);
            return -1;
        }
        p += len;
    }
    return size;
}

/**
 * Read the packet header.
 *
 * @param fd    File descriptor.
 * @param cmdID Command ID. (output)
 * @return Number of bytes of the packet body, -1 when error.
 */
static int readHeader(int fd, int *cmdID)
{
    int len;
    char header[8];

    len = fullRead(fd, header, 8);
    if (len <= 0) {
        return -1;
    }
    *cmdID = getInt(header);
    return getInt(header + 4);
}

/**
 * Read the packet body when the body is array of the int data.
 *
 * @param fd   File descriptor.
 * @param cnt  Number of the data.
 * @param args Array of the int data. (output)
 * @return Number of bytes of the packet body, -1 when error.
 */
static int readBody(int fd, int cnt, int args[])
{
    int len, i;
    char body[cnt * 4];

    len = fullRead(fd, body, cnt * 4);
    if (len <= 0) {
        return -1;
    }
    for (i = 0; i < cnt; i++) {
        args[i] = getInt(body + i * 4);
    }
    return cnt * 4;
}

/**
 * Write the packet when the body is array of the int data.
 *
 * @param fd    File descriptor.
 * @param cmdID Command ID.
 * @param cnt   Number of the data.
 * @param args  Array of the int data.
 * @return Number of bytes of the write packet, -1 when error.
 */
static int writePacket(int fd, int cmdID, int cnt, int args[])
{
    int len, i;
    char packet[4 + 4 + cnt * 4];

    setInt(cmdID, packet);
    setInt(cnt * 4, packet + 4);
    for (i = 0; i < cnt; i++) {
        setInt(args[i], packet + 8 + i * 4);
    }
    len = fullWrite(fd, packet, 4 + 4 + cnt * 4);
    if (len == -1) {
        return -1;
    }
    return 4 + 4 + cnt * 4;
}

/**
 * Write the packet when the body is string data.
 *
 * @param fd    File descriptor.
 * @param cmdID Command ID.
 * @param arg1  Length of the string data.
 * @param arg2  String data.
 * @return Number of bytes of the write packet, -1 when error.
 */
static int writePacket2(int fd, int cmdID, int arg1, char *arg2)
{
    int len, i;
    int bsize = 4 + 4 + strlen(arg2);
    char packet[4 + 4 + bsize];

    setInt(cmdID, packet);
    setInt(bsize, packet + 4);
    setInt(arg1, packet + 8);
    setInt(strlen(arg2), packet + 12);
    strncpy(packet + 16, arg2, strlen(arg2));
    len = fullWrite(fd, packet, 4 + 4 + bsize);
    if (len == -1) {
        return -1;
    }
    return 4 + 4 + bsize;
}

/**
 * Create the process type of Plug-in module binded by 2 bi-directinal pipes.
 *
 * @param fds     Pointer of the struct data includes the File descriptors of 
 *                2 bi-directinal pipes. (output)
 * @param pName   Name of the Bi-di Plug-in library or Bi-di Plug-in program.
 * @param fdRead  File descriptor for reading the printer status data from 
 *                the Bi-di Plug-in module.
 * @param fdWrite File descriptor for writing the printer command data to 
 *                the Bi-di Plug-in module.
 * @param pURI    Device URI.
 * @return ID of the created process, -1 when error.
 */
static int createProcess(struct _pipeFDs *fds, char *pName, int fdRead,
                         int fdWrite, char *pURI)
{
    int dataWriteFD[2];     // data FD: Caller-->Plug-in
    int dataReadFD[2];      // data FD: Caller<--Plug-in
    int cmdWriteFD[2];      // command FD: Caller-->Plug-in
    int cmdReadFD[2];       // commnad FD: Caller<-->lug-in
    int pid;                // Process ID

    // open pipe for data writing
    if (pipe(dataWriteFD) == -1) {
        _dbgPrint(sys_errlist[errno]);
        return -1;
    }

    // open pipe for data reading
    if (pipe(dataReadFD) == -1) {
        _dbgPrint(sys_errlist[errno]);
        return -1;
    }

    // open pipe for command writing
    if (pipe(cmdWriteFD) == -1) {
        _dbgPrint(sys_errlist[errno]);
        return -1;
    }

    // open pipe for command reading
    if (pipe(cmdReadFD) == -1) {
        _dbgPrint(sys_errlist[errno]);
        return -1;
    }

    // generating process
    if ((pid = fork()) == 0) {
        /* processing of child process (Plug-in module) */
        // close unused fds
        close(dataWriteFD[1]);
        close(dataReadFD[0]);
        close(cmdWriteFD[1]);
        close(cmdReadFD[0]);

        // preparation for exec
        char dwfd[25];
        char drfd[25];
        char cwfd[25];
        char crfd[25];
        char outfd[25];
        char infd[25];
        sprintf(dwfd, "--data-write-fd=%d", dataWriteFD[0]);
        sprintf(drfd, "--data-read-fd=%d", dataReadFD[1]);
        sprintf(cwfd, "--cmd-write-fd=%d", cmdWriteFD[0]);
        sprintf(crfd, "--cmd-read-fd=%d", cmdReadFD[1]);
        sprintf(outfd, "--output-fd=%d", fdWrite);
        sprintf(infd, "--input-fd=%d", fdRead);

        // preparation of pURI arg string
        char *uri = NULL;
        if (pURI != NULL) {
            uri = (char *) malloc(strlen("--printer-uri=") + strlen(pURI) + 1);
            if (uri == NULL) {
                _dbgPrint(sys_errlist[errno]);
                return -1;
            }
            sprintf(uri, "--printer-uri=%s", pURI);
        }

        // exec new process
        int ret;
        if (pURI == NULL) {
            ret = execlp(pName, pName, dwfd, drfd, cwfd, crfd, outfd,
                         infd, (char *)0);
        } else {
            ret = execlp(pName, pName, dwfd, drfd, cwfd, crfd, outfd,
                         infd, uri, (char *)0);
        }
        if (ret == -1) {
            // case of exec failure
            _dbgPrint(sys_errlist[errno]);
            free(uri);

            // receive command
            int len;
            int cmdID;
            int args[1];
            len = readHeader(cmdWriteFD[0], &cmdID);
            if (cmdID != BIDI_CMD_NEW || len != 4) {
                _dbgPrint("illegal command.");
                len = writePacket(cmdReadFD[1], BIDI_CMD_ERROR, 0, NULL);
                sleep(-1);
            }

            len = readBody(cmdWriteFD[0], 1, args);
            if (len != 4) {
                _dbgPrint("illegal command.");
                len = writePacket(cmdReadFD[1], BIDI_CMD_ERROR, 0, NULL);
                sleep(-1);
            }

            // send response
            args[0] = BIDI_ERROR;
            len = writePacket(cmdReadFD[1], BIDI_CMD_OK, 1, args);
            if (len == -1) {
                sleep(-1);
            }

            sleep(-1);  // wait for SIGTERM
        }
    } else if (pid >= 1) {
        /* processing of parent process (Caller) */
        // close unused fds
        close(dataWriteFD[0]);
        close(dataReadFD[1]);
        close(cmdWriteFD[0]);
        close(cmdReadFD[1]);
        fds->dataWrite = dataWriteFD[1];
        fds->dataRead = dataReadFD[0];
        fds->cmdWrite = cmdWriteFD[1];
        fds->cmdRead = cmdReadFD[0];
        return pid;

    } else {
        /* error */
        _dbgPrint(sys_errlist[errno]);
        return -1;
    }
}

/**
 * Create a new Bi-di Object. This function MUST be supported.
 *
 * @param pName   Name of the Bi-di Plug-in library or Bi-di Plug-in program.
 * @param fdRead  File descriptor for reading the printer status data from 
 *                the Bi-di Plug-in module.
 * @param fdWrite File descriptor for writing the printer command data to 
 *                the Bi-di Plug-in module.
 * @param pURI    Device URI.
 * @return Pointer to the Bi-di Object, NULL when error.
 */
BidiC *bidiNew(char *pName, int fdRead, int fdWrite, char *pURI)
{
    BidiC *p;                       // pointer of control structure
    void *handle;                   // handle of shared lib
    void *pLibC;                    // handle used by shared lib module API
    char fnameBuf[MAX_PATH_LEN];    // work buffer of shared lib name

    int pid = 0;                    // pid of process type module
    struct _pipeFDs pipeFDs;        // structure includes fds of pipesfor work

    sighandler_t oldHandler;               // variable to save signal handler

    // save current stack context
    if (setjmp(env) == 0) {
        // set sinal handler for longjmp
        oldHandler = signal(SIGALRM, (void *)sigExit);
        if (oldHandler == SIG_ERR) {
            _dbgPrint(sys_errlist[errno]);
            return NULL;
        }
    } else {
        /* context of signal raised */
        // if child process forked, kill it.
        if (pid >= 1) {
            kill(pid, SIGTERM);
        }
        free(p);
        p = NULL;
        goto errRet;
    }

    // try dlopen by pName + ".so" name
    sprintf(fnameBuf, "%s.so", pName);
    handle = dlopen(fnameBuf, RTLD_LAZY);
    if (handle == NULL) {
        // try dlopne by "lib" + pName + ".so" name
        sprintf(fnameBuf, "lib%s.so", pName);
        handle = dlopen(fnameBuf, RTLD_LAZY);
        if (handle == NULL) {
            /* processing for process typeprocess generation*/
            pid = createProcess(&pipeFDs, pName, fdRead, fdWrite, pURI);
            if (pid == -1) {
                p = NULL;
                goto errRet;
            }
        }
    }

    if (handle != NULL) {
        /* case of shared lib type */
        // linking with bidiLibNew
        void *(*func)(int, int, char*);
        char *errStr;
        func = dlsym(handle, "bidiLibNew");
        if ((errStr = dlerror()) != NULL)  {
            _dbgPrint(errStr);
            p = NULL;
            goto errRet;
        }

        // call bidiLibNew func
        pLibC = (*func)(fdRead, fdWrite, pURI);
        if (pLibC == NULL) {
            p = NULL;
            goto errRet;
        }
    } else {
        /* case of process type */
        // send command
        int len;
        int args[1] = {BIDI_PLUGIN_VERSION};
        len = writePacket(pipeFDs.cmdWrite, BIDI_CMD_NEW, 1, args);
        if (len == -1) {
            if (pid >= 1) {
                kill(pid, SIGTERM);
            }
            p = NULL;
            goto errRet;
        }

        // receive response
        int cmdID;
        len = readHeader(pipeFDs.cmdRead, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            if (pid >= 1) {
                kill(pid, SIGTERM);
            }
            p = NULL;
            goto errRet;
        }
        len = readBody(pipeFDs.cmdRead, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            if (pid >= 1) {
                kill(pid, SIGTERM);
            }
            p = NULL;
            goto errRet;
        }
        if (args[0] != BIDI_OK) {
            _dbgPrint("illegal response.");
            if (pid >= 1) {
                kill(pid, SIGTERM);
            }
            p = NULL;
            goto errRet;
        }
    }

    // allocation of BidiC area
    p = (BidiC *) malloc(sizeof(BidiC));
    if (p == NULL) {
        _dbgPrint(sys_errlist[errno]);
        p = NULL;
        goto errRet;
    }

    // generating BidiC
    if (handle != NULL) {
        /* case of shared lib type */
        p->mod.pLibC = pLibC;
    } else {
        /* case of process type */
        p->mod.prc.pid = pid;
        p->mod.prc.dataWriteFD = pipeFDs.dataWrite;
        p->mod.prc.dataReadFD = pipeFDs.dataRead;
        p->mod.prc.cmdWriteFD = pipeFDs.cmdWrite;
        p->mod.prc.cmdReadFD = pipeFDs.cmdRead;
    }
    p->dlHandle = handle;
    p->context = BIDI_CXT_NORMAL;
    p->saveContext = BIDI_CXT_NORMAL;

errRet:
    signal(SIGALRM, oldHandler);
    _dbgPrint("** terminated.");
    return p;
}

/**
 * Destroy the Bi-di Object. This function MUST be supported.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @return None.
 */
void bidiDestroy(BidiC *pBidiC)
{
    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_DESTROY, 0, NULL);
        if (len == -1) {
            kill(pBidiC->mod.prc.pid, SIGTERM);
            goto errRetProc;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 0) {
            _dbgPrint("illegal response.");
            kill(pBidiC->mod.prc.pid, SIGTERM);
            goto errRetProc;
        }

errRetProc:
        close(pBidiC->mod.prc.dataWriteFD);
        close(pBidiC->mod.prc.dataReadFD);
        close(pBidiC->mod.prc.cmdWriteFD);
        close(pBidiC->mod.prc.cmdReadFD);
        goto termRet;
    }

    /* processing of shared lib type */
    // linking with bidiLibDestroy
    void (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibDestroy");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        goto errRetLib;
    }

    // call bidiLibDestroy func
    (*func)(pBidiC->mod.pLibC);

    // unlink to shared lib
errRetLib:
    dlclose(pBidiC->dlHandle);
    // free allocated area
termRet:
    free(pBidiC);
    _dbgPrint("** terminated.");
}

/**
 * Get the Bi-di Object capabilities. This function MUST be supported.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @param cap    Enum of the Bi-di Object capabilities.
 * @retval BIDI_TRUE  The condition specified by cap is true.
 * @retval BIDI_FALSE The condition specified by cap is false.
 * @retval BIDI_ERROR When error.
 */
int bidiGetCap(BidiC *pBidiC, BidiCap cap)
{
    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1] = {cap};
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_GETCAP, 1, args);
        if (len == -1) {
            return BIDI_ERROR;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        return args[0];
    }

    /* processing of shared lib type */
    // linking with bidiLibGetCap
    int (*func)(void*, BidiCap);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibGetCap");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibGetCap func
    return (*func)(pBidiC->mod.pLibC, cap);
}

/**
 * Start the printing job sequence. This function is OPTIONAL.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @param idJob  Job ID.
 * @retval BIDI_OK        Normal terminated.
 * @retval BIDI_ERROR     When error.
 * @retval BIDI_EPROGRESS When the job is running.
 */
int bidiStartJob(BidiC *pBidiC, int idJob)
{
    // check context
    if (pBidiC->context == BIDI_CXT_JOBIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // set context
    pBidiC->saveContext = pBidiC->context;
    pBidiC->context = BIDI_CXT_JOBIN;

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1] = {idJob};
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_STARTJOB, 1, args);
        if (len == -1) {
            return BIDI_ERROR;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        return args[0];
    }

    /* processing of shared lib type */
    // linking with bidiLibStartJob
    int (*func)(void*, int);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibStartJob");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibStartJob func
    return (*func)(pBidiC->mod.pLibC, idJob);
}

/**
 * Stop the printing job sequence. This function is OPTIONAL.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @retval BIDI_OK        Normal terminated.
 * @retval BIDI_ERROR     When error.
 * @retval BIDI_EPROGRESS When the job can't stop(the job is running).
 */
int bidiEndJob(BidiC *pBidiC)
{
    // check context
    if (pBidiC->context != BIDI_CXT_JOBIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1];
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_ENDJOB, 0, NULL);
        if (len == -1) {
            return BIDI_ERROR;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        if (args[0] == BIDI_OK) {
            goto normalRet;
        }
        return args[0];
    }

    /* processing of shared lib type */
    // linking with bidiLibEndJob
    int (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibEndJob");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibEndJob func
    int ret = (*func)(pBidiC->mod.pLibC);
    if (ret != BIDI_OK) {
        return ret;
    }

    // set contextonly case of BIDI_OK
normalRet:
    pBidiC->context = pBidiC->saveContext;
    return BIDI_OK;
}

/**
 * Cancel the printing job sequence. This function is OPTIONAL.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @param idJob  Job ID.
 * @retval BIDI_OK        Normal terminated.
 * @retval BIDI_ERROR     When error.
 * @retval BIDI_EPROGRESS When the job can't stop(the job is running).
 * @retval BIDI_ENDJOB    When the job is not exist in Bi-di Plug-in module.
 */
int bidiCancelJob(BidiC *pBidiC, int idJob)
{
    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1] = {idJob};
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_CANCELJOB, 1, args);
        if (len == -1) {
            return BIDI_ERROR;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        return args[0];
    }

    /* processing of shared lib type */
    // linking with bidiLibCancelJob
    int (*func)(void*, int);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibCancelJob");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibCancelJob func
    return (*func)(pBidiC->mod.pLibC, idJob);
}

/**
 * Get the file descriptor for reading from the Bi-di Object.
 * This function MUST be supported.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @return File descriptor. BIDI_ERROR when error.
 */
int bidiGetReadFD(BidiC *pBidiC)
{
    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        return pBidiC->mod.prc.dataReadFD;
    }

    /* processing of shared lib type */
    // linking with bidiLibGetReadFD
    int (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibGetReadFD");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibGetReadFD func
    return (*func)(pBidiC->mod.pLibC);
}

/**
 * Start the reading sequence. This function MUST be supported.
 *
 * @param pBidiC     Pointer to the Bi-di Object.
 * @param idReadMode Enum of the Bi-di Object reading mode.
 * @param pLang      Pointer to the language string.
 *                   (ʸƱͤθ+ϰ+ʸɻʸ)
 *                   NULL
 * @retval BIDI_OK        Normal terminated.
 * @retval BIDI_ERROR     When error.
 * @retval BIDI_EINTR     When accepted signals.
 */
int bidiStartRead(BidiC *pBidiC, BidiReadMode idReadMode, char *pLang)
{
    int ret;
    sighandler_t oldHandler;   // variable to save signal handler

    // check context
    if (pBidiC->context == BIDI_CXT_READIN || pBidiC->context == BIDI_CXT_WRITEIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // set context
    pBidiC->saveContext = pBidiC->context;
    pBidiC->context = BIDI_CXT_READIN;

    // save current stack context
    if (setjmp(env) == 0) {
        // set sinal handler for longjmp
        oldHandler = signal(SIGALRM, (void *)sigExit);
        if (oldHandler == SIG_ERR) {
            _dbgPrint(sys_errlist[errno]);
            return BIDI_ERROR;
        }
    } else {
        /* context of signal raised */
        // if process type, kill SIGHUP.
        if (pBidiC->dlHandle == NULL) {
            kill(pBidiC->mod.prc.pid, SIGHUP);
        }
        ret = BIDI_EINTR;
        goto errRet;
    }

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        char *arg2 = (pLang != NULL) ? pLang : "";
        len = writePacket2(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_STARTREAD,
                           idReadMode, arg2);
        if (len == -1) {
            ret = BIDI_ERROR;
            goto errRet;
        }

        // receive response
        int cmdID;
        int args[1];
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            ret = BIDI_ERROR;
            goto errRet;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            ret = BIDI_ERROR;
            goto errRet;
        }

        // read 1 byte dummy data for bidiRead
        char dum;
        len = read(pBidiC->mod.prc.dataReadFD, &dum, 1);
        if (len != 1) {
            _dbgPrint(sys_errlist[errno]);
            _dbgPrint("illegal pipe setup.");
            ret = BIDI_ERROR;
            goto errRet;
        }
        ret = args[0];
        goto termRet;
    }

    /* processing of shared lib type */
    // linking with bidiLibStartRead
    int (*func)(void*, BidiReadMode, char*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibStartRead");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        ret = BIDI_ERROR;
        goto errRet;
    }

    // call bidiLibStartRead func
    ret = (*func)(pBidiC->mod.pLibC, idReadMode, pLang);
termRet:
errRet:
    signal(SIGALRM, oldHandler);
    return ret;
}

/**
 * Read the printer status from the Bi-di Object.
 * This function MUST be supported.
 *
 * @param pBidiC    Pointer to the Bi-di Object.
 * @param pBuf      Pointer to the buffer for storing the printer status data.
 * @param nBufBytes Number of bytes of the printer status data buffer.
 * @retval >0           Number of bytes of the printer status data read.
 * @retval 0            When no data remains.
 * @retval BIDI_ERROR   When error.
 * @retval BIDI_EINTR   When accepted signals.(read 0byte)
 */
int bidiRead(BidiC *pBidiC, void *pBuf, int nBufBytes)
{
    int len = 0;
    sighandler_t oldHandler;   // variable to save signal handler

    // check context
    if (pBidiC->context != BIDI_CXT_READIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // save current stack context
    if (setjmp(env) == 0) {
        // set sinal handler for longjmp
        oldHandler = signal(SIGALRM, (void *)sigExit);
        if (oldHandler == SIG_ERR) {
            _dbgPrint(sys_errlist[errno]);
            return BIDI_ERROR;
        }
    } else {
        /* context of signal raised */
        // if process type, kill SIGHUP.
        if (pBidiC->dlHandle == NULL && len == 0) {
            kill(pBidiC->mod.prc.pid, SIGHUP);
        }
        if (len == 0) {
            len = BIDI_EINTR;
        }
        goto errRet;
    }

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len2;
        int args[1] = {nBufBytes};
        len2 = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_READ, 1, args);
        if (len2 == -1) {
            len = BIDI_ERROR;
            goto errRet;
        }

        // receive response
        int cmdID;
        len2 = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len2 != 4) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }
        len2 = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len2 != 4) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }

        // read data
        len = read(pBidiC->mod.prc.dataReadFD, pBuf, args[0]);
        if (len == -1) {
            _dbgPrint(sys_errlist[errno]);
            len = BIDI_ERROR;
            goto errRet;
        }

        goto termRet;
    }

    /* processing of shared lib type */
    // linking with bidiLibRead
    int (*func)(void*, void*, int);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibRead");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        len = BIDI_ERROR;
        goto errRet;
    }

    // call bidiLibRead func
    len = (*func)(pBidiC->mod.pLibC, pBuf, nBufBytes);
termRet:
errRet:
    signal(SIGALRM, oldHandler);
    return len;
}

/**
 * Stop the reading sequence. This function MUST be supported.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @retval BIDI_OK        Normal terminated.
 * @retval BIDI_ERROR     When error.
 */
int bidiEndRead(BidiC *pBidiC)
{
    // check context
    if (pBidiC->context != BIDI_CXT_READIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // set context
    pBidiC->context = pBidiC->saveContext;
    pBidiC->saveContext = BIDI_CXT_NORMAL;

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1];
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_ENDREAD, 0, NULL);
        if (len == -1) {
            return BIDI_ERROR;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        return args[0];
    }

    /* processing of shared lib type */
    // linking with bidiLibEndRead
    int (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibEndRead");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibEndRead func
    return (*func)(pBidiC->mod.pLibC);
}

/**
 * Get the file descriptor for writing to the Bi-di Object.
 * This function MUST be supported.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @return File descriptor,BIDI_ERROR when error.
 */
int bidiGetWriteFD(BidiC *pBidiC)
{
    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        return pBidiC->mod.prc.dataWriteFD;
    }

    /* processing of shared lib type */
    // linking with bidiLibGetWriteFD
    int (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibGetWriteFD");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibGetWriteFD func
    return (*func)(pBidiC->mod.pLibC);
}

/**
 * Start the writing sequence. This function is OPTIONAL.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @retval BIDI_OK      Normal terminated.
 * @retval BIDI_ERROR   When error.
 * @retval BIDI_EINTR   When accepted signals.(read 0byte)
 */
int bidiStartWrite(BidiC *pBidiC)
{
    int ret;
    sighandler_t oldHandler;   // variable to save signal handler

    // check context
    if (pBidiC->context == BIDI_CXT_READIN || pBidiC->context == BIDI_CXT_WRITEIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // set context
    pBidiC->saveContext = pBidiC->context;
    pBidiC->context = BIDI_CXT_WRITEIN;

    // save current stack context
    if (setjmp(env) == 0) {
        // set sinal handler for longjmp
        oldHandler = signal(SIGALRM, (void *)sigExit);
        if (oldHandler == SIG_ERR) {
            _dbgPrint(sys_errlist[errno]);
            return BIDI_ERROR;
        }
    } else {
        /* context of signal raised */
        // if process type, kill SIGHUP.
        if (pBidiC->dlHandle == NULL) {
            kill(pBidiC->mod.prc.pid, SIGHUP);
        }
        ret = BIDI_EINTR;
        goto errRet;

    }

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1];
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_STARTWRITE, 0, NULL);
        if (len == -1) {
            ret = BIDI_ERROR;
            goto errRet;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            ret = BIDI_ERROR;
            goto errRet;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            ret = BIDI_ERROR;
            goto errRet;
        }
        ret = args[0];
        goto termRet;
    }

    /* processing of shared lib type */
    // linking with bidiLibStartWrite
    int (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibStartWrite");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        ret = BIDI_ERROR;
        goto errRet;
    }

    // call bidiLibStartWrite func
    ret = (*func)(pBidiC->mod.pLibC);
termRet:
errRet:
    signal(SIGALRM, oldHandler);
    return ret;
}

/**
 * Write data to the Bi-di Object. This function is OPTIONAL.
 *
 * @param pBidiC    Pointer to the Bi-di Object.
 * @param pBuf      Pointer to the buffer for writing data.
 * @param nBufBytes Number of bytes of the writing data.
 * @retval >0           Number of bytes of the writing data.
 * @retval 0            When no data written.
 * @retval BIDI_ERROR   When error.
 * @retval BIDI_EINTR   When accepted signals.(write 0byte)
 */
int bidiWrite(BidiC *pBidiC, void *pBuf, int nBufBytes)
{
    int len = 0;
    sighandler_t oldHandler;   // variable to save signal handler

    // check context
    if (pBidiC->context != BIDI_CXT_WRITEIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // save current stack context
    if (setjmp(env) == 0) {
        // set sinal handler for longjmp
        oldHandler = signal(SIGALRM, (void *)sigExit);
        if (oldHandler == SIG_ERR) {
            _dbgPrint(sys_errlist[errno]);
            return BIDI_ERROR;
        }
    } else {
        /* context of signal raised */
        // if process type, kill SIGHUP.
        if (pBidiC->dlHandle == NULL && len == 0) {
            kill(pBidiC->mod.prc.pid, SIGHUP);
        }
        if (len == 0) {
            len = BIDI_EINTR;
        }
        goto errRet;
    }

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len2;
        int args[1] = {nBufBytes};
        len2 = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_WRITE, 1, args);
        if (len2 == -1) {
            len = BIDI_ERROR;
            goto errRet;
        }

        // receive response
        int cmdID;
        len2 = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len2 != 4) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }
        len2 = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len2 != 4) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }
        int nBytes = args[0];

        // write data
        len = write(pBidiC->mod.prc.dataWriteFD, pBuf, nBytes);
        if (len == -1) {
            _dbgPrint(sys_errlist[errno]);
            len = BIDI_ERROR;
            goto errRet;
        }

        goto termRet;
    }

    /* processing of shared lib type */
    // linking with bidiLibWrite
    int (*func)(void*, void*, int);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibWrite");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        len = BIDI_ERROR;
        goto errRet;
    }

    // call bidiLibWrite func
    len = (*func)(pBidiC->mod.pLibC, pBuf, nBufBytes);
termRet:
errRet:
    signal(SIGALRM, oldHandler);
    return len;
}

/**
 * Stop the writing sequence. This function is OPTIONAL.
 *
 * @param pBidiC Pointer to the Bi-di Object.
 * @retval BIDI_OK      Normal terminated.
 * @retval BIDI_ERROR   When error.
 */
int bidiEndWrite(BidiC *pBidiC)
{
    // check context
    if (pBidiC->context != BIDI_CXT_WRITEIN) {
        _dbgPrint("illegal call.");
        return BIDI_ERROR;
    }

    // set context
    pBidiC->context = pBidiC->saveContext;
    pBidiC->saveContext = BIDI_CXT_NORMAL;

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len;
        int args[1];
        len = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_ENDWRITE, 0, NULL);
        if (len == -1) {
            return BIDI_ERROR;
        }

        // receive response
        int cmdID;
        len = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        len = readBody(pBidiC->mod.prc.cmdReadFD, 1, args);
        if (len != 4) {
            _dbgPrint("illegal response.");
            return BIDI_ERROR;
        }
        return args[0];
    }

    /* processing of shared lib type */
    // linking with bidiLibEndWrite
    int (*func)(void*);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibEndWrite");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        return BIDI_ERROR;
    }

    // call bidiLibEndWrite func
    return (*func)(pBidiC->mod.pLibC);
}

/**
 * Control the Bi-di Object. This function is OPTIONAL.
 *
 * @param pBidiC      Pointer to the Bi-di Object.
 * @param idRequest   Request ID.
 * @param pData       Pointer to the for each request.
 * @param nDataBytes  Number of bytes of the data.
 * @retval >0         Number of bytes of the data read.
 * @retval 0          When no data remains.
 * @retval BIDI_ERROR When error.
 * @retval BIDI_EINTR When accepted signals.
 */
int bidiCtrl(BidiC *pBidiC, int idRequest, void *pData, int nDataBytes)
{
    int len;
    sighandler_t oldHandler;   // variable to save signal handler

    // save current stack context
    if (setjmp(env) == 0) {
        // set sinal handler for longjmp
        oldHandler = signal(SIGALRM, (void *)sigExit);
        if (oldHandler == SIG_ERR) {
            _dbgPrint(sys_errlist[errno]);
            return BIDI_ERROR;
        }
    } else {
        /* context of signal raised */
        if (pBidiC->dlHandle == NULL) {
            kill(pBidiC->mod.prc.pid, SIGHUP);
        }
        len = BIDI_EINTR;
        goto errRet;
    }

    if (pBidiC->dlHandle == NULL) {
        /* processing of process type */
        // send command
        int len2;
        int args[2] = {idRequest, nDataBytes};
        len2 = writePacket(pBidiC->mod.prc.cmdWriteFD, BIDI_CMD_CTRL, 2, args);
        if (len2 == -1) {
            len = BIDI_ERROR;
            goto errRet;
        }

        // receive response 1
        int cmdID;
        len2 = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len2 != 0) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }

        // write data
        len = write(pBidiC->mod.prc.dataWriteFD, pData, nDataBytes);
        if (len == -1) {
            _dbgPrint(sys_errlist[errno]);
            len = BIDI_ERROR;
            goto errRet;
        }

        // receive response 2
        len2 = readHeader(pBidiC->mod.prc.cmdReadFD, &cmdID);
        if (cmdID != BIDI_CMD_OK || len2 != 0) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }
        len2 = readBody(pBidiC->mod.prc.cmdReadFD, 2, args);
        if (len != 8) {
            _dbgPrint("illegal response.");
            len = BIDI_ERROR;
            goto errRet;
        }
        if (args[0] == BIDI_ERROR) {
            _dbgPrint("unknown error in plug-in process.");
            len = BIDI_ERROR;
            goto errRet;
        }

        // read data
        len = write(pBidiC->mod.prc.dataReadFD, pData, args[1]);
        if (len == -1) {
            _dbgPrint(sys_errlist[errno]);
            len = BIDI_ERROR;
            goto errRet;
        }

        goto termRet;
    }

    /* processing of shared lib type */
    // linking with bidiLibCtrl
    int (*func)(void*, int, void*, int);
    char *errStr;
    func = dlsym(pBidiC->dlHandle, "bidiLibCtrl");
    if ((errStr = dlerror()) != NULL) {
        _dbgPrint(errStr);
        len = BIDI_ERROR;
        goto errRet;
    }

    // call bidiLibCtrl func
    len = (*func)(pBidiC->mod.pLibC, idRequest, pData, nDataBytes);
termRet:
errRet:
    signal(SIGALRM, oldHandler);
    return len;
}
