/**
 * @file  bidiPlugin.h
 * @brief Header file of Bi-di Plug-in caller API
 *
 * @date 2004/02/13
 * @version 1.0.0
 *
 * Copyright (c) 2002-2004, BBR Co.Ltd., All rights reserved.
 */
/*
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _BIDIPLUGIN_H
#define _BIDIPLUGIN_H

/* definitions of constants */
// return values
/** Return value shows normal return. */
#define BIDI_OK             0
/** Return value shows false value. */
#define BIDI_FALSE          0
/** Return value shows true value. */
#define BIDI_TRUE           1
/** Return value shows error return. */
#define BIDI_ERROR          (-1)
/** Return value shows signal interuption. */
#define BIDI_EINTR          (-2)
/** Return value shows in processing. */
#define BIDI_EPROGRESS      (-3)
/** Return value shows no job. */
#define BIDI_ENOJOB         (-4)

// command IDs used in process type
/** Command ID of the bidiNew() function. */
#define BIDI_CMD_NEW        0x00000001
/** Command ID of the bidiDestroy() function. */
#define BIDI_CMD_DESTROY    0x00000002
/** Command ID of the bidiGetCap() function. */
#define BIDI_CMD_GETCAP     0x00000003
/** Command ID of the bidiStartJob() function. */
#define BIDI_CMD_STARTJOB   0x00000011
/** Command ID of the bidiEndJob() function. */
#define BIDI_CMD_ENDJOB     0x00000012
/** Command ID of the bidiCancelJob() function. */
#define BIDI_CMD_CANCELJOB  0x00000013
/** Command ID of the bidiStartRead() function. */
#define BIDI_CMD_STARTREAD  0x00000021
/** Command ID of the bidiEndRead() function. */
#define BIDI_CMD_ENDREAD    0x00000022
/** Command ID of the bidiRead() function. */
#define BIDI_CMD_READ       0x00000023
/** Command ID of the bidiStartWrite() function. */
#define BIDI_CMD_STARTWRITE 0x00000031
/** Command ID of the bidiEndWrite() function. */
#define BIDI_CMD_ENDWRITE   0x00000032
/** Command ID of the bidiWrite() function. */
#define BIDI_CMD_WRITE      0x00000033
/** Command ID of the bidiCtrl() function. */
#define BIDI_CMD_CTRL       0x00000034

// command IDs for response
/** Command ID of response shows normal processing. */
#define BIDI_CMD_OK         0x80000000
/** Command ID of response shows illegal operation. */
#define BIDI_CMD_ERROR      0x80000001

// other definitions
/** Minimum version of Plug-in module which can be accepted. */
#define BIDI_PLUGIN_VERSION 0x00010000


/* type definitions */
/** Enum type used in the bidiGetCap() function. */
typedef enum {
    /** Supports the bidiStartWrite(), bidiEndWrite() and 
        the bidiWrite() functions. */
    BIDI_CAP_WRITE = 1,
    /** Supports the bidiStartJob(), bidiEndJob() and 
        the bidiCancelJob() functions. */
    BIDI_CAP_JOB   = 2,
    /** Supports the bidiCtrl() function. */
    BIDI_CAP_CTRL  = 3
} BidiCap;

/** Enum type shows the read mode of the print status. */
typedef enum {
    /** û̷Υץ󥿥ơread롣 */
    BIDI_READ_PRT_MIB_SUMMARY = 1
} BidiReadMode;

/** Enum type shows the context status. */
typedef enum {
    /** Context shows normal status. */
    BIDI_CXT_NORMAL,
    /** Context shows between the start and the end read. */
    BIDI_CXT_READIN,
    /** Context shows between the start and the end write. */
    BIDI_CXT_WRITEIN,
    /** Context shows between the start and the end job. */
    BIDI_CXT_JOBIN
} BidiCxt;

/** Control object used in all Bi-di Plug-in APIs. */
typedef struct {
    void *dlHandle;     /**< Handle returned by dlopen(),
                             if NULL, indicates process type. */
    /** Union part in the control object. */
    union {
        /* for shared lib type */
        void *pLibC;    /**< Control object used in Bi-di Plug-in module. */

        /* for process type */
        /** Struct data for process type in the control object. */
        struct {
            int pid;            /**< Process ID. */
            int dataWriteFD;    /**< Pipe fd for data writing. */
            int dataReadFD;     /**< Pipe fd for data reading. */
            int cmdWriteFD;     /**< Pipe fd for command writing. */
            int cmdReadFD;      /**< Pipe fd for command reading. */
        } prc;
    } mod;
    BidiCxt context;        /**< Current context. */
    BidiCxt saveContext;    /**< Previous context. */
} BidiC;


/* declarations of Bidi Plug-in APIs */
BidiC *bidiNew(char *pName, int fdRead, int fdWrite, char *pURI);

void bidiDestroy(BidiC *pBidiC);

int bidiGetCap(BidiC *pBidiC, BidiCap cap);

int bidiStartJob(BidiC *pBidiC, int idJob);

int bidiEndJob(BidiC *pBidiC);

int bidiCancelJob(BidiC *pBidiC, int idJob);

int bidiGetReadFD(BidiC *pBidiC);

int bidiStartRead(BidiC *pBidiC, BidiReadMode idReadMode, char *pLang);

int bidiRead(BidiC *pBidiC, void *pBuf, int nBufBytes);

int bidiEndRead(BidiC *pBidiC);

int bidiGetWriteFD(BidiC *pBidiC);

int bidiStartWrite(BidiC *pBidiC);

int bidiWrite(BidiC *pBidiC, void *pBuf, int nBufBytes);

int bidiEndWrite(BidiC *pBidiC);

int bidiCtrl(BidiC *pBidiC, int idRequest, void *pData, int nDataBytes);

#endif // _BIDIPLUGIN_H
