using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace Oratorio
{
    public partial class TableDefineForm : Form
    {
        protected DatabaseContext _database;
        private string _tableName;

        public string TableName
        {
            get { return _tableName; }
            set { _tableName = value; }
        }

        private string _catalogName;

        public string CatalogName
        {
            get { return _catalogName; }
            set { _catalogName = value; }
        }

        private string _schemaName;

        public string SchemaName
        {
            get { return _schemaName; }
            set { _schemaName = value; }
        }

        protected bool _columnInfoLoaded;
        protected bool _indexInfoLoaded;

        public TableDefineForm(DatabaseContext database, string catalogName, string schemaName, string tableName)
        {
            if (database == null)
            {
                throw new ArgumentNullException("database");
            }
            if (schemaName == null)
            {
                throw new ArgumentNullException("schemaName");
            }
            if (catalogName == null)
            {
                throw new ArgumentNullException("catalogName");
            }
            if (tableName == null)
            {
                throw new ArgumentNullException("tableName");
            }
            InitializeComponent();

            _database = database;
            _schemaName = schemaName;
            _catalogName = catalogName;
            _tableName = tableName;
        }

        private void TableDefineForm_Load(object sender, EventArgs e)
        {
            _gridColumn.AllowUserToAddRows = false;
            _gridColumn.ReadOnly = true;
            _gridColumn.Columns.Add("name", "Name");
            _gridColumn.Columns.Add("type", "Type");
            _gridColumn.Columns.Add("notNull", "Not Null");
            _gridColumn.Columns.Add("key", "Key");
            _gridColumn.Columns.Add("order", "Order");

            _gridIndex.Columns.Add("indexName", "Index Name");
            _gridIndex.Columns.Add("unique", "Unique");
            _gridIndex.Columns.Add("columnName", "Column Name");
            _gridIndex.Columns.Add("order", "Order");
        }

        public void RefreshPanel()
        {
            if (_tabPanel.SelectedIndex == 0)
            {
                if (_columnInfoLoaded)
                {
                    return;
                }
                _gridColumn.Rows.Clear();
                try
                {
                    List<DatabaseColumn> columns = new List<DatabaseColumn>();
                    _database.GetTableColumnList(_catalogName, _schemaName, _tableName, columns);

                    foreach (DatabaseColumn column in columns)
                    {
                        int rowIndex = _gridColumn.Rows.Add();
                        DataGridViewRow row = _gridColumn.Rows[rowIndex];
                        string columnName = column.ColumnName;
                        row.Cells[0].Value = columnName;
                        row.Cells[1].Value = column.TypeAndSize;
                        row.Cells[2].Value = (column.IsNullable ? "" : "NOT NULL");
                        row.Cells[3].Value = (column.IsKey ? "PRIMARY" : "");
                        row.Cells[4].Value = (column.Index + 1).ToString();
                    }
                }
                catch (DatabaseContextException de)
                {
                    throw new ProgramException("Get schema information failed.", de);
                }
                _columnInfoLoaded = true;
            }
            else if (_tabPanel.SelectedIndex == 1)
            {
                if (_indexInfoLoaded)
                {
                    return;
                }
                _gridIndex.Rows.Clear();
                try
                {
                    DatabaseObjectIndex[] indexList = _database.GetIndexList(_catalogName, _schemaName, _tableName);
                    foreach (DatabaseObjectIndex indexObject in indexList)
                    {
                        int rowIndex = _gridIndex.Rows.Add();
                        DataGridViewRow row = _gridIndex.Rows[rowIndex];
                        row.Cells[0].Value = indexObject.ObjectName;
                        row.Cells[1].Value = (indexObject.IsUnique ? "UNIQUE" : "NON UNIQUE");
                        foreach (DatabaseColumn column in indexObject.ColumnList)
                        {
                            if (column.Index > 0)
                            {
                                rowIndex = _gridIndex.Rows.Add();
                                row = _gridIndex.Rows[rowIndex];
                            }
                            row.Cells[2].Value = column.ColumnName;
                            row.Cells[3].Value = column.Index + 1;
                        }
                    }
                }
                catch (DatabaseContextException de)
                {
                    throw new ProgramException("Get schema information failed.", de);
                }
                _indexInfoLoaded = true;
            }
        }

        private void _tabPanel_SelectedIndexChanged(object sender, EventArgs e)
        {
            RefreshPanel();
        }

        protected DataGridViewCell FindTextWithForm(DataGridView dataGridView, string targetText, bool senseCase, bool searchNext)
        {
            if (targetText.Length == 0)
            {
                return null;
            }
            //FindTextForm form = new FindTextForm();
            //if (form.ShowDialog(this) != DialogResult.OK)
            //{
            //    return null;
            //}
            DataGridViewCell beforeCell = dataGridView.CurrentCell;
            int startRowIndex = 0;
            int startColumnIndex = 0;
            if (beforeCell != null)
            {
                startRowIndex = beforeCell.RowIndex;
                startColumnIndex = beforeCell.ColumnIndex;
            }

            StringComparison compareOption = StringComparison.OrdinalIgnoreCase;
            if (senseCase)
            {
                compareOption = StringComparison.Ordinal;
            }

            if (searchNext)
            {
                startColumnIndex++;
                if (startColumnIndex == dataGridView.ColumnCount)
                {
                    startColumnIndex = 0;
                    startRowIndex++;
                }
                if (startRowIndex + 1 == dataGridView.RowCount)
                {
                    //if (MessageBox.Show(this, "Search grid head?", Application.ProductName, MessageBoxButtons.YesNo) != DialogResult.Yes)
                    //{
                    //    break;
                    //}
                    startColumnIndex = 0;
                    startRowIndex = 0;
                }
            }

            int columnCount = dataGridView.ColumnCount;
            DataGridViewRowCollection rows = dataGridView.Rows;
            DataGridViewCell findCell = null;
            foreach (DataGridViewRow row in rows)
            {
                if (row.Index < startRowIndex)
                {
                    continue;
                }
                for (int column = 0; column < columnCount; column++)
                {
                    if (row.Index == startRowIndex && column < startColumnIndex)
                    {
                        continue;
                    }
                    object value = row.Cells[column].Value;
                    if (value == null)
                    {
                        continue;
                    }
                    if (value.ToString().IndexOf(targetText, compareOption) >= 0)
                    {
                        findCell = row.Cells[column];
                        break;
                    }
                }
                if (findCell != null)
                {
                    break;
                }
            }
            return findCell;
        }

        public DataGridView GetSelectedDataGridView()
        {
            if (_tabPanel.SelectedIndex == 0)
            {
                return _gridColumn;
            }
            else if (_tabPanel.SelectedIndex == 1)
            {
                return _gridIndex;
            }
            else
            {
                System.Diagnostics.Trace.Assert(false);
                throw new InvalidOperationException();
            }
        }

        public void ShowFindForm()
        {
            DataGridView dataGridView = GetSelectedDataGridView();

            if (dataGridView.ColumnCount == 0 || dataGridView.RowCount == 0)
            {
                ProgramUtility.ShowError(this, "Warning: Grid is empty.", null);
                return;
            }

            FindTextForm form = new FindTextForm();

            bool beforeFind = false;
            while (true)
            {
                if (form.ShowDialog(this) != DialogResult.OK)
                {
                    return;
                }
                DataGridViewCell findCell = FindTextWithForm(dataGridView, form.TargetText, form.SenseCase, beforeFind);
                if (findCell == null)
                {
                    MessageBox.Show(this, "Not found.", Application.ProductName);
                    beforeFind = false;
                }
                else
                {
                    dataGridView.CurrentCell = findCell;
                    beforeFind = true;
                }
            }
            //DataGridView dataGridView = _gridColumn;
            //if (dataGridView.ColumnCount == 0 || dataGridView.RowCount == 0)
            //{
            //    ProgramUtility.ShowError(this, "Warning: Grid is empty.", null);
            //    return;
            //}

            //FindTextForm form = new FindTextForm();

            //bool beforeFind = false;
            //while (true)
            //{
            //    DataGridViewCell beforeCell = dataGridView.CurrentCell;
            //    int startRowIndex = 0;
            //    int startColumnIndex = 0;
            //    if (beforeCell != null)
            //    {
            //        startRowIndex = beforeCell.RowIndex;
            //        startColumnIndex = beforeCell.ColumnIndex;
            //    }
            //    if (form.ShowDialog(this) != DialogResult.OK)
            //    {
            //        return;
            //    }
            //    StringComparison compareOption = StringComparison.OrdinalIgnoreCase;
            //    if (form.SenseCase)
            //    {
            //        compareOption = StringComparison.Ordinal;
            //    }
            //    string targetText = form.TargetText;
            //    if (targetText.Length == 0)
            //    {
            //        return;
            //    }

            //    if (beforeFind)
            //    {
            //        startColumnIndex++;
            //        if (startColumnIndex == dataGridView.ColumnCount)
            //        {
            //            startColumnIndex = 0;
            //            startRowIndex++;
            //        }
            //        if (startRowIndex + 1 == dataGridView.RowCount)
            //        {
            //            if (MessageBox.Show(this, "Search grid head?", Application.ProductName, MessageBoxButtons.YesNo) != DialogResult.Yes)
            //            {
            //                break;
            //            }
            //            startColumnIndex = 0;
            //            startRowIndex = 0;
            //        }
            //    }

            //    int columnCount = _gridControl.GridView.ColumnCount;
            //    DataGridViewRowCollection rows = _gridControl.GridView.Rows;
            //    DataGridViewCell findCell = null;
            //    foreach (DataGridViewRow row in rows)
            //    {
            //        if (row.Index < startRowIndex)
            //        {
            //            continue;
            //        }
            //        for (int column = 0; column < columnCount; column++)
            //        {
            //            if (row.Index == startRowIndex && column < startColumnIndex)
            //            {
            //                continue;
            //            }
            //            object value = row.Cells[column].Value;
            //            if (value == null)
            //            {
            //                continue;
            //            }
            //            if (value.ToString().IndexOf(targetText, compareOption) >= 0)
            //            {
            //                findCell = row.Cells[column];
            //                break;
            //            }
            //        }
            //        if (findCell != null)
            //        {
            //            break;
            //        }
            //    }
            //    if (findCell == null)
            //    {
            //        MessageBox.Show(this, "Not found.", Application.ProductName);
            //        beforeFind = false;
            //    }
            //    else
            //    {
            //        _gridControl.GridView.CurrentCell = findCell;
            //        beforeFind = true;
            //    }
            //}
        }

        private void _copyMenuItem_Click(object sender, EventArgs e)
        {
            ProgramUtility.CallEditAction(GetSelectedDataGridView(), ProgramUtility.CallEditActionModes.COPY);
        }

        private void _selectAllMenuItem_Click(object sender, EventArgs e)
        {
            ProgramUtility.CallEditAction(GetSelectedDataGridView(), ProgramUtility.CallEditActionModes.SELECT_ALL);
        }

    }
}