using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using Microsoft.Win32;
using System.IO;
using System.Text.RegularExpressions;


namespace Oratorio
{
	public partial class ConnectDetailForm : Form
	{
		public enum ConnectMethods
		{
			OdbcGeneric,
			OdbcOracle,
            OdbcSqlServer,
            OdbcMySQL,
        }

		public ConnectDetailForm()
		{
			InitializeComponent();
		}

		protected ConnectSettingData _setting = new ConnectSettingData();

		public ConnectSettingData Setting
		{
			get { return _setting; }
			set { _setting = value; }
		}
	
		private void ConnectDetailForm_Load(object sender, EventArgs e)
		{
            Control[] children = 
			{
                this,
                _settingNameLabel,
                _userIdLabel,
                _passwordLabel,
                _donotSaveCertCheck,
                _dataSourceLabel,
                _odbcDriverLabel,
                _serverLabel,
                _portLabel,
                _databaseLabel,
                _optionsLabel,
                _saveButton,
                _cancelButton,
			};
            ProgramUtility.LoadControlStringsFromResource(children, GetType().FullName);

            _settingNameText.Text = _setting.SettingName;
            _userIdText.Text = _setting.Uid;
            _passwordText.Text = _setting.Pwd;
            _donotSaveCertCheck.Checked = _setting.IsUnsaveCert();

            if (_setting.SettingMethod == "" || _setting.SettingMethod == ConnectSettingData.METHOD_ODBC_GENERAL)
            {
                _connectMethodTabPanel.SelectedIndex = (int)ConnectMethods.OdbcGeneric;
            }
            else if (_setting.SettingMethod == ConnectSettingData.METHOD_ODBC_ORACLE)
            {
                _connectMethodTabPanel.SelectedIndex = (int)ConnectMethods.OdbcOracle;
            }
            else if (_setting.SettingMethod == ConnectSettingData.METHOD_ODBC_SQL_SERVER)
            {
                _connectMethodTabPanel.SelectedIndex = (int)ConnectMethods.OdbcSqlServer;
            }
            else if (_setting.SettingMethod == ConnectSettingData.METHOD_ODBC_MY_SQL)
            {
                _connectMethodTabPanel.SelectedIndex = (int)ConnectMethods.OdbcMySQL;
            }
            else
            {
                System.Diagnostics.Debug.Assert(false);
                throw new NotSupportedException();
            }
            this.RefreshTabPanel();
            if (_settingNameText.Text.Length == 0)
            {
                _saveButton.Enabled = false;
            }
        }

        protected void ShowValidateError(string message)
        {
            MessageBox.Show(this, message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Warning);
        }

        protected static List<string> GetOracleTnsList()
        {
            List<string> resultList = new List<string>();

            const string oracleHomePath = @"SOFTWARE\ORACLE\HOME0";
            RegistryKey regKey = Registry.LocalMachine.OpenSubKey(oracleHomePath);
            if (regKey == null)
            {
                return null;
            }
            try
            {
                string oracleHome = (regKey.GetValue("ORACLE_HOME") != null ? regKey.GetValue("ORACLE_HOME").ToString() : null);
                if (oracleHome == null)
                {
                    return null;
                }
                string oracleNetAdmin = Path.Combine(oracleHome, @"network\ADMIN");
                string oracleTnsNamesPath = Path.Combine(oracleNetAdmin, @"tnsnames.ora");
                if (!Directory.Exists(oracleNetAdmin) || !File.Exists(oracleTnsNamesPath))
                {
                    return null;
                }
                string tnsFileText = File.ReadAllText(oracleTnsNamesPath);
                Regex regex = new Regex("^([a-zA-Z0-9_]+)[ ]*=[ ]*", RegexOptions.Multiline);
                for (int begin = 0; begin < tnsFileText.Length; )
                {
                    Match match = regex.Match(tnsFileText, begin);
                    if (!match.Success)
                    {
                        break;
                    }
                    string foundName = match.Groups[1].Value;
                    resultList.Add(foundName);
                    begin = match.Index + match.Length;
                }
            }
            finally
            {
                regKey.Close();
            }
            return resultList;
        }

        protected void RefreshTabPanel()
        {
            int selectedIndex = _connectMethodTabPanel.SelectedIndex;
            if (selectedIndex == (int)ConnectMethods.OdbcGeneric)
            {
                UpdateDataSourceCombo(_dataSourceCombo, null);
                UpdateOdbcDriverCombo(_odbcDriverCombo, null);
                _dataSourceCombo.Text = _setting.Dsn;
                _odbcDriverCombo.Text = _setting.Driver;
                _serverText.Text = _setting.Server;
                _portText.Text = _setting.Port;
                _databaseText.Text = _setting.Database;
                _optionsText.Text = _setting.SettingOptions;
            }
            else if (selectedIndex == (int)ConnectMethods.OdbcOracle)
            {
                UpdateOdbcDriverCombo(_oracleOdbcDriverCombo, "ORACLE");
                _oracleOdbcDriverCombo.Text = _setting.Driver;
                _tnsNameCombo.Text = _setting.Dbq;
                if (_tnsNameCombo.Items.Count == 0)
                {
                    _tnsNameCombo.Items.Add("");
                    List<string> tnsList = GetOracleTnsList();
                    if(tnsList != null)
                    {
                        tnsList.Sort();
                        foreach (string tnsName in tnsList)
                        {
                            _tnsNameCombo.Items.Add(tnsName);
                        }
                    }
                }
            }
            else if (selectedIndex == (int)ConnectMethods.OdbcSqlServer)
            {
                UpdateOdbcDriverCombo(_sqlsvrOdbcDriverCombo, "SQL Server");
                _sqlsvrOdbcDriverCombo.Text = _setting.Driver;
                _sqlsvrServerText.Text = _setting.Server;
                //                _sqlsvrPortText.Text = _setting.Port;
                _sqlsvrDatabaseText.Text = _setting.Database;
            }
            else if (selectedIndex == (int)ConnectMethods.OdbcMySQL)
            {
                UpdateOdbcDriverCombo(_mysqlOdbcDriverCombo, "MySQL");
                _mysqlOdbcDriverCombo.Text = _setting.Driver;
                _mysqlServerText.Text = _setting.Server;
                _mysqlPortText.Text = _setting.Port;
                _mysqlDatabaseText.Text = _setting.Database;
            }
            else
            {
                System.Diagnostics.Debug.Assert(false);
                throw new NotSupportedException();
            }
        }

        protected bool ValidateControl()
        {
            _setting.Clear();
            _setting.SettingName = _settingNameText.Text;
            _setting.Uid = _userIdText.Text;
            _setting.Pwd = _passwordText.Text;
            _setting.SettingUnsaveCert = (_donotSaveCertCheck.Checked ? "1" : "0");

            if (_connectMethodTabPanel.SelectedIndex == (int)ConnectMethods.OdbcGeneric)
            {
                _setting.SettingMethod = ConnectSettingData.METHOD_ODBC_GENERAL;
                _setting.Dsn = _dataSourceCombo.Text;
                _setting.Driver = _odbcDriverCombo.Text;
                _setting.Server = _serverText.Text;
                _setting.Port = _portText.Text;
                _setting.Database = _databaseText.Text;
                _setting.SettingOptions = _optionsText.Text;

            }
            else if (_connectMethodTabPanel.SelectedIndex == (int)ConnectMethods.OdbcOracle)
            {
                _setting.SettingMethod = ConnectSettingData.METHOD_ODBC_ORACLE;
                _setting.Driver = _oracleOdbcDriverCombo.Text;
                _setting.Dbq = _tnsNameCombo.Text;
                if (_setting.Driver.Length == 0)
                {
                    ShowValidateError("Select ODBC Driver.");
                    base.ActiveControl = _oracleOdbcDriverCombo;
                    return false;
                }
                if (_setting.Dbq.Length == 0)
                {
                    ShowValidateError("Select TNS name.");
                    base.ActiveControl = _tnsNameCombo;
                    return false;
                }
            }
            else if (_connectMethodTabPanel.SelectedIndex == (int)ConnectMethods.OdbcSqlServer)
            {
                _setting.SettingMethod = ConnectSettingData.METHOD_ODBC_SQL_SERVER;
                _setting.Driver = _sqlsvrOdbcDriverCombo.Text;

                _setting.Server = _sqlsvrServerText.Text;
//                _setting.Port = _sqlsvrPortText.Text;
                _setting.Database = _sqlsvrDatabaseText.Text;

                if (_setting.Driver.Length == 0)
                {
                    ShowValidateError("Select ODBC Driver.");
                    base.ActiveControl = _sqlsvrOdbcDriverCombo;
                    return false;
                }
            }
            else if (_connectMethodTabPanel.SelectedIndex == (int)ConnectMethods.OdbcMySQL)
            {
                _setting.SettingMethod = ConnectSettingData.METHOD_ODBC_MY_SQL;
                _setting.Driver = _mysqlOdbcDriverCombo.Text;

                _setting.Server = _mysqlServerText.Text;
                _setting.Port = _mysqlPortText.Text;
                _setting.Database = _mysqlDatabaseText.Text;

                if (_setting.Driver.Length == 0)
                {
                    ShowValidateError("Select ODBC Driver.");
                    base.ActiveControl = _mysqlOdbcDriverCombo;
                    return false;
                }
            }
            else
            {
                System.Diagnostics.Debug.Assert(false);
                throw new NotSupportedException();
            }
            return true;
        }

        private void _saveButton_Click(object sender, EventArgs e)
		{

            if (!ValidateControl())
            {
                return;
            }

            DialogResult = DialogResult.OK;
		}

        //protected static List<string> GetOdbcDriverList()
        //{
        //    List<string> resultList = new List<string>();
        //    const string sOdbcRegPath = "SOFTWARE\\ODBC\\ODBCINST.INI\\ODBC Drivers";
        //    RegistryKey regKey = Registry.LocalMachine.OpenSubKey(sOdbcRegPath);
        //    if (regKey != null)
        //    {
        //        try
        //        {
        //            foreach (string sName in regKey.GetValueNames())
        //            {
        //                resultList.Add(sName);
        //            }
        //        }
        //        finally
        //        {
        //            regKey.Close();
        //        }
        //    }
        //    return resultList;
        //}

        protected static void UpdateOdbcDriverCombo(ComboBox combo, string filter)
        {
            if (combo.Items.Count > 0)
            {
                return;
            }
            combo.Items.Clear();
            combo.Items.Add("");
            const string sOdbcRegPath = "SOFTWARE\\ODBC\\ODBCINST.INI\\ODBC Drivers";
            RegistryKey regKey = Registry.LocalMachine.OpenSubKey(sOdbcRegPath);
            if (regKey != null)
            {
                try
                {
                    foreach (string sName in regKey.GetValueNames())
                    {
                        if (filter == null || sName.IndexOf(filter, StringComparison.OrdinalIgnoreCase) >= 0)
                        {
                            combo.Items.Add(sName);
                        }
                    }
                }
                finally
                {
                    regKey.Close();
                }
            }
        }
        protected static void UpdateDataSourceCombo(ComboBox combo, string filter)
        {
            if (combo.Items.Count > 0)
            {
                return;
            }
            combo.Items.Clear();
            combo.Items.Add("");
            const string sUserDsnRegPath = "Software\\ODBC\\ODBC.INI\\ODBC Data Sources";
            RegistryKey regKey = Registry.CurrentUser.OpenSubKey(sUserDsnRegPath);
            if (regKey != null)
            {
                try
                {
                    foreach (string sName in regKey.GetValueNames())
                    {
                        combo.Items.Add(sName);
                    }
                }
                finally
                {
                    regKey.Close();
                }
            }
        }
        private void _connectMethodTabPanel_SelectedIndexChanged(object sender, EventArgs e)
		{
            RefreshTabPanel();
		}

        private void _settingNameText_TextChanged(object sender, EventArgs e)
        {
            _saveButton.Enabled = (_settingNameText.Text.Length > 0) ;
        }
	}
}