using System;
using System.Collections.Generic;
using System.Text;
using System.Data;
using System.Data.Common;

namespace Oratorio
{
    public class DatabaseContextCommon : DatabaseContext, IDisposable
    {
		public enum ProductTypes
		{
			Unknown,
            Odbc,
            Oracle,
            MsSqlServer,
			Postgres,
			MySql,
			//not supported
//            SQLite,
//            Access,
//            Excel,
//            DB2,
        }

        public static readonly string META_DATA_TABLE = System.Data.Odbc.OdbcMetaDataCollectionNames.Tables;
        public static readonly string META_DATA_VIEWS = System.Data.Odbc.OdbcMetaDataCollectionNames.Views;
        public static readonly string META_DATA_COLUMNS = System.Data.Odbc.OdbcMetaDataCollectionNames.Columns;
        public static readonly string META_DATA_INDEX = System.Data.Odbc.OdbcMetaDataCollectionNames.Indexes;
        public static readonly string META_DATA_PROCEDURES = System.Data.Odbc.OdbcMetaDataCollectionNames.Procedures;

        protected DbConnection _connection;
        protected DbProviderFactory _factory;
        protected bool _connected;
        protected DatabaseScriptGenerator _scriptGenerator;
        protected ProductTypes _productType = ProductTypes.Unknown;
        protected string _currentSchemaName;
        protected string _currentCatalogName;
        protected string _currentDatabaseName;

        public DatabaseScriptGenerator ScriptGenerator
        {
            get { return _scriptGenerator; }
        }

        public ProductTypes ProductType
        {
            get { return _productType; }
        }

        public int ProductTypeNumber
        {
            get { return (int)_productType; }
        }

        public bool Connected
		{
			get { return _connected; }
		}

        public DbProviderFactory Factory
        {
            get { return _factory; }
        }

        public DbConnection Connection
		{
			get { return _connection; }
		}

        public virtual string CurrentSchemaName
        {
            get { return _currentSchemaName; }
        }

        public virtual string CurrentCatalogName
        {
            get { return _currentCatalogName; }
        }

        public virtual string CurrentDatabaseName
        {
            get { return _currentDatabaseName; }
        }


        public virtual string ProductName
        {
            get { return ""; }
        }

        public virtual bool AllowDatabaseChange
        {
            get { return false; }
        }

        protected DatabaseContextCommon(DbConnection conn)
        {
            if (conn == null)
            {
                throw new ArgumentNullException();
            }
            _connection = conn;
            _connected = true;
        }

        public virtual void Close()
        {
            if (_connection != null)
            {
                _connection.Close();
                _connection.Dispose();
            }
            _currentSchemaName = "";
            _connected = false;
        }

        public virtual string[] GetSchemaList()
		{
            throw new NotSupportedException();
		}

        public virtual string GetCurrentSchema()
        {
            throw new NotSupportedException();
        }

        public virtual DbDataAdapter CreateDataAdapter()
		{
            if (_factory == null)
            {
                throw new NotSupportedException();
            }
            return _factory.CreateDataAdapter();
		}

        public virtual DbCommand CreateCommand(string sql)
        {
            if (_factory == null)
            {
                throw new NotSupportedException();
            }
            //            DbCommand command = _factory.CreateCommand();
            DbCommand command = _connection.CreateCommand();
            command.CommandText = sql;
            return command;
        }

        public virtual DbCommandBuilder CreateCommandBuilder()
        {
            if (_factory == null)
            {
                throw new NotSupportedException();
            }
            return _factory.CreateCommandBuilder();
        }

        public virtual DbParameter CreateParameter(string name, object value)
        {
            if (_factory == null)
            {
                throw new NotSupportedException();
            }
            DbParameter param = _factory.CreateParameter();
            param.ParameterName = name;
            param.Value = value;
            return param;
        }

        public virtual DatabaseTable[] GetTableList2(string catalog, string schema)
        {
            string[] aFilter = new string[] { catalog, schema };
            DataTable tableInfos = null;
            try
            {
                tableInfos = Connection.GetSchema(META_DATA_TABLE, aFilter);
            }
            catch (InvalidOperationException ie)
            {
                throw new DatabaseContextException("Get table information failed.", ie);
            }
#if DEBUG
            ProgramUtility.WriteLine("--------------------------");
            for (int nColumn = 0; nColumn < tableInfos.Columns.Count; nColumn++)
            {
                ProgramUtility.WriteLine(tableInfos.Columns[nColumn].ColumnName);
            }
            ProgramUtility.WriteLine("--------------------------");
#endif
            List<DatabaseTable> tableList = new List<DatabaseTable>();
            foreach (DataRow row in tableInfos.Rows)
            {
                string sTableName = row["TABLE_NAME"].ToString();
                if (sTableName == null || sTableName.Length == 0)
                {
                    continue;
                }
                
                DatabaseTable tableObj = new DatabaseTable();
                tableObj.Name = sTableName;
                tableObj.Schema = (row["TABLE_SCHEM"] == null? "": row["TABLE_SCHEM"].ToString());
                tableList.Add(tableObj);
            }
            return tableList.ToArray();
        }

        public virtual string[] GetTableList(string catalog, string schema)
        {
            string[] aFilter = new string[] { catalog, schema };
            DataTable tableInfos = null;
			try
			{
				tableInfos = Connection.GetSchema(META_DATA_TABLE, aFilter);
			}
			catch (InvalidOperationException ie)
			{
				throw new DatabaseContextException("Get table information failed.", ie);
			}
#if DEBUG
            ProgramUtility.WriteLine("--------------------------");
            for (int nColumn = 0; nColumn < tableInfos.Columns.Count; nColumn++)
            {
                ProgramUtility.WriteLine(tableInfos.Columns[nColumn].ColumnName);
            }
            ProgramUtility.WriteLine("--------------------------");
#endif
            List<string> tableList = new List<string>();
            foreach (DataRow row in tableInfos.Rows)
            {
                string sTableName = row["TABLE_NAME"].ToString();
                if (sTableName.Length == 0)
                {
                    continue;
                }
                /*
                ProgramUtility.WriteLine(row[0].ToString());
                ProgramUtility.WriteLine(row[1].ToString());
                ProgramUtility.WriteLine(row[2].ToString());
                ProgramUtility.WriteLine(row[3].ToString());
                ProgramUtility.WriteLine(row[4].ToString());
                 */
                tableList.Add(sTableName);
            }
            return tableList.ToArray();
        }

        public virtual DatabaseObjectIndex[] GetIndexList(string catalogFilter, string schemaFilter, string objectFilter)
        {
            string[] aFilter = new string[] { catalogFilter, schemaFilter, objectFilter };

            DataTable tableInfos = Connection.GetSchema(META_DATA_INDEX, aFilter);
            List<DatabaseObjectIndex> indexList = new List<DatabaseObjectIndex>();
            DatabaseObjectIndex indexObject = null;
            foreach (DataRow row in tableInfos.Rows)
            {
#if DEBUG
                //ProgramUtility.WriteLine("");
                //for (int nColumn = 0; nColumn < tableInfos.Columns.Count; nColumn++)
                //{
                //    ProgramUtility.WriteLine(tableInfos.Columns[nColumn].ColumnName + "=" + row[nColumn]);
                //}
#endif
                string catalogName = row["TABLE_CAT"].ToString();
                string schemaName = row["TABLE_SCHEM"].ToString();
                string tableName = row["TABLE_NAME"].ToString();
                string nonUnique = row["NON_UNIQUE"].ToString();
                string indexName = row["INDEX_NAME"].ToString();
                string columnName = row["COLUMN_NAME"].ToString();
                
                if (indexObject == null || indexObject.ObjectName != indexName)
                {
                    indexObject = new DatabaseObjectIndex(catalogName, schemaName, indexName);
                    indexObject.ObjectName = indexName;
                    indexObject.IsUnique = (nonUnique != "1");
                    indexList.Add(indexObject);
                }
                DatabaseColumn column = new DatabaseColumn();
                column.ColumnName = columnName;
                column.Index = indexObject.ColumnList.Count;
                indexObject.ColumnList.Add(column);
            }
            return indexList.ToArray();
        }

        public virtual string[] GetViewList(string catalog, string schema)
        {
            string[] aFilter = new string[] { catalog, schema };
            DataTable tableInfos = Connection.GetSchema(META_DATA_VIEWS, aFilter);
            List<string> tableList = new List<string>();
            foreach (DataRow row in tableInfos.Rows)
            {
                string sTableName = row["TABLE_NAME"].ToString();
                if (sTableName.Length == 0)
                {
                    continue;
                }
                //ProgramUtility.WriteLine("");
                //for (int nColumn = 0; nColumn < tableInfos.Columns.Count; nColumn++)
                //{
                //    ProgramUtility.WriteLine(tableInfos.Columns[nColumn].ColumnName + "=" + row[nColumn]);
                //}
                /*
                ProgramUtility.WriteLine(row[0].ToString());
                ProgramUtility.WriteLine(row[1].ToString());
                ProgramUtility.WriteLine(row[2].ToString());
                ProgramUtility.WriteLine(row[3].ToString());
                ProgramUtility.WriteLine(row[4].ToString());
                 */
                tableList.Add(sTableName);
            }
            return tableList.ToArray();
        }

        public virtual void GetTableColumnList(string catalog, string schema, string tableName, List<DatabaseColumn> list)
        {
            string[] sFilter = { catalog, schema, tableName };
            DataTable tableInfos = null;
            try
            {
                tableInfos = Connection.GetSchema(META_DATA_COLUMNS, sFilter);
            }
            catch (DbException oe)
            {
                throw new DatabaseContextException("Get schema information failed.", oe);
            }
            int nColumnCount = tableInfos.Columns.Count;
            int index = 0;
            foreach (DataRow row in tableInfos.Rows)
            {
#if DEBUG
                //ProgramUtility.WriteLine(GetType().FullName);
                //ProgramUtility.WriteLine("--------------------------");
                //for (int nColumn = 0; nColumn < nColumnCount; nColumn++)
                //{
                //    ProgramUtility.WriteLine(tableInfos.Columns[nColumn].ColumnName + "=" + row[nColumn]);
                //}
#endif
                DatabaseColumn column = new DatabaseColumn();
                column.Index = index++;
                column.IsNullable = (row["IS_NULLABLE"].ToString() == "YES");
                column.ColumnName = row["COLUMN_NAME"].ToString();
                column.TypeName = row["TYPE_NAME"].ToString();
                string typeAndSize = column.TypeName;
                object charLength = row["CHAR_OCTET_LENGTH"];
                if (charLength != null && charLength.ToString().Length > 0 && charLength.ToString() != "0")
                {
                    typeAndSize += "(" + charLength + ")";
                }
                if (IsDateType(column.TypeName))
                {
                    column.IsDate = true;
                }
                else if (IsNumberType(column.TypeName))
                {
                    column.IsNumber = true;
                }
                //object numRadix = row["NUM_PREC_RADIX"];
				//if (numRadix != null && numRadix.ToString().Length > 0 && numRadix.ToString() != "0")
				//{
				//    //TODO
				//    typeAndSize += "(" + row["COLUMN_SIZE"] + ")";
				//}
                column.TypeAndSize = typeAndSize;
                list.Add(column);
            }
        }

        public virtual bool IsDateType(string typeName)
        {
            return false;
        }
        public virtual bool IsNumberType(string typeName)
        {
            return false;
        }

        public virtual string[] GetProcedureList(string catalog, string schema)
        {
            string[] aFilter = new string[] { catalog, schema };
            DataTable tableInfos = Connection.GetSchema(META_DATA_PROCEDURES, aFilter);
            List<string> tableList = new List<string>();
#if DEBUG
            ProgramUtility.WriteLine("GetProcedureList:");
            for (int nColumn = 0; nColumn < tableInfos.Columns.Count; nColumn++)
            {
                ProgramUtility.WriteLine("  column=" + tableInfos.Columns[nColumn].ColumnName);
            }
#endif
            foreach (DataRow row in tableInfos.Rows)
            {
                string sTableName = row["PROCEDURE_NAME"].ToString();
                //ProgramUtility.WriteLine("");
                //for (int nColumn = 0; nColumn < tableInfos.Columns.Count; nColumn++)
                //{
                //    ProgramUtility.WriteLine(tableInfos.Columns[nColumn].ColumnName + "=" + row[nColumn]);
                //}
                /*
                ProgramUtility.WriteLine(row[0].ToString());
                ProgramUtility.WriteLine(row[1].ToString());
                ProgramUtility.WriteLine(row[2].ToString());
                ProgramUtility.WriteLine(row[3].ToString());
                ProgramUtility.WriteLine(row[4].ToString());
                 */
                tableList.Add(sTableName);
            }
            return tableList.ToArray();
        }

        public virtual string[] GetDatabaseList()
        {
            return new string[0];
        }

        public virtual void ChangeDatabase(string databaseName)
        {
            throw new NotSupportedException();
        }


        #region IDisposable o

		public virtual void Dispose()
		{
            Close();
            _connection = null;
		}

		#endregion

    }
}
