using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Data.Common;
using System.IO;
using Microsoft.VisualBasic.FileIO;

namespace Oratorio
{
	public partial class ObjectTreeControl : UserControl
	{
		private DatabaseContext _database;

		public DatabaseContext Database
		{
			get { return _database; }
			set { _database = value; }
		}
	

		public TreeView TreeView
		{
			get { return this._treeView; }
		}
	
		public ObjectTreeControl()
		{
			InitializeComponent();
		}

        public MainForm MainForm
        {
            get { return (MainForm)Parent; }
        }

        private void ObjectTreeControl_Load(object sender, EventArgs e)
        {
            TreeView treeView = _treeView;
            treeView.ImageList = new ImageList();
            treeView.ImageList.Images.Add(Properties.Resources.ObjectUnknown);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectDatabase);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);

            ProgramUtility.LoadMenuStringsFromResource(_contextMenuStrip.Items, GetType().FullName);
        }

        public void RefreshNode()
        {
            TreeView treeView = this.TreeView;

            MainForm.OutputScriptLogLine("-- #Update object tree...", Color.Blue);
            treeView.Nodes.Clear();

            TreeNode databaseNode = new TreeNode("Database [" + _database.CurrentDatabaseName + "]");
            databaseNode.SelectedImageIndex = databaseNode.ImageIndex = (int)ObjectTreeData.NodeTypes.DATABASE;
            databaseNode.Tag = new ObjectTreeData(ObjectTreeData.NodeTypes.DATABASE);

            treeView.Nodes.Add(databaseNode);

            TreeNode currentTablesNode = new TreeNode("Tables");
            ObjectTreeData currentTablesData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE_LIST);
            currentTablesData.Catalog = _database.CurrentCatalogName;
            currentTablesData.Schema = _database.CurrentSchemaName;
            currentTablesNode.Tag = currentTablesData;
            currentTablesNode.SelectedImageIndex = currentTablesNode.ImageIndex = (int)ObjectTreeData.NodeTypes.TABLE_LIST;
            currentTablesNode.Nodes.Add(ObjectTreeData.CreateReloadNode());

            databaseNode.Nodes.Add(currentTablesNode);

            TreeNode currentViewsNode = new TreeNode("Views");
            ObjectTreeData currentViewsData = new ObjectTreeData(ObjectTreeData.NodeTypes.VIEW_LIST);
            currentViewsData.Catalog = _database.CurrentCatalogName;
            currentViewsData.Schema = _database.CurrentSchemaName;
            currentViewsNode.Tag = currentViewsData;
            currentViewsNode.SelectedImageIndex = currentViewsNode.ImageIndex = (int)ObjectTreeData.NodeTypes.VIEW_LIST;
            currentViewsNode.Nodes.Add(ObjectTreeData.CreateReloadNode());

            databaseNode.Nodes.Add(currentViewsNode);

            TreeNode currentProceduresNode = new TreeNode("Procedures");
            ObjectTreeData currentProceduresData = new ObjectTreeData(ObjectTreeData.NodeTypes.PROCEDURE_LIST);
            currentProceduresData.Catalog = _database.CurrentCatalogName;
            currentProceduresData.Schema = _database.CurrentSchemaName;
            currentProceduresNode.Tag = currentProceduresData;
            currentProceduresNode.SelectedImageIndex = currentProceduresNode.ImageIndex = (int)ObjectTreeData.NodeTypes.PROCEDURE_LIST;
            currentProceduresNode.Nodes.Add(ObjectTreeData.CreateReloadNode());

            databaseNode.Nodes.Add(currentProceduresNode);

            TreeNode schemaListNode = new TreeNode("Schemas");
            ObjectTreeData schemaListData = new ObjectTreeData(ObjectTreeData.NodeTypes.SCHEMA_LIST);
            schemaListNode.Tag = schemaListData;
            schemaListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
            schemaListNode.SelectedImageIndex = schemaListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.TABLE_LIST;
            databaseNode.Nodes.Add(schemaListNode);

            databaseNode.Expand();
            currentTablesNode.Expand();
            treeView.SelectedNode = databaseNode;

            MainForm.OutputScriptLogLine("-- #Update object tree success.", Color.Blue);
        }

        private void _treeView_BeforeExpand(object sender, TreeViewCancelEventArgs e)
		{
			TreeNode node = e.Node;
            ObjectTreeData data = GetNodeData(node);

			if (data.Type == ObjectTreeData.NodeTypes.DATABASE)
			{
				return;
			}
            else if (data.Type == ObjectTreeData.NodeTypes.TABLE_LIST)
			{
				TreeNodeCollection targetNodes = node.Nodes;
				if (targetNodes.Count == 1 && ObjectTreeData.IsReloadNode(targetNodes[0]))
				{
					string sTargetCatalog = data.Catalog;
					string sTargetSchema = data.Schema;
                    //string[] aTableList = _database.GetTableList(sTargetCatalog, sTargetSchema);
                    //targetNodes.Clear();
                    //foreach (string sTableName in aTableList)
                    //{
                    //    TreeNode tableNode = new TreeNode(sTableName);
                    //    ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE);
                    //    tableData.Catalog = sTargetCatalog;
                    //    tableData.Schema = data.Schema;
                    //    tableData.Name = sTableName;
                    //    tableNode.Tag = tableData;
                    //    targetNodes.Add(tableNode);
                    //}
                    DatabaseTable[] aTableList = _database.GetTableList2(sTargetCatalog, sTargetSchema);
                    targetNodes.Clear();
                    foreach (DatabaseTable tableObj in aTableList)
                    {
                        string tableNameShow = tableObj.Name;
                        if (tableObj.Schema != sTargetSchema && tableObj.Schema.Length > 0)
                        {
                            tableNameShow = tableObj.Schema + "." + tableObj.Name;
                        }
                        TreeNode tableNode = new TreeNode(tableNameShow);
                        ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE);
                        tableData.Catalog = sTargetCatalog;
                        tableData.Schema = tableObj.Schema;
                        tableData.Name = tableObj.Name;
                        tableNode.Tag = tableData;
                        targetNodes.Add(tableNode);
                    }
                }
			}
            else if (data.Type == ObjectTreeData.NodeTypes.PROCEDURE_LIST)
            {
                TreeNodeCollection targetNodes = node.Nodes;
                if (ObjectTreeData.IsReloadNode(targetNodes))
                {
                    string sTargetCatalog = data.Catalog;
                    string sTargetSchema = data.Schema;
                    string[] aTableList = _database.GetProcedureList(sTargetCatalog, sTargetSchema);
                    targetNodes.Clear();
                    foreach (string sTableName in aTableList)
                    {
                        TreeNode tableNode = new TreeNode(sTableName);
                        ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.PROCEDURE);
                        tableData.Catalog = sTargetCatalog;
                        tableData.Schema = data.Schema;
                        tableData.Name = sTableName;
                        tableNode.Tag = tableData;
                        targetNodes.Add(tableNode);
                    }
                }
            }
            else if (data.Type == ObjectTreeData.NodeTypes.VIEW_LIST)
            {
                TreeNodeCollection targetNodes = node.Nodes;
                if (targetNodes.Count == 1 && ObjectTreeData.IsReloadNode(targetNodes[0]))
                {
                    string sTargetCatalog = data.Catalog;
                    string sTargetSchema = data.Schema;
                    string[] aTableList = _database.GetViewList(sTargetCatalog, sTargetSchema);
                    targetNodes.Clear();
                    foreach (string sTableName in aTableList)
                    {
                        TreeNode tableNode = new TreeNode(sTableName);
                        ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.VIEW);
                        tableData.Catalog = sTargetCatalog;
                        tableData.Schema = data.Schema;
                        tableData.Name = sTableName;
                        tableNode.Tag = tableData;
                        targetNodes.Add(tableNode);
                    }
                }
            }
            else if (data.Type == ObjectTreeData.NodeTypes.SCHEMA_LIST)
			{
				TreeNodeCollection targetNodes = node.Nodes;
				if (targetNodes.Count == 1 && ObjectTreeData.IsReloadNode(targetNodes[0]))
				{
					string sTargetDatabase = _database.CurrentCatalogName;
					string[] aSchemaList = null;
					try
					{
						aSchemaList = _database.GetSchemaList();
					}
					catch (DatabaseContextException dce)
					{
						ProgramUtility.ShowError(dce);
						e.Cancel = true;
						return;
					}
					catch (NotSupportedException nse)
					{
						ProgramUtility.ShowError(nse);
						e.Cancel = true;
						return;
					}
					targetNodes.Clear();
					foreach (string schemaName in aSchemaList)
					{
						TreeNode schemaNode = new TreeNode(schemaName);
						ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.SCHEMA);
						tableData.Catalog = _database.CurrentCatalogName;
						tableData.Schema = schemaName;
						schemaNode.Tag = tableData;
                        schemaNode.ImageIndex = (int)ObjectTreeData.NodeTypes.SCHEMA_LIST;
                        schemaNode.SelectedImageIndex = schemaNode.ImageIndex;
                        targetNodes.Add(schemaNode);

                        TreeNode tableListNode = new TreeNode("Tables");
						ObjectTreeData tableListData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE_LIST);
						tableListData.Catalog = _database.CurrentCatalogName;
						tableListData.Schema = schemaName;
                        tableListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.TABLE_LIST;
                        tableListNode.SelectedImageIndex = tableListNode.ImageIndex;
                        tableListNode.Tag = tableListData;
						tableListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
						schemaNode.Nodes.Add(tableListNode);

                        TreeNode viewListNode = new TreeNode("Views");
                        ObjectTreeData viewListData = new ObjectTreeData(ObjectTreeData.NodeTypes.VIEW_LIST);
                        viewListData.Catalog = _database.CurrentCatalogName;
                        viewListData.Schema = schemaName;
                        viewListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.VIEW_LIST;
                        viewListNode.SelectedImageIndex = viewListNode.ImageIndex;
                        viewListNode.Tag = viewListData;
                        viewListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
                        schemaNode.Nodes.Add(viewListNode);

                        TreeNode procedureListNode = new TreeNode("Procedure");
                        ObjectTreeData procedureListData = new ObjectTreeData(ObjectTreeData.NodeTypes.PROCEDURE_LIST);
                        procedureListData.Catalog = _database.CurrentCatalogName;
                        procedureListData.Schema = schemaName;
                        procedureListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.PROCEDURE_LIST;
                        procedureListNode.SelectedImageIndex = procedureListNode.ImageIndex;
                        procedureListNode.Tag = procedureListData;
                        procedureListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
                        schemaNode.Nodes.Add(procedureListNode);

                    }
				}
			}
		}

		private void _treeView_NodeMouseDoubleClick(object sender, TreeNodeMouseClickEventArgs e)
		{
			MainForm mainForm = (MainForm)Parent;
			TreeNode node = e.Node;
			ObjectTreeData data = GetNodeData(node);
            if (data.Type == ObjectTreeData.NodeTypes.TABLE || data.Type == ObjectTreeData.NodeTypes.VIEW)
			{
				ScriptForm form = mainForm.CreateGridForm(data.Catalog, data.Schema, data.Name, false);
                if (form != null)
                {
                    form.GridCtrl.Focus();
                }
			}
            else if (data.Type == ObjectTreeData.NodeTypes.PROCEDURE)
            {
                string script = null;
                try
                {
                    script = _database.ScriptGenerator.GetCreateProcedureScript(data.Catalog, data.Schema, data.Name);
                }
                catch (DatabaseContextException de)
                {
                    ProgramUtility.ShowError(de);
                    return;
                }
                ScriptForm form = MainForm.CreateEditForm();
                if (form != null)
                {
                    form.EditControl.TextBox.Text = script;
                    form.GridCtrl.Focus();
                }
            }
		}

        private void _treeView_NodeMouseClick(object sender, TreeNodeMouseClickEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                TreeNode node = e.Node;
                _treeView.SelectedNode = node;

                ObjectTreeData data = GetNodeData(node);
                switch(data.Type)
                {
                    case ObjectTreeData.NodeTypes.TABLE:
                    case ObjectTreeData.NodeTypes.VIEW:
                    case ObjectTreeData.NodeTypes.DATABASE:
                    case ObjectTreeData.NodeTypes.PROCEDURE:
                        _contextMenuStrip.Show(_treeView, e.X, e.Y);
                        break;
                }
            }
        }

        private void _openMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string sTableName = node.Text;
            MainForm mainForm = (MainForm)Parent;
            ScriptForm form = mainForm.CreateGridForm(data.Catalog, data.Schema, sTableName, false);
            if (form != null)
            {
                form.GridCtrl.Focus();
            }
        }

        private void _selectMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();

        }

        private void _createScriptMenu_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
			string script = "";
            try
            {
                if (data.NodeType == ObjectTreeData.NodeTypes.TABLE)
                {
                    script = _database.ScriptGenerator.GetCreateTableScript(data.Catalog, data.Schema, data.Name);
                }
                else if (data.NodeType == ObjectTreeData.NodeTypes.VIEW)
                {
                    script = _database.ScriptGenerator.GetCreateViewScript(data.Catalog, data.Schema, data.Name);
                }
                else if (data.NodeType == ObjectTreeData.NodeTypes.PROCEDURE)
                {
                    script = _database.ScriptGenerator.GetCreateProcedureScript(data.Catalog, data.Schema, data.Name);
                }
                else
                {
                    System.Diagnostics.Trace.Assert(false);
                    throw new NotSupportedException();
                }
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
			form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _copyNameMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            Clipboard.SetText(node.Text);
        }

        private void _selectAsteriskMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectAsteriskScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _selectWhereMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectWhereScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _newRecordMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string sTableName = node.Text;
            MainForm mainForm = (MainForm)Parent;
            ScriptForm form = mainForm.CreateGridForm(data.Catalog, data.Schema, sTableName, true);
            if (form != null)
            {
                form.GridCtrl.Focus();
            }
        }

        private void _updateScriptMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetUpdateScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _deleteScriptMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetDeleteScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        protected string GetDataReaderToCsv(IDataReader reader, string separator)
        {
            StringBuilder headBuffer = new StringBuilder("", 1024 * 1024);
            int columnCount = reader.FieldCount;
            for (int column = 0; column < columnCount; column++)
            {
                string name = reader.GetName(column);
                if (headBuffer.Length > 0)
                {
                    headBuffer.Append(separator);
                }
                if (name.IndexOf('\"') >= 0)
                {
                    name = name.Replace("\"", "\"\"");
                }
                headBuffer.Append('\"');
                headBuffer.Append('[');
                headBuffer.Append(name);
                headBuffer.Append(']');
                headBuffer.Append('\"');
            }
            headBuffer.Append("\r\n");
            StringBuilder buffer = new StringBuilder("", 1024 * 1024);
            while (reader.Read())
            {
                StringBuilder lineBuffer = new StringBuilder();
                for (int column = 0; column < columnCount; column++)
                {
                    object value = reader.GetValue(column);
                    if (column > 0)
                    {
                        lineBuffer.Append(separator);
                    }
                    if(value == null)
                    {
                        value = "";
                    }
                    string text = value.ToString();
                    if (text.IndexOf('\"') >= 0)
                    {
                        text = text.Replace("\"", "\"\"");
                    }
                    text = "\"" + text + "\"";
                    lineBuffer.Append(text);
                }
                lineBuffer.Append("\r\n");
                buffer.Append(lineBuffer);
            }
            String result = headBuffer.Append(buffer).ToString();
            return result;
        }

        protected string GetDataReaderToSql(IDataReader reader, string tableName)
        {
            StringBuilder scriptHead = new StringBuilder("");
            scriptHead.Append("INSERT INTO ");
            scriptHead.Append(tableName);
            scriptHead.Append("\r\n");
            scriptHead.Append("(");

            int columnCount = reader.FieldCount;
            for (int column = 0; column < columnCount; column++)
            {
                string name = reader.GetName(column);
                if (column > 0)
                {
                    scriptHead.Append(",");
                }
                scriptHead.Append("\r\n\t");
                scriptHead.Append(name);
            }
            scriptHead.Append("\r\n");
            scriptHead.Append(")\r\n");
            scriptHead.Append("VALUES\r\n");
            scriptHead.Append("(");

            StringBuilder buffer = new StringBuilder("", 1024 * 1024);
            while (reader.Read())
            {
                StringBuilder lineBuffer = new StringBuilder();
                for (int column = 0; column < columnCount; column++)
                {
                    object value = reader.GetValue(column);
                    if (column > 0)
                    {
                        lineBuffer.Append(",");
                    }
                    lineBuffer.Append("\r\n\t");
                    string text = value.ToString();
                    if (value == null || value == DBNull.Value)
                    {
                        text = "NULL";
                    }
                    else
                    {
                        if (text.IndexOf('\'') >= 0)
                        {
                            text = text.Replace("\'", "\'\'");
                        }
                        text = "\'" + text + "\'";
                    }
                    lineBuffer.Append(text);
                }
                lineBuffer.Append("\r\n");
                buffer.Append(scriptHead);
                buffer.Append(lineBuffer);
                buffer.Append(")\r\n;\r\n\r\n");
            }
            String result = buffer.ToString();
            return result;
        }

        /*
        private void _exportMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);

            string tableName = data.Name;

            ExportForm form = new ExportForm();
            if (form.ShowDialog(this) != DialogResult.OK)
                return;

            string folder = form.Folder;
            string where = form.Filter;
            string path = Path.Combine(folder, tableName);
            ExportForm.Formats format = (ExportForm.Formats)form.Format;
            switch(format)
            {
                case ExportForm.Formats.CSV:
                    path += ".csv";
                    break;
                case ExportForm.Formats.TSV:
                    path += ".tsv";
                    break;
                case ExportForm.Formats.SQL:
                    path += ".sql";
                    break;
                default:
                    throw new NotSupportedException();
            }
            if (File.Exists(path))
            {
                if (MessageBox.Show(this, "Over write file?\r\n" + Path.GetFileName(path), Application.ProductName, MessageBoxButtons.OKCancel) != DialogResult.OK)
                {
                    return;
                }
            }

            where = where.Trim();
            if(where.Length == 0 || where.Equals("WHERE", StringComparison.OrdinalIgnoreCase))
            {
                where = "";
            }
            string query = "SELECT * FROM " + tableName;
            if(where.Length > 0)
            {
                query += " " + where;
            }
            DatabaseContext database = _database;
            try
            {
                Cursor.Current = Cursors.WaitCursor;

                string exportText = "";
                using (DbCommand command = _database.CreateCommand(query))
                {
                    using (DbDataReader reader = command.ExecuteReader())
                    {
                        command.Dispose();
                        switch (format)
                        {
                            case ExportForm.Formats.CSV:
                                exportText = GetDataReaderToCsv(reader, ",");
                                break;
                            case ExportForm.Formats.TSV:
                                exportText = GetDataReaderToCsv(reader, "\t");
                                break;
                            case ExportForm.Formats.SQL:
                                exportText = GetDataReaderToSql(reader, tableName);
                                break;
                            default:
                                throw new NotSupportedException();
                        }
                    }
                }
                Encoding encoding = Encoding.Default;
                byte[] byteBuffer = encoding.GetBytes(exportText);
                using (FileStream stream = new FileStream(path, FileMode.Create, FileAccess.Write))
                {
                    stream.Write(byteBuffer, 0, byteBuffer.Length);
                }
            }
            catch (DbException dbe)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", dbe);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", de);
            }
            catch (IOException ioe)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", ioe);
            }
            finally
            {
                Cursor.Current = Cursors.Default;
            }

        }
        */
        protected ObjectTreeData GetNodeData(TreeNode node)
        {
            if (node == null)
            {
                System.Diagnostics.Debug.Assert(false);
                throw new ArgumentNullException("node");
            }
            ObjectTreeData data = node.Tag as ObjectTreeData;
            if (data == null)
            {
                System.Diagnostics.Debug.Assert(false);
                throw new ArgumentNullException("data");
            }
            return data;
        }

        private void _propertyMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);

            if (data.NodeType == ObjectTreeData.NodeTypes.TABLE
                    || data.NodeType == ObjectTreeData.NodeTypes.VIEW)
            {
                if (data.Schema == null)
                {
                    ProgramUtility.ShowError(MainForm, "Default schema is empty.", null);
                    return;
                }
                try
                {
                    MainForm.CreateTableDefineForm(data.Catalog, data.Schema, data.Name);
                }
                catch (ProgramCancelException)
                {
                }
            }
            else if (data.NodeType == ObjectTreeData.NodeTypes.DATABASE)
            {
                ObjectDetailForm detailForm = new ObjectDetailForm();

                detailForm.Property.Add("Object Type", "Database");
                detailForm.Property.Add("Product Name", Database.ProductName);
                detailForm.Property.Add("Data Source", Database.Connection.DataSource);
                detailForm.Property.Add("Server Version", Database.Connection.ServerVersion);
                detailForm.Property.Add("Database Name", Database.Connection.Database);
                if (Database.Connection is System.Data.Odbc.OdbcConnection)
                {
                    System.Data.Odbc.OdbcConnection odbcConnection = (System.Data.Odbc.OdbcConnection)Database.Connection;
                    detailForm.Property.Add("Driver", odbcConnection.Driver);
                }
                
//                detailForm.Property.Add("Type", "Database");
                detailForm.ShowDialog(this);
            }


        }

        private void _insertScriptMenuItem_Click(object sender, EventArgs e)
        {
            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetInsertScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
        }

        private void _dropScriptMenuItem_Click(object sender, EventArgs e)
        {
            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            string script = "";
            try
            {
                if (data.NodeType == ObjectTreeData.NodeTypes.TABLE)
                {
                    script = _database.ScriptGenerator.GetDropTableScript(data.Catalog, data.Schema, data.Name);
                }
                else if (data.NodeType == ObjectTreeData.NodeTypes.VIEW)
                {
                    script = _database.ScriptGenerator.GetDropViewScript(data.Catalog, data.Schema, data.Name);
                }
                else
                {
                    System.Diagnostics.Trace.Assert(false);
                    throw new NotSupportedException();
                }
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
        }

        private void _contextMenuStrip_Opening(object sender, CancelEventArgs e)
        {
            //Refresh all menu items to enable or disable.
            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            Dictionary<ToolStripItem, bool> enableMap = new Dictionary<ToolStripItem, bool>();

            if (data.NodeType == ObjectTreeData.NodeTypes.TABLE)
            {
                enableMap[_openMenuItem] = true;
                enableMap[_newRecordMenuItem] = true;
                enableMap[_copyNameMenuItem] = true;
                enableMap[_exportMenuItem] = true;
                enableMap[_importMenuItem] = true;
                enableMap[_propertyMenuItem] = true;

                enableMap[_editScriptSubMenu] = true;
                enableMap[_selectMenuItem] = true;
                enableMap[_selectAsteriskMenuItem] = true;
                enableMap[_selectWhereMenuItem] = true;
                enableMap[_insertScriptMenuItem] = true;
                enableMap[_updateScriptMenuItem] = true;
                enableMap[_deleteScriptMenuItem] = true;
                enableMap[_createScriptMenu] = true;
                enableMap[_dropScriptMenuItem] = true;
                enableMap[_filterMenuItem] = true;
                enableMap[_recordCountMenuItem] = true;
                enableMap[_taskMenuItem] = true;
            }
            else if (data.NodeType == ObjectTreeData.NodeTypes.VIEW)
            {
                enableMap[_openMenuItem] = true;
                enableMap[_newRecordMenuItem] = true;
                enableMap[_copyNameMenuItem] = true;
                enableMap[_exportMenuItem] = true;
                enableMap[_importMenuItem] = true;
                enableMap[_propertyMenuItem] = true;

                enableMap[_editScriptSubMenu] = true;
                enableMap[_selectMenuItem] = true;
                enableMap[_selectAsteriskMenuItem] = true;
                enableMap[_selectWhereMenuItem] = true;
                enableMap[_insertScriptMenuItem] = true;
                enableMap[_updateScriptMenuItem] = true;
                enableMap[_deleteScriptMenuItem] = true;
                enableMap[_createScriptMenu] = true;
                enableMap[_dropScriptMenuItem] = true;
                enableMap[_filterMenuItem] = true;
                enableMap[_recordCountMenuItem] = true;
                enableMap[_taskMenuItem] = true;

//                enableMap[_recompileMenuItem] = true;
            }
            else if (data.NodeType == ObjectTreeData.NodeTypes.DATABASE)
            {
                enableMap[_propertyMenuItem] = true;
                enableMap[_taskMenuItem] = true;
                enableMap[_changeDatabaseMenuItem] = true;
            }
            else if (data.NodeType == ObjectTreeData.NodeTypes.PROCEDURE)
            {
                enableMap[_editScriptSubMenu] = true;
                enableMap[_createScriptMenu] = true;
            }

            ToolStripItemCollection items = _contextMenuStrip.Items;
            List<ToolStripItemCollection> itemList = new List<ToolStripItemCollection>();
            itemList.Add(items);
            while (itemList.Count > 0)
            {
                ToolStripItemCollection targetItems = itemList[0];
                itemList.RemoveAt(0);
                foreach (ToolStripItem item in targetItems)
                {
                    item.Enabled = (enableMap.ContainsKey(item) && enableMap[item]);
                    if (item is ToolStripMenuItem)
                    {
                        itemList.Add(((ToolStripMenuItem)item).DropDownItems);
                    }
                }
            }
        }

        private void _filterMenuItem_Click(object sender, EventArgs e)
        {
            List<DatabaseColumn> columns = new List<DatabaseColumn>();

            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            _database.GetTableColumnList(data.Catalog, data.Schema, data.Name, columns);

            FilterForm form = new FilterForm();

            foreach(DatabaseColumn column in columns)
            {
                form.ColumnList.Add(column.ColumnName);
            }
            if (form.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            StringBuilder whereBuffer = new StringBuilder();
            StringBuilder orderBuffer = new StringBuilder();
            foreach (FilterForm.FilterFormItem item in form.FilterList)
            {
                if (item.Value.Length > 0)
                {
                    if (whereBuffer.Length > 0)
                    {
                        whereBuffer.Append("\tAND ");
                    }
                    else
                    {
                        whereBuffer.Append("\t");
                    }
                    whereBuffer.Append(item.Column + " = '" + item.Value + "'\r\n");
                }
                if (item.Sort > 0)
                {
                    if (orderBuffer.Length > 0)
                    {
                        orderBuffer.Append("\t,");
                    }
                    else
                    {
                        orderBuffer.Append("\t");
                    }
                    orderBuffer.Append(item.Column);
                    orderBuffer.Append("\r\n");
                }
            }
            if (whereBuffer.Length > 0)
            {
                script += "WHERE\r\n" + whereBuffer;
            }
            if (orderBuffer.Length > 0)
            {
                script += "ORDER BY\r\n" + orderBuffer;
            }

            ScriptForm form2 = MainForm.CreateEditForm();
            form2.EditControl.TextBox.Text = script;
            try
            {
                form2.DoQueryScript(script);
            }
            catch (ProgramException pe)
            {
                ProgramUtility.ShowError(pe.Message);
            }
        }

        private void _recordCountMenuItem_Click(object sender, EventArgs e)
        {
            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            string fullTableName = data.Schema + "." + data.Name;
            try
            {
                try
                {
                    DbCommand command = Database.CreateCommand("SELECT COUNT(*) FROM " + fullTableName);
                    DbDataReader reader = command.ExecuteReader();
                    Int64 rowCount = 0;
                    while (reader.Read())
                    {
                        rowCount = reader.GetInt64(0);
                    }
                    MessageBox.Show("Table Name:" + data.Name + "\r\n" + "Row Count:" + rowCount, Application.ProductName);
                }
                catch (DbException de)
                {
                    throw new ProgramException("Get count SQL execute failed.", de);
                }
            }
            catch (ProgramException pe)
            {
                ProgramUtility.ShowError(pe.Message);
            }

            
        }


        public void ShowFindForm()
        {
            TreeNode beginNode = _treeView.SelectedNode;
            TreeNode parentNode = _treeView.TopNode;
            if (beginNode != null && beginNode.Parent != null)
            {
                parentNode = beginNode.Parent;
            }

            FindTextForm form = new FindTextForm();

            while (true)
            {
                if (form.ShowDialog(this) != DialogResult.OK)
                {
                    return;
                }

                string targetText = form.TargetText;
                StringComparison senceCase = (form.SenseCase? StringComparison.Ordinal: StringComparison.OrdinalIgnoreCase);

                bool found = false;
                TreeNode foundNode = null;
                foreach (TreeNode node in parentNode.Nodes)
                {
                    if (beginNode != null)
                    {
                        if (beginNode != node)
                        {
                            continue;
                        }
                        beginNode = null;
                    }

                    string nodeName = GetNodeData(node).Name;
                    if (nodeName != null && nodeName.StartsWith(targetText, senceCase))
                    {
                        found = true;
                        foundNode = node;
                        break;
                    }
                }
                if (found)
                {
                    _treeView.SelectedNode = foundNode;
                    break;
                }
                else
                {
                    beginNode = foundNode;
                    if (parentNode.Nodes.Count > 0 && parentNode.Nodes[parentNode.Nodes.Count - 1] == foundNode)
                    {
                        beginNode = null;
                    }
                    MessageBox.Show(this, "Not found.", Application.ProductName);
                }
            }
        }

        private void _changeDatabaseMenuItem_Click(object sender, EventArgs e)
        {
            if (!Database.AllowDatabaseChange)
            {
                MessageBox.Show(this, "Change database not supported.", Application.ProductName);
                return;
            }

            string[] databaseList = Database.GetDatabaseList();
            //string result = "";
            //foreach (string dbName in databaseList)
            //{
            //    result += dbName + "\n";
            //}
            //MessageBox.Show(result);
            ChangeDatabaseForm changeForm = new ChangeDatabaseForm();
            changeForm.DatabaseName = Database.CurrentCatalogName;
            changeForm.OptionList = new List<string>();
            foreach (string dbName in databaseList)
            {
                changeForm.OptionList.Add(dbName);
            }
            if (changeForm.ShowDialog(MainForm) != DialogResult.OK)
            {
                return;
            }
            string changeDatabase = changeForm.DatabaseName;
            try
            {
                Database.ChangeDatabase(changeDatabase);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(MainForm, "Change database failed.", de);
                return;
            }
//            MessageBox.Show("Change database:" + changeDatabase, Application.ProductName);
            RefreshNode();
        }

        private void _exportMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);

            string tableName = data.Name;

            ExportForm form = new ExportForm();
            if (form.ShowDialog(this) != DialogResult.OK)
                return;

            string folder = form.Folder;
            string where = form.Filter;
            string path = Path.Combine(folder, tableName);
            ExportForm.Formats format = (ExportForm.Formats)form.Format;
            switch (format)
            {
                case ExportForm.Formats.CSV:
                    path += ".csv";
                    break;
                case ExportForm.Formats.TSV:
                    path += ".tsv";
                    break;
                case ExportForm.Formats.SQL:
                    path += ".sql";
                    break;
                default:
                    throw new NotSupportedException();
            }
            if (File.Exists(path))
            {
                if (MessageBox.Show(this, "Over write file?\r\n" + Path.GetFileName(path), Application.ProductName, MessageBoxButtons.OKCancel) != DialogResult.OK)
                {
                    return;
                }
            }

            where = where.Trim();
            if (where.Length == 0 || where.Equals("WHERE", StringComparison.OrdinalIgnoreCase))
            {
                where = "";
            }
            string query = "SELECT * FROM " + tableName;
            if (where.Length > 0)
            {
                query += " " + where;
            }
            DatabaseContext database = _database;
            try
            {
                Cursor.Current = Cursors.WaitCursor;

                string exportText = "";
                using (DbCommand command = _database.CreateCommand(query))
                {
                    using (DbDataReader reader = command.ExecuteReader())
                    {
                        command.Dispose();
                        switch (format)
                        {
                            case ExportForm.Formats.CSV:
                                exportText = GetDataReaderToCsv(reader, ",");
                                break;
                            case ExportForm.Formats.TSV:
                                exportText = GetDataReaderToCsv(reader, "\t");
                                break;
                            case ExportForm.Formats.SQL:
                                exportText = GetDataReaderToSql(reader, tableName);
                                break;
                            default:
                                throw new NotSupportedException();
                        }
                    }
                }
                Encoding encoding = Encoding.Default;
                byte[] byteBuffer = encoding.GetBytes(exportText);
                using (FileStream stream = new FileStream(path, FileMode.Create, FileAccess.Write))
                {
                    stream.Write(byteBuffer, 0, byteBuffer.Length);
                }
            }
            catch (DbException dbe)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", dbe);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", de);
            }
            catch (IOException ioe)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", ioe);
            }
            finally
            {
                Cursor.Current = Cursors.Default;
            }
        }

        private void _importMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);

            string tableName = data.Name;

            ImportForm form = new ImportForm();
            form.FilePath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Desktop), tableName);
            if (form.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }
            string sourceFile = form.FilePath;

            StringBuilder builder = new StringBuilder();

            List<DatabaseColumn> columns = new List<DatabaseColumn>();
            _database.GetTableColumnList(data.Catalog, data.Schema, tableName, columns);

            StringBuilder valueBuilder = new StringBuilder("");
            int valueCount = 1;
            foreach (DatabaseColumn column in columns)
            {
                if (builder.Length > 0)
                {
                    builder.Append(",\r\n");
                    valueBuilder.Append(",\r\n");
                }
                builder.Append("\t");
                valueBuilder.Append("\t");
                string columnName = column.ColumnName;
                columnName = columnName.ToLower();
                builder.Append(columnName);
//                valueBuilder.Append("%" + valueCount.ToString());
                valueBuilder.Append("?");
                valueCount++;
            }

            tableName = tableName.ToLower();
            string sColumns = builder.ToString();
            string result = "INSERT INTO " + tableName + "\r\n(\r\n" + sColumns + "\r\n";
            result += ")\r\nVALUES\r\n(";
            result += "\r\n";
            result += valueBuilder.ToString();
            result += "\r\n)\r\n";
            string insertSql = result;
#if DEBUG
            System.Diagnostics.Debug.WriteLine("insert sql:" + insertSql);
#endif
            try
            {
                using (StreamReader reader = new StreamReader(sourceFile, Encoding.Default))
                {
                    ImportCsv(reader, ",", columns, insertSql);
                }
            }
            catch (ProgramException pe)
            {
                ProgramUtility.ShowError(this.MainForm, "Data import failed.", pe);
            }
        }

        protected void ImportCsv(TextReader reader, string delimiter, List<DatabaseColumn> columns, string insertSql)
        {

            TextFieldParser parser = new TextFieldParser(reader);
            parser.SetDelimiters(delimiter);
            int maxColumn = columns.Count;
            bool headLine = true;
            while (!parser.EndOfData)
            {
                string[] textFields = null;
                try
                {
                    textFields = parser.ReadFields();
                }
                catch (MalformedLineException mle)
                {
                    throw new ProgramException("Text parse failed.", mle);
                }
                if (headLine)
                {
                    headLine = false;
                    continue;
                }
                if (textFields.Length == 0)
                {
                    continue;
                }
                object[] fields = new object[maxColumn];
                Array.Copy(textFields, 0, fields, 0, Math.Min(maxColumn, textFields.Length));
                using (DbCommand command = _database.CreateCommand(insertSql))
                {
                    for (int index = 0; index < maxColumn; index++)
                    {
                        string columnName = columns[index].ColumnName;
                        if (columns[index].IsDate && fields[index] != null)
                        {
                            DateTime tempTime;
                            DateTime.TryParse(fields[index].ToString(), out tempTime);
                            fields[index] = tempTime;
                        }
                        else if (columns[index].IsNumber && fields[index] != null)
                        {
                            Decimal tempDecimal;
                            Decimal.TryParse(fields[index].ToString(), out tempDecimal);
                            fields[index] = tempDecimal;
                        }
                        command.Parameters.Add(_database.CreateParameter(null, fields[index]));
                    }
                    try
                    {
                        command.ExecuteNonQuery();
                    }
                    catch (DbException de)
                    {
                        throw new ProgramException("Invalid data.", de);
                    }
                    
                    //catch (ArgumentException ae)
                    //{
                    //    throw new ProgramException("Invalid data.", ae);
                    //}
                }

            }
        }
    }
}
