﻿using System;
using System.Threading.Tasks;
using Microsoft.Extensions.Configuration;
using NLog;
using NLog.Fluent;
using Sirkadirov.Overtest.SharedLibraries.Shared;
using Sirkadirov.Overtest.TestingAgent.TestingServices;

namespace Sirkadirov.Overtest.TestingAgent
{
    internal class Program
    {
        private const string ConfigurationFileName = "otestingagent.config.json";
        private const string DefaultLoggerName = "OVERTEST_FATAL";

        private readonly string[] _consoleArguments;
        private readonly ILogger _logger;
        
        public static async Task Main(string[] args) => await new Program(args).ExecuteAsync();

        public Program(string[] consoleArguments)
        {
            _consoleArguments = consoleArguments;
            _logger = LogManager.GetCurrentClassLogger();
        }
        
        public async Task ExecuteAsync()
        {
            // Видобуваємо конфігурацію додатку з файлу
            IConfiguration configuration = new ConfigurationBuilder().AddJsonFile(ConfigurationFileName).Build();
            
            // Налаштовуємо логування операцій
            ConfigureLogging(configuration);

            _logger.Info().Message("Overtest Testing Agent is starting...")
                .Property("Start time (UTC)", DateTime.UtcNow)
                .Property("Operating system", Environment.OSVersion)
                .Property("Is AMD64 process mode", Environment.Is64BitProcess)
                .Property("CLR version", Environment.Version)
                .Write();
            
            // Створюємо і запускаємо Investigator agent
            await new InvestigatorAgent(configuration).ExecuteAsync();
        }
        
        private static void ConfigureLogging(IConfiguration configuration)
        {
            // Передаємо конфігурацію 
            LogManager.Configuration = configuration.ConfigureLogging();
            
            // Встановлюємо обробник критичних виключень
            AppDomain.CurrentDomain.UnhandledException += (sender, eventArgs) =>
            {
                LogManager.GetLogger(DefaultLoggerName)
                    .Fatal()
                    .Exception((Exception) eventArgs.ExceptionObject)
                    .Property(nameof(eventArgs.IsTerminating), eventArgs.IsTerminating)
                    .Write();
                
                Environment.Exit(1);
            };
        }
    }
}