﻿using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using NLog;
using Sirkadirov.Overtest.SharedLibraries.Database;
using Sirkadirov.Overtest.SharedLibraries.Shared;

namespace Sirkadirov.Overtest.TestingAgent.Services.Storage
{
    public class TestingDataCacheProvider
    {
        private readonly IConfiguration _configuration;
        private readonly OvertestDatabaseContext _databaseContext;
        private readonly ILogger _logger;
        private readonly SemaphoreSlim _cacheCreationRequestSemaphore;

        public TestingDataCacheProvider(IConfiguration configuration, OvertestDatabaseContext databaseContext)
        {
            _configuration = configuration;
            _databaseContext = databaseContext;
            _logger = LogManager.GetCurrentClassLogger();
            _cacheCreationRequestSemaphore = new SemaphoreSlim(1, 1);
        }
        
        public async Task<TempDirectoryAccessPoint> GetTestingDataForCurrentSubmissionAsync(Guid programmingTaskId)
        {
            _logger.Info($"Testing data copy formation for programming task {programmingTaskId} started...");
            
            var (_, testingDataFullPath) = await CreateTestingDataCacheAsync(programmingTaskId);
            var storageConfiguration = _configuration.GetSection("general:storage");
            
            _logger.Info($"Starting testing data copying process for programming task {programmingTaskId}...");
            
            var tempDirectoryPath = Path.Combine(
                storageConfiguration.GetValue<string>("path"),
                storageConfiguration.GetValue<string>("subdirectories:temp_directory"),
                Guid.NewGuid().ToString()
            );
            
            /*
             * Створюємо копію директорії з тестовими даними.
             * Це необхідно, щоб забезпечити чистоту тестування,
             * у випадку використання автором завдання скриптових
             * сценаріїв доповнення тестових даних.
             */
            FileSystemSharedMethods.SecureCopyDirectory(
                testingDataFullPath,
                tempDirectoryPath
            );
            
            _logger.Info($"Testing data copying process for programming task {programmingTaskId} finished!");
            
            // Передаємо інформацію про директорію у обгортці Disposable об'єкту
            return new TempDirectoryAccessPoint(tempDirectoryPath, false);
        }
        
        public async Task<int /* updated count */> UnsafeUpdateTestingDataCacheAsync()
        {
            _logger.Info("Starting *unsafe* update of local testing data cache...");
            
            var storageConfiguration = _configuration.GetSection("general:storage");
            var testingDataCacheDirectoryFullPath = Path.Combine(
                storageConfiguration.GetValue<string>("path"),
                storageConfiguration.GetValue<string>("subdirectories:testing_data_directory")
            );
            
            _logger.Info("Gathering current testing data hashes from the database...");
            
            // Видобуваємо список актуальних хешей даних для тестування усіх завдань станом на поточний момент
            var currentTestingDataHashes = await _databaseContext.ProgrammingTasks
                .Include(i => i.TestingData)
                .Where(t => t.TestingData.DataPackageFile != null)
                .Select(s => s.TestingData.DataPackageHash)
                .AsNoTracking().ToListAsync();
            
            _logger.Info($"Current testing data cache hashes: {string.Join(", ", currentTestingDataHashes)}");
            
            _logger.Info("Gathering the information about local testing data cache hashes...");
            
            // Формуємо список хешей кешованих тестових даних на локальному девайсі
            var localTestingDataHashes = Directory.GetDirectories(testingDataCacheDirectoryFullPath);
            
            _logger.Info($"Local testing data cache hashes: {string.Join(", ", localTestingDataHashes)}");
            
            if (currentTestingDataHashes.Any())
            {
                _logger.Info("Clearning outdated and downloading updated testing data items...");
                
                // Створюємо список хешів кешованих даних, які потрібно видалити
                var localCacheItemsToPurge = new List<string>();
                
                /*
                 * Переглядаємо список хешей кешованих тестових даних
                 * в пошуках співпадінь з актуальних списком хешей.
                 *
                 * Якщо елементи співпадають, видаляємо відповідний елемент
                 * зі списку [currentTestingDataHashes], у іншому випадку
                 * додаємо хеш у список хешів кешованих даних, які потрібно
                 * видалити з локального сховища даних для тестування.
                 *
                 * На виході з циклу отримуємо 2 нові списки:
                 * 
                 *  - [currentTestingDataHashes], який містить хеші тих
                 *    завдань, які треба зберегти у локальному сховищі;
                 *  
                 *  - [localCacheItemsToPurge], який містить хеші тих
                 *    даних для тестування, кеш яких потрібно видалити
                 *    з локального сховища тестових даних.
                 */
                foreach (var localCacheItem in localTestingDataHashes)
                {
                    if (currentTestingDataHashes.Contains(localCacheItem))
                        currentTestingDataHashes.Remove(localCacheItem);
                    else
                        localCacheItemsToPurge.Add(localCacheItem);
                }
                
                /*
                 * Видаляємо застарілі елементи кешу даних
                 * для тестування з локального сховища.
                 */
                foreach (var cacheItemHash in localCacheItemsToPurge)
                {
                    FileSystemSharedMethods.SecureDeleteDirectory(GetTestingDataExpectedFullPath(cacheItemHash));
                }
                
                // Видобуваємо ідентифікатори завдань з програмування, тестові дані яких ми зараз маємо закешувати
                var programmingTasksToFormCache = await _databaseContext.ProgrammingTasks
                    .Include(i => i.TestingData)
                    .Where(t => t.TestingData.DataPackageFile != null)
                    .Where(t => currentTestingDataHashes.Contains(t.TestingData.DataPackageHash))
                    .Select(s => s.Id)
                    .ToArrayAsync();
                
                /*
                 * Виконуємо операцію хешування даних для
                 * кожного завдання, ідентифікатор якого
                 * занесено до раніше видобутого масиву.
                 */
                foreach (var programmingTaskId in programmingTasksToFormCache)
                {
                    await CreateTestingDataCacheAsync(programmingTaskId);
                }
                
                _logger.Info("Local testing data cache updated successfully!");
            }
            else
            {
                _logger.Info("No testing data found in the database! Clearing local cache...");
                
                if (Directory.GetDirectories(testingDataCacheDirectoryFullPath).Length > 0)
                    FileSystemSharedMethods.SecureRecreateDirectory(testingDataCacheDirectoryFullPath);
            }
            
            /*
             * Повертаємо кількість оновлень даних для тестування.
             * Вважаємо, що 
             */
            return currentTestingDataHashes.Count;
        }
        
        private async Task<(bool /* is updated */, string /* cache full path */)> CreateTestingDataCacheAsync(Guid programmingTaskId)
        {
            _logger.Info($"Creating local cache for programming task {programmingTaskId}...");
            
            /*
             * Тільки один потік може входити у наступну область.
             * Це пов'язано з ризиком виникнення колізії при
             * створенні нового екземпляра кешу тестових даних.
             */
            _logger.Info($"Entering semaphore in {nameof(CreateTestingDataCacheAsync)} method...");
            await _cacheCreationRequestSemaphore.WaitAsync();
            
            try
            {
                _logger.Info($"Gathering current testing data hash for programming task {programmingTaskId}...");
                
                var currentTestingDataHash = await GetActualTestingDataHashAsync(programmingTaskId);
                var testingDataExpectedPath = GetTestingDataExpectedFullPath(currentTestingDataHash);
                
                _logger.Info($"Current testing data hash for programming task {programmingTaskId} is {currentTestingDataHash}.");
                
                /*
                 * Перевіряємо у семафорі, щоб не сталося
                 * ситуації, коли ми дані вже оновили, а
                 * інший потік про це і не здогадується.
                 */
                if (ActualCacheExists(currentTestingDataHash))
                {
                    _logger.Info($"Local cache for programming task {programmingTaskId} is up to date!");
                    return (false, testingDataExpectedPath);
                }
                
                _logger.Info($"Local cache for programming task {programmingTaskId} is outdated! Now trying to update...");
                
                _logger.Info($"Downloading testing data package from the database...");

                // Видобуваємо дані для тестування з бази даних системи
                var testingDataPackage = await _databaseContext.ProgrammingTasks
                    .Include(i => i.TestingData)
                    .Where(t => t.Id == programmingTaskId)
                    .Select(s => s.TestingData.DataPackageFile)
                    .AsNoTracking().FirstAsync();
                
                _logger.Info($"Initializing local cache for programming task {programmingTaskId}...");
                
                // Створюємо директорію перед розпаковкою даних для тестування у неї (в безпечний спосіб)
                FileSystemSharedMethods.SecureRecreateDirectory(testingDataExpectedPath);

                // ReSharper disable once ConvertToUsingDeclaration
                await using (var fileStream = new MemoryStream(testingDataPackage))
                {
                    // Передаємо потік новому екземпляру ZipArchive, який розглядає його як ZIP архів
                    using var zipArchive = new ZipArchive(fileStream, ZipArchiveMode.Read, false, Encoding.UTF8);

                    // Розпаковуємо архів у вказану директорію з можливим перезаписом збіжностей
                    zipArchive.ExtractToDirectory(testingDataExpectedPath, true);
                }
                
                _logger.Info($"Local testing data cache for programming task {programmingTaskId} updated successfully!");
                
                // Передаємо у викликаючу функцію повний шлях до директорії з даними для тестування
                return (true, testingDataExpectedPath);
            }
            catch (Exception ex)
            {
                _logger.Fatal($"A fatal exception thrown while creating or updating cache for programming task {programmingTaskId}!");
                _logger.Fatal(ex);
                throw;
            }
            finally
            {
                /*
                 * Виходимо з області семафора, щоб дозволити іншим
                 * потокам увійти в нього і перевірити сховище кешу
                 * на наявність необхідних даних, і, за потреби,
                 * оновити ці дані.
                 */
                _logger.Info($"Leaving semaphore in {nameof(CreateTestingDataCacheAsync)} method...");
                _cacheCreationRequestSemaphore.Release();
            }
        }
        
        public async Task<bool> TestingDataExistsAsync(Guid programmingTaskId)
        {
            return await _databaseContext.ProgrammingTasks
                .Include(i => i.TestingData)
                .Where(t => t.Id == programmingTaskId)
                .Where(t => t.TestingData.DataPackageFile != null)
                .AnyAsync();
        }
        
        public async Task<bool> ActualCacheExistsAsync(Guid programmingTaskId)
            => ActualCacheExists(await GetActualTestingDataHashAsync(programmingTaskId));
        private bool ActualCacheExists(string currentTestingDataHash)
            => Directory.Exists(GetTestingDataExpectedFullPath(currentTestingDataHash));

        private async Task<string> GetActualTestingDataHashAsync(Guid programmingTaskId)
        {
            return await _databaseContext.ProgrammingTasks
                .Include(i => i.TestingData)
                .Where(t => t.Id == programmingTaskId)
                .Select(s => s.TestingData.DataPackageHash)
                .AsNoTracking().FirstAsync();
        }
        
        private string GetTestingDataExpectedFullPath(string testingDataHash)
        {
            var storageConfiguration = _configuration.GetSection("general:storage");
            return Path.Combine(
                storageConfiguration.GetValue<string>("path"),
                storageConfiguration.GetValue<string>("subdirectories:testing_data_directory"),
                testingDataHash
            );
        }
    }
}