﻿using System;
using System.IO;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Newtonsoft.Json;
using NLog;
using Sirkadirov.Overtest.SharedLibraries.Database;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications;
using Sirkadirov.Overtest.SharedLibraries.Shared.TestingApplications;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramCompilationAgent;
using Sirkadirov.Overtest.TestingAgent.Services.Storage;
using Sirkadirov.Overtest.TestingAgent.TestingServices.Skeleton;
using Sirkadirov.Overtest.TestingAgent.TestingServices.VerificationStages;

namespace Sirkadirov.Overtest.TestingAgent.TestingServices
{
    public class ApplicationTestingWorker : TestingServiceWorkerBase, IDisposable
    {
        private const string TestingDataConfigurationFileName = "testingdata.config.json";
        
        private readonly ILogger _logger;
        private readonly CompilationAgent _compilationAgent;
        
        private readonly TempDirectoryAccessPoint _testingDataAccessor;
        private TestingDataConfiguration _testingDataConfiguration;

        private TempDataDirectoryOperator _userProgramTempDirectory;
        
        // Зберігаємо час початку і закінчення тестування (зі статистичнизх міркувань)
        private DateTime _testingStartedDateTime;
        private DateTime _testingEndedDateTime;
        
        public ApplicationTestingWorker(
            IConfiguration configuration,
            OvertestDatabaseContext databaseContext,
            TestingApplication testingApplication,
            TempDirectoryAccessPoint testingDataAccessor,
            CompilationAgent compilationAgent
        ) : base(configuration, databaseContext, testingApplication)
        {
            _logger = LogManager.GetCurrentClassLogger();
            _testingDataAccessor = testingDataAccessor;
            _compilationAgent = compilationAgent;
            TestingApplication.TestingResult = new TestingApplicationResult();
        }

        public async Task ExecuteAsync()
        {
            // Set testing START time
            _testingStartedDateTime = DateTime.UtcNow;
            
            // Prepare application data and files
            await PrepareApplicationAsync();
            
            var userSolutionCompilationResult = ExecuteCompilationStage();
            
            if (TestingApplication.TestingType != TestingApplication.ApplicationTestingType.SyntaxMode && userSolutionCompilationResult.isSuccessful)
            {
                await new UserSolutionTestingStage(
                    Configuration, DatabaseContext,
                    TestingApplication,
                    _testingDataAccessor, _testingDataConfiguration,
                    _userProgramTempDirectory, userSolutionCompilationResult.userSolutionCompilationPlugin
                ).ExecuteAsync();
            }
            else
            {
                TestingApplication.TestingResult.RawTestingResults = TestingApplication.TestingType.ToString();
                TestingApplication.TestingResult.SolutionAdjudgement = TestingApplicationResult.SolutionAdjudgementType.ZeroSolution;
                TestingApplication.TestingResult.GivenDifficulty = 0;
            }
            
            // Set testing END time
            _testingEndedDateTime = DateTime.UtcNow;
            
            // Upload results to the database
            await FormatAndUploadResultsAsync();
        }

        private (bool isSuccessful, OvertestProgramCompilationPluginBase userSolutionCompilationPlugin) ExecuteCompilationStage()
        {
            // ReSharper disable once UseDeconstruction
            var userSolutionCompilationResult = new UserProgramCompilationStage(
                Configuration, TestingApplication, _compilationAgent,
                _userProgramTempDirectory.TempDirectoryFullName
            ).Execute();

            TestingApplication.TestingResult.CompilationResult = new TestingApplicationResult.CompilationStageResult
            {
                IsSuccessful = userSolutionCompilationResult.isSuccessful,
                CompilationOutput = userSolutionCompilationResult.compilerOutput
            };
            
            return (userSolutionCompilationResult.isSuccessful, userSolutionCompilationResult.compilationPlugin);
        }
        
        private async Task PrepareApplicationAsync()
        {
            PrepareTestingData();
            CompileAuthorSolution();
            
            var storangeConfiguration = Configuration.GetSection("general:storage");
            
            _userProgramTempDirectory = new TempDataDirectoryOperator(
                Path.Combine(
                    storangeConfiguration.GetValue<string>("path"),
                    storangeConfiguration.GetSection("subdirectories").GetValue<string>("testing_applications_directory")
                )
            );
            
            void PrepareTestingData()
            {
                // Читаємо і десеріалізуємо конфігураційний файл даних для тестування
                try
                {
                    using var streamReader = File.OpenText(Path.Combine(_testingDataAccessor.DirectoryFullName, TestingDataConfigurationFileName));
                    _testingDataConfiguration = (TestingDataConfiguration) new JsonSerializer().Deserialize(streamReader, typeof(TestingDataConfiguration));
                }
                catch (JsonException ex)
                {
                    throw new ApplicationException($"Testing data configuration file is invalid for programming task id {TestingApplication.ProgrammingTaskId}!", ex);
                }
                catch (Exception ex)
                {
                    throw new ApplicationException($"Testing data configuration file not found or not accessible for testing application {TestingApplication.Id}!", ex);
                }
            }
            
            void CompileAuthorSolution()
            {
                if (TestingApplication.TestingType == TestingApplication.ApplicationTestingType.SyntaxMode)
                    return;

                if (_testingDataConfiguration.Verification.TestLib.GeneratorPath == null)
                    // ReSharper disable once RedundantJumpStatement
                    return;

                // TODO: CompileAuthorSolution
            }
        }
        
        private async Task FormatAndUploadResultsAsync()
        {
            try
            {
                // Оновлюємо статус тестування на "Verified"
                // TODO: Реалізувати підтримку виставлення оцінок у ручному режимі
                await UpdateTestingStatusAsync(TestingApplication.ApplicationStatus.Verified);
                
                // Створюємо запис з інформацією про результати тестування
                var testingApplicationResult = new TestingApplicationResult
                {
                    ProcessingTime = _testingEndedDateTime - _testingStartedDateTime,
                    
                    CompilationResult = TestingApplication.TestingResult.CompilationResult,
                    RawTestingResults = TestingApplication.TestingResult.RawTestingResults,
                    
                    GivenDifficulty = TestingApplication.TestingResult.GivenDifficulty,
                    SolutionAdjudgement = TestingApplication.TestingResult.SolutionAdjudgement,

                    TestingApplicationId = TestingApplication.Id
                };
                
                // Додаємо виществорений запис до необхідної нам таблиці
                await DatabaseContext.TestingApplicationsResults.AddAsync(testingApplicationResult);
                // Зберігаємо поточні зміни до бази даних системи
                await DatabaseContext.SaveChangesAsync();
            }
            catch (Exception ex)
            {
                _logger.Error($"{nameof(FormatAndUploadResultsAsync)} failed due to an unhandled exception. Testing application id: {TestingApplication.Id}. Exception details: {ex}");
                throw; // Обов'язково повторно викидаємо те ж саме виключення, щоб перервати процес тестування
            }
        }

        private async Task UpdateTestingStatusAsync(TestingApplication.ApplicationStatus newTestingStatus)
        {
            /*
             * Через деякі наявні особливості будови Entity Framework
             * Core, щоб не вивантажувати весь об'єкт з бази даних
             * для оновлення лиш одної його властивості, ми створюємо
             * фальш-об'єкт і використовуємо можливості ручної прив'язки
             * і відслідковування змін до даних.
             */
            
            var attachedTestingApplicationObject = new TestingApplication
            {
                Id = TestingApplication.Id,
                Status = newTestingStatus
            };
            
            // Вмикаємо відслідковування новостворенного об'єкта
            DatabaseContext.TestingApplications.Attach(attachedTestingApplicationObject);
            
            /*
             * Вказуємо попереднє значення поля, яке ми хочемо змінити.
             * Якщо цього не зробити, отримаємо звичайнісіньке виключення
             * типу DBConcurrencyException (ще одна особливість EF Core).
             */
            DatabaseContext.Entry(attachedTestingApplicationObject)
                .Property(p => p.Status)
                .OriginalValue = TestingApplication.Status;
            
            // Вказуємо, що значення необхідної нам властивості було змінено
            DatabaseContext.Entry(attachedTestingApplicationObject)
                .Property(p => p.Status).
                IsModified = true;
            
            // Зберігаємо зміни до бази даних
            await DatabaseContext.SaveChangesAsync();
            
            /*
             * Тепер можемо вимкнути попередньо ввімкнене
             * відслідковування. Це треба робити вручну,
             * бо його було увімкнено вручну.
             */
            DatabaseContext.Entry(attachedTestingApplicationObject).State = EntityState.Detached;
        }

        public void Dispose()
        {
            _userProgramTempDirectory?.Dispose();
        }
    }
}