﻿using System;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.Logging;
using NLog;
using NLog.Extensions.Logging;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TasksArchive;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications.Extras;
using Sirkadirov.Overtest.SharedLibraries.Shared;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramCompilationAgent;
using Sirkadirov.Overtest.TestingAgent.Services.Storage;
using ILogger = NLog.ILogger;

namespace Sirkadirov.Overtest.TestingAgent.TestingServices
{
    
    public class InvestigatorAgent
    {
        private readonly IConfiguration _configuration;
        private readonly ILogger _logger;
        private readonly ILoggerProvider _loggerProvider;
        
        private readonly TestingDataCacheProvider _testingDataCacheProvider;
        private readonly CompilationAgent _compilationAgent;

        public InvestigatorAgent(IConfiguration configuration)
        {
            _configuration = configuration;
            _logger = LogManager.GetCurrentClassLogger();
            _loggerProvider = new NLogLoggerProvider();
            
            _testingDataCacheProvider = new TestingDataCacheProvider(_configuration, configuration.GetDbContext(_loggerProvider));
            _compilationAgent = new CompilationAgent(string.Empty); // TODO: Implement compilation plugins lookup by path
        }
        
        public async Task ExecuteAsync()
        {
            _logger.Info("Starting investigating agent execution process...");
            await Initialize();
            CreateSightParticlesAndWait();
        }

        private async Task Initialize()
        {
            _logger.Info("Investigating agent initialization started!");
            
            // Ініціалізуємо усі можливі сховища зберігання
            InitStorage();
            
            // Оновлюємо кеш даних для тестування розв'язків до завдань
            await _testingDataCacheProvider.UnsafeUpdateTestingDataCacheAsync();
            
            // Здійснюємо пошук плагінів, що відповідають за компіляцію програм
            _compilationAgent.LookupCompilerPlugins();
            
            _logger.Info("Investigating agent initialization finished!");

            void InitStorage()
            {
                var storageConfiguration = _configuration.GetSection("general:storage");

                Directory.CreateDirectory(storageConfiguration.GetValue<string>("path"));
                
                foreach (var subdirectory in storageConfiguration.GetSection("subdirectories").GetChildren())
                {
                    Directory.CreateDirectory(Path.Combine(
                        storageConfiguration.GetValue<string>("path"),
                        subdirectory.Value
                    ));
                }
            }
        }
        
        private void CreateSightParticlesAndWait()
        {
            var tasks = new List<Task>();
            
            // Запускаємо на виконання стільки потоків тестування, скільки вказано у файлі конфігурації
            for (var i = 0; i < _configuration.GetValue<int>("general:parallelism"); i++)
            {
                var task = new Task(() => ExecuteSightParticle().Wait());
                
                task.Start();
                tasks.Add(task);
            }
            
            // Чекаємо на завершення усіх пов'язаних завдань (довічно)
            Task.WaitAll(tasks.ToArray());
        }
        
        private async Task ExecuteSightParticle()
        {
            while (true)
            {
                try
                {
                    _logger.Info("Searching for waiting testing applications...");
                    
                    /*
                     * Створюємо екземпляр контексту бази даних, одразу ж
                     * розпочинаючи нову транзакцію. Це дозволить нам
                     * обрати і отримати інформацію про єдиний очікуючий
                     * запит на тестування, не утворюючи колізії з іншими
                     * потоками і агентами тестування.
                     */
                    await using var databaseContext = _configuration.GetDbContext(_loggerProvider);
                    await using var transaction = await databaseContext.Database.BeginTransactionAsync();
                    
                    // Отримуємо список ідентифікаторів компіляторів, які підтримуються цих агентом тестування
                    var supportedCompilers = _configuration
                        .GetSection("compilers")
                        .GetChildren()
                        .Select(c => c.GetValue<string>("id"))
                        .ToList();
                    
                    // Переходимо до пошуку очікуючого запиту на тестування
                    var testingApplication = await databaseContext.TestingApplications
                        .Include(i => i.SourceCode)
                        
                        .Include(i => i.ProgrammingTask)
                        .ThenInclude(i => i.TestingData)
                        
                        .Where(a => a.Status == TestingApplication.ApplicationStatus.Waiting)
                        .Where(a => supportedCompilers.Contains(a.SourceCode.ProgrammingLanguageId.ToString()))
                        .Where(a => a.ProgrammingTask.TestingData.DataPackageFile != null)
                        
                        .OrderByDescending(o => o.CompetitionId != null)
                        .ThenBy(o => o.Created)
                        
                        // Видобуваємо лише необхідну нам інформацію
                        .Select(s => new TestingApplication
                        {
                            Id = s.Id,
                            Created = s.Created,
                            
                            SourceCode = new TestingApplicationSourceCode
                            {
                                ProgrammingLanguageId = s.SourceCode.ProgrammingLanguageId,
                                SourceCode = s.SourceCode.SourceCode
                            },
                            
                            TestingType = s.TestingType,
                            Status = s.Status,
                            
                            AuthorId = s.AuthorId,
                            CompetitionId = s.CompetitionId,
                            
                            ProgrammingTaskId = s.ProgrammingTaskId,
                            ProgrammingTask = new ProgrammingTask
                            {
                                Difficulty = s.ProgrammingTask.Difficulty
                            }
                        }).AsNoTracking()
                        .FirstOrDefaultAsync();
                    
                    /*
                     * Запит на тестування знайдено, тож наступним кроком
                     * буде оновлення запису у базі даних системи, яке
                     * свідчитиме про те, що ми вже розпочати опрацювання
                     * цього запиту на тестування.
                     */
                    if (testingApplication != null)
                    {
                        _logger.Info($"Waiting testing application attached. Id: {testingApplication.Id}.");
                        /*
                         * Щоб не вивантажувати повний запис з бази даних,
                         * використовуємо "хитрий" метод для його оновлення
                         * без вивантаження будь-якої додаткової інформації
                         */
                        var attachedTestingApplicationObject = new TestingApplication
                        {
                            Id = testingApplication.Id,
                            Status = testingApplication.Status = TestingApplication.ApplicationStatus.Selected
                        };

                        databaseContext.TestingApplications.Attach(attachedTestingApplicationObject);
                        
                        databaseContext.Entry(attachedTestingApplicationObject)
                            .Property(p => p.Status)
                            .OriginalValue = TestingApplication.ApplicationStatus.Waiting;
                        
                        databaseContext.Entry(attachedTestingApplicationObject)
                            .Property(p => p.Status)
                            .IsModified = true;
                        
                        /*
                         * Використовуємо try для відсортовування нецікавих нам виключень,
                         * здебільшого пов'язаних з можливим одночасним вивантаженням одного
                         * і того ж запиту на тестування різними агентами чи потоками
                         * тестування з бази даних системи (що є цілком можливим).
                         */
                        try
                        {
                            await databaseContext.SaveChangesAsync();
                            await transaction.CommitAsync();
                        }
                        catch (InvalidOperationException ex)
                        {
                            // Обробляємо усі помилки як DBConcurrencyException
                            throw new DBConcurrencyException(string.Empty, ex);
                        }
                        
                        /*
                         * Не забуваємо позбутися слідкування за нашим об'єктом (інакше
                         * ми не зможемо використовувати цей контекст бази даних для того,
                         * щоб оновлювати інформацію про поточний запит на тестування).
                         */
                        databaseContext.Entry(attachedTestingApplicationObject).State = EntityState.Detached;
                        
                        _logger.Info($"Copying testing data for testing application {testingApplication.Id}...");
                        
                        var programmingTaskTestingDataAccessor =
                            await _testingDataCacheProvider.GetTestingDataForCurrentSubmissionAsync(
                                testingApplication.ProgrammingTaskId
                            );
                        
                        _logger.Info($"Testing data for testing application {testingApplication.Id} copied successfully.");
                        _logger.Info($"Initializing testing worker for application {testingApplication.Id}...");
                        
                        // Розпочинаємо тестування розв'язку
                        using var applicationTestingWorker = new ApplicationTestingWorker(
                            _configuration,
                            databaseContext,
                            testingApplication,
                            programmingTaskTestingDataAccessor,
                            _compilationAgent
                        );
                        
                        _logger.Info($"Executing testing worker for testing application {testingApplication.Id}");
                        
                        await applicationTestingWorker.ExecuteAsync();
                        
                        programmingTaskTestingDataAccessor.Dispose();
                        
                        // Force to collect some garbage
                        GC.Collect();
                        
                        _logger.Info($"Testing application {testingApplication.Id} verification process finished!");
                    }
                    // У іншому випадку відміняємо транзакцію і виконуємо таймаут до наступної перевірки
                    else
                    {
                        await transaction.RollbackAsync();
                        _logger.Info("No waiting submissions found. Executing delay...");
                        await Task.Delay(_configuration.GetValue<int>("general:empty_delay"));
                    }
                }
                catch (DbUpdateConcurrencyException) { /* Ігноруємо подібні виключення задля свого ж добра */ }
                catch (Exception ex)
                {
                    _logger.Error($"Got an exception during TestingApplication processing in {nameof(ExecuteSightParticle)} method! Exception details: {ex}");
                }
            }
            
            // За задумкою, ця функція ніколи не закінчує своєї роботи
            // тому ми можемо ігнорувати наступну інспекцію ReSharper:
            // ReSharper disable once FunctionNeverReturns
        }
    }
}