﻿using System.IO;
using System.Threading.Tasks;
using Microsoft.Extensions.Configuration;
using Sirkadirov.Overtest.SharedLibraries.Database;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications;
using Sirkadirov.Overtest.SharedLibraries.Shared.TestingApplications;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramCompilationAgent;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramExecutor.DataStructures;
using Sirkadirov.Overtest.TestingAgent.Libraries.TestLib;
using Sirkadirov.Overtest.TestingAgent.Libraries.TestLib.Default;
using Sirkadirov.Overtest.TestingAgent.Services.Storage;

namespace Sirkadirov.Overtest.TestingAgent.TestingServices.Skeleton
{
    public abstract class VerificationStageBase : TestingServiceWorkerBase
    {
        protected readonly TempDirectoryAccessPoint TestingDataAccessor;
        protected readonly TestingDataConfiguration TestingDataConfiguration;
        
        protected readonly TempDataDirectoryOperator UserProgramTempDirectory;
        protected readonly OvertestProgramCompilationPluginBase UserProgramCompilationPlugin;
        
        protected VerificationStageBase(
            IConfiguration configuration, OvertestDatabaseContext databaseContext,
            TestingApplication testingApplication,
            TempDirectoryAccessPoint testingDataAccessor, TestingDataConfiguration testingDataConfiguration,
            TempDataDirectoryOperator userProgramTempDirectory, OvertestProgramCompilationPluginBase userProgramCompilationPlugin
        ) : base(configuration, databaseContext, testingApplication)
        {
            TestingDataAccessor = testingDataAccessor;
            TestingDataConfiguration = testingDataConfiguration;
            UserProgramTempDirectory = userProgramTempDirectory;
            UserProgramCompilationPlugin = userProgramCompilationPlugin;
        }

        public abstract Task ExecuteAsync();

        protected ProgramExecutionRequest GetProgramExecutorRequest(TestingDataConfiguration.VerificationStruct.TestInfoStruct testInfo, string programTempPath)
        {
            var (userProgramExecutionFileName, userProgramExecutionArguments) = UserProgramCompilationPlugin.GetProgramExecutionArguments(
                Path.Combine(
                    programTempPath,
                    UserProgramCompilationPlugin.GetCompilationOutputFileName()
                ), string.Empty // TODO: Allow arguments customization
            );
            
            return new ProgramExecutionRequest
            {
                StartInformation = new ProgramExecutionRequest.ProgramStartInformation
                {
                    Path = userProgramExecutionFileName, Arguments = userProgramExecutionArguments,
                    WorkingDirectory = programTempPath
                },
                RunAsFeature = new ProgramExecutionRequest.ProgramRunAsFeature()
                {
                    Enabled = Configuration.GetValue<bool>("security:run_as:enabled"),
                    UserName = Configuration.GetValue<string>("security:run_as:username"),
                    UserPasswordClearText = Configuration.GetValue<string>("security:run_as:password"),
                    UserDomain = Configuration.GetValue<string>("security:run_as:domain")
                },
                RuntimeLimitsFeature = new ProgramExecutionRequest.ProgramRuntimeLimitsFeature
                {
                    ProcessorTimeLimit = new ProgramExecutionRequest.ProgramRuntimeLimitsFeature.RuntimeLimit<long> { Enabled = true, Limit = testInfo.RuntimeLimits.MaxProcessorTime },
                    RealTimeLimit = new ProgramExecutionRequest.ProgramRuntimeLimitsFeature.RuntimeLimit<int> { Enabled = true, Limit = testInfo.RuntimeLimits.MaxRealTime },
                    MemoryUsageLimit = new ProgramExecutionRequest.ProgramRuntimeLimitsFeature.RuntimeLimit<long> { Enabled = true, Limit = testInfo.RuntimeLimits.PeakWorkingSet }
                },
                StandardStreamsRedirectionOptions = new ProgramExecutionRequest.ProgramStandardStreamsRedirectionOptions
                {
                    InputSourceFilePath = Path.Combine(programTempPath, TestingDataConfiguration.Verification.TestLib.DefaultInputFileName),
                    OutputRedirectionFileName = Path.Combine(programTempPath, TestingDataConfiguration.Verification.TestLib.DefaultOutputFileName)
                }
            };
        }

        protected TempDataDirectoryOperator ExecuteTestLibGenerator(TestingDataConfiguration.VerificationStruct.TestInfoStruct testInfo)
        {
            ISolutionInputGenerator generator;
            
            // Do we need to use a generator?
            if (testInfo.InputInfo.UseGenerator && TestingDataConfiguration.Verification.TestLib.GeneratorPath != null)
                generator = OvertestTestingLibraryAgent.GetGeneratorInstanceByPath(Path.Combine(
                    TestingDataAccessor.DirectoryFullName, TestingDataConfiguration.Verification.TestLib.GeneratorPath));
            else
                generator = new DefaultGenerator(); // default generator
            
            var inputDataStorageDirectoryAccessor = GetRandomTempDirectoryOperator();
            
            // Execute generator in the specified location
            generator.Execute(
                TestingDataConfiguration, TestingDataAccessor.DirectoryFullName,
                testInfo, inputDataStorageDirectoryAccessor.TempDirectoryFullName
            );
            
            return inputDataStorageDirectoryAccessor;
        }

        protected SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType ExecuteTestLibChecker(
            TestingDataConfiguration.VerificationStruct.TestInfoStruct testInfo,
            string userProgramOutputDirectoryPath, string authorProgramOutputDirectoryPath)
        {
            ISolutionOutputChecker checker;
            
            // Do we need to use a checker?
            if (testInfo.OutputInfo.UseChecker && TestingDataConfiguration.Verification.TestLib.CheckerPath != null)
                checker = OvertestTestingLibraryAgent.GetCheckerInstanceByPath(Path.Combine(
                    TestingDataAccessor.DirectoryFullName, TestingDataConfiguration.Verification.TestLib.CheckerPath));
            else
                checker = new DefaultChecker(); // default checker
            
            // Execute checker in the specified location
            return checker.Execute(
                TestingDataConfiguration, TestingDataAccessor.DirectoryFullName, testInfo,
                userProgramOutputDirectoryPath, authorProgramOutputDirectoryPath
            );
        }

        protected TempDataDirectoryOperator GetRandomTempDirectoryOperator()
        {
            var storageConfiguration = Configuration.GetSection("general:storage");
            var globalTempDirectoryPath = Path.Combine(
                storageConfiguration.GetValue<string>("path"),
                storageConfiguration.GetSection("subdirectories").GetValue<string>("temp_directory")
            );
            return new TempDataDirectoryOperator(globalTempDirectoryPath);
        }
    }
}