﻿using System;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.Extensions.Configuration;
using Newtonsoft.Json;
using NLog;
using Sirkadirov.Overtest.SharedLibraries.Database;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications;
using Sirkadirov.Overtest.SharedLibraries.Shared;
using Sirkadirov.Overtest.SharedLibraries.Shared.TestingApplications;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramCompilationAgent;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramExecutor;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramExecutor.DataStructures;
using Sirkadirov.Overtest.TestingAgent.Services.Storage;
using Sirkadirov.Overtest.TestingAgent.TestingServices.Skeleton;

namespace Sirkadirov.Overtest.TestingAgent.TestingServices.VerificationStages.TestingType
{
    public class SolutionReleaseTestingStage : VerificationStageBase
    {
        public SolutionReleaseTestingStage(
            IConfiguration configuration, OvertestDatabaseContext databaseContext,
            TestingApplication testingApplication,
            TempDirectoryAccessPoint testingDataAccessor, TestingDataConfiguration testingDataConfiguration,
            TempDataDirectoryOperator userProgramTempDirectory,
            OvertestProgramCompilationPluginBase userProgramCompilationPlugin
        ) : base(configuration, databaseContext, testingApplication, testingDataAccessor, testingDataConfiguration,
            userProgramTempDirectory, userProgramCompilationPlugin) {  }
        
        public override async Task ExecuteAsync()
        {
            var verificationDetailedResult = new SolutionReleaseVerificationDetailedResult();
            
            foreach (var test in TestingDataConfiguration.Verification.Tests)
            {
                /*
                 * > User's and author's solutions duplication section
                 *
                 * Here we creating two temporary folders, one of which
                 * is for user's solution and the other is for author's
                 * solution, then filling them with the duplicates of
                 * user's and author's solutions' executables respectively.
                 */
                
                using var currentTestUserTempDirectory = GetRandomTempDirectoryOperator();
                
                FileSystemSharedMethods.SecureCopyDirectory(
                    UserProgramTempDirectory.TempDirectoryFullName,
                    currentTestUserTempDirectory.TempDirectoryFullName
                );
                
                /*
                 * > Input generation and copying section
                 *
                 * Here we execution Overtest's Testing Library
                 * generator plugin (system default or specially
                 * developed and defined by programming task's
                 * author) and copying generated input files to
                 * user's and author's solutions' directories.
                 */
                
                using var inputDataAccessor = ExecuteTestLibGenerator(test);
                FileSystemSharedMethods.SecureCopyDirectory(
                    inputDataAccessor.TempDirectoryFullName,
                    currentTestUserTempDirectory.TempDirectoryFullName
                );
                // TODO: Copy input data to author's folder
                
                /*
                 * > User's and author's programs execution section.
                 * 
                 * Here we launch author's and user's solutions
                 * with specified runtime limits and collecting
                 * data about their execution results using default
                 * or user-defined checker.
                 */
                
                // TODO: Author's solution support
                var userProgramExecutionResult = await ExecuteUserSolution(test, currentTestUserTempDirectory.TempDirectoryFullName);
                
                LogManager.GetCurrentClassLogger().Error(userProgramExecutionResult.StandardOutputData);
                LogManager.GetCurrentClassLogger().Error(userProgramExecutionResult.StandardErrorData);
                
                GenerateTestingParticleResult(userProgramExecutionResult, test, currentTestUserTempDirectory.TempDirectoryFullName, null); // TODO: Author's solution support
            }
            
            TestingApplication.TestingResult.RawTestingResults = JsonConvert.SerializeObject(verificationDetailedResult, Formatting.Indented);
            ApplyAutomatedJudgement(verificationDetailedResult);
            
            async Task<ProgramExecutionResult> ExecuteUserSolution(TestingDataConfiguration.VerificationStruct.TestInfoStruct testInfo, string userDirectory)
            {
                using var programExecutor = new ProgramExecutor(GetProgramExecutorRequest(testInfo, userDirectory));
                return await programExecutor.ExecuteAsync();
            }

            void GenerateTestingParticleResult(ProgramExecutionResult userProgramExecutionResult, TestingDataConfiguration.VerificationStruct.TestInfoStruct testInfo, string userDirectory, string authorDirectory)
            {
                var testingParticleResult = GetReleaseTestingPartcleResult(userProgramExecutionResult, testInfo);
                if (testingParticleResult.Verdict == SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.Successful)
                    testingParticleResult.Verdict = ExecuteTestLibChecker(testInfo, userDirectory, authorDirectory);
                verificationDetailedResult.TestingParticles.Add(testingParticleResult);
            }
        }

        private SolutionReleaseVerificationDetailedResult.TestingParticleResult GetReleaseTestingPartcleResult(
            ProgramExecutionResult programExecutionResult, TestingDataConfiguration.VerificationStruct.TestInfoStruct test)
        {
            var particleResult = new SolutionReleaseVerificationDetailedResult.TestingParticleResult
            {
                TestTitle = test.Title,
                ExitCode = programExecutionResult.ProcessExitCode,

                ProcessorTimeUsed = programExecutionResult.RuntimeResourcesUsage.ProcessorTime,
                ProcessorTimeLimit = test.RuntimeLimits.MaxProcessorTime,

                WorkingSetUsed = programExecutionResult.RuntimeResourcesUsage.PeakMemoryUsage,
                WorkingSetLimit = test.RuntimeLimits.PeakWorkingSet,

                DiskSpaceUsed = programExecutionResult.RuntimeResourcesUsage.DiskSpaceUsage,
                DiskSpaceLimit = 0,

                RealExecutionTimeUsed = programExecutionResult.RuntimeResourcesUsage.RealExecutionTime,
                RealExecutionTimeLimit = test.RuntimeLimits.MaxRealTime
            };

            switch (programExecutionResult.Result)
            {
                case ProgramExecutionResult.ExecutionResultType.RuntimeError:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.RuntimeError;
                    break;
                case ProgramExecutionResult.ExecutionResultType.UsedProcessorTimeLimitReached:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.ProcessorTimeLimitReached;
                    break;
                case ProgramExecutionResult.ExecutionResultType.RealExecutionTimeLimitReached:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.IdlingTimeout;
                    break;
                case ProgramExecutionResult.ExecutionResultType.WorkingSetLimitReached:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.WorkingSetLimitReached;
                    break;
                case ProgramExecutionResult.ExecutionResultType.DiskSpaceUsageLimitReached:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.DiskSpaceLimitReached;
                    break;
                case ProgramExecutionResult.ExecutionResultType.StandardOutputLengthLimitReached:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.WrongOutputFormat;
                    break;
                case ProgramExecutionResult.ExecutionResultType.Successful:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.Successful;
                    break;
                default:
                    particleResult.Verdict = SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.VerificationFailed;
                    break;
            }

            return particleResult;
        }
        
        private void ApplyAutomatedJudgement(SolutionReleaseVerificationDetailedResult verificationDetailedResult)
        {
            var passedTestsCount = verificationDetailedResult.TestingParticles.Count(p =>
                p.Verdict == SolutionReleaseVerificationDetailedResult.TestingParticleResult.VerdictType.Successful
            );
            var totalTestsCount = TestingDataConfiguration.Verification.Tests.Count;

            if (passedTestsCount == totalTestsCount)
            {
                TestingApplication.TestingResult.SolutionAdjudgement = TestingApplicationResult.SolutionAdjudgementType.CompleteSolution;
                TestingApplication.TestingResult.GivenDifficulty = TestingApplication.ProgrammingTask.Difficulty;
            }
            else if (passedTestsCount == 0)
            {
                TestingApplication.TestingResult.SolutionAdjudgement = TestingApplicationResult.SolutionAdjudgementType.ZeroSolution;
                TestingApplication.TestingResult.GivenDifficulty = 0;
            }
            else
            {
                TestingApplication.TestingResult.SolutionAdjudgement = TestingApplicationResult.SolutionAdjudgementType.PartialSolution;
                TestingApplication.TestingResult.GivenDifficulty = Convert.ToByte((passedTestsCount / totalTestsCount) * TestingApplication.ProgrammingTask.Difficulty);
            }
        }
    }
}