﻿using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Sirkadirov.Overtest.TestingAgent.Libraries.ProgramExecutor.DataStructures;

namespace Sirkadirov.Overtest.TestingAgent.Libraries.ProgramExecutor
{
    public class ProgramExecutor : IDisposable
    {
        // ReSharper disable once MemberCanBePrivate.Global
        public ProgramExecutionRequest ExecutionRequest { get; }
        // ReSharper disable once MemberCanBePrivate.Global
        public ProgramExecutionResult ExecutionResult { get; }
        
        private readonly Process _process;
        private readonly SemaphoreSlim _stopExecutionSemaphore;
        
        public ProgramExecutor(ProgramExecutionRequest executionRequest)
        {
            _stopExecutionSemaphore = new SemaphoreSlim(1);
            
            ExecutionRequest = executionRequest;
            ExecutionResult = new ProgramExecutionResult();
            
            _process = new Process
            {
                EnableRaisingEvents = false,
                //PriorityClass = ProcessPriorityClass.BelowNormal,
                StartInfo = new ProcessStartInfo
                {
                    FileName = ExecutionRequest.StartInformation.Path,
                    Arguments = ExecutionRequest.StartInformation.Arguments,
                    WorkingDirectory = ExecutionRequest.StartInformation.WorkingDirectory,
                    RedirectStandardInput = true, RedirectStandardOutput = true, RedirectStandardError = true,
                    
                    StandardInputEncoding = Encoding.ASCII, StandardOutputEncoding = Encoding.ASCII, StandardErrorEncoding = Encoding.ASCII,
                    
                    CreateNoWindow = true, WindowStyle = ProcessWindowStyle.Hidden,
                    UseShellExecute = false, ErrorDialog = false
                }
            };
            
            // Add custom environment variables
            //_process.StartInfo.Environment.Add(ExecutionRequest.StartInformation.Environment);
            
            if (ExecutionRequest.RunAsFeature.Enabled)
            {
                _process.StartInfo.UserName = ExecutionRequest.RunAsFeature.UserName;
                    
                // To workaround not implemented features in current .NET Core release
                if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
                {
                    // No implementation in other systems
                    _process.StartInfo.PasswordInClearText = ExecutionRequest.RunAsFeature.UserPasswordClearText;
                        
                    // Windows-only features
                    _process.StartInfo.Domain = ExecutionRequest.RunAsFeature.UserDomain;
                    _process.StartInfo.LoadUserProfile = false;
                }
                else
                {
                    // TODO: Find an alternative way to do that!
                    throw new NotImplementedException("Operating systems except Windows are not supported yet!");
                }
            }
        }

        // ReSharper disable once UnusedMember.Global
        public async Task<ProgramExecutionResult> ExecuteAsync()
        {
            try { _process.Start(); }
            catch (Exception ex) { throw new ExecutionException("Method execution failed!", ex, ExecutionResult); }
            
            try { await WriteInput(); }
            catch (Exception ex) { throw new StreamRedirectionExecutionException(ExecutionResult, ex); }

            new Thread(ExecuteProcessResourcesUsageWatcher) { Priority = ThreadPriority.Highest }.Start();

            /*if (ExecutionRequest.RuntimeLimitsFeature.RealTimeLimit.Enabled)
            {
                *
                 * When we use [ bool Process.WaitForExit(int milliseconds) ]
                 * we need to call [ void Process.WaitForExit() ] afterwards
                 * to work around the "Intermittent: Empty Process stdout" issue.
                 * More information: https://github.com/dotnet/runtime/issues/27128
                 *
                
                if (!_process.WaitForExit(ExecutionRequest.RuntimeLimitsFeature.RealTimeLimit.Limit))
                {
                    FinalizeExecution(ProgramExecutionResult.ExecutionResultType.RealExecutionTimeLimitReached);
                }
            }*/
            
            // Temporary workaround
            _process.WaitForExit();
            
            // Get process information, output and resources usage
            FinalizeExecution(ProgramExecutionResult.ExecutionResultType.Successful);

            // Return execution result to the caller
            return ExecutionResult;
        }
        
        private async Task WriteInput()
        {
            var inputDataSourcePath = ExecutionRequest.StandardStreamsRedirectionOptions.InputSourceFilePath;

            if (!string.IsNullOrWhiteSpace(inputDataSourcePath) && File.Exists(inputDataSourcePath))
            {
                var inputData = (await File.ReadAllTextAsync(inputDataSourcePath, Encoding.ASCII))
                    .Replace("\r", string.Empty)
                    .TrimEnd('\r', '\n')
                    .Replace("\n", "\r\n");

                if (!inputData.EndsWith("\r\n"))
                    inputData += "\r\n";
                    
                await _process.StandardInput.WriteAsync(inputData);
                await _process.StandardInput.FlushAsync();
            }
        }
        
        private void ExecuteProcessResourcesUsageWatcher()
        {
            try
            {
                while (!_process.HasExited)
                {
                    _process.Refresh();
                    UpdateProcessResourcesUsage();

                    /* Processor time usage limit */
                    if (ExecutionRequest.RuntimeLimitsFeature.ProcessorTimeLimit.Enabled)
                    {
                        if (ExecutionRequest.RuntimeLimitsFeature.ProcessorTimeLimit.Limit <
                            ExecutionResult.RuntimeResourcesUsage.ProcessorTime)
                            FinalizeExecution(ProgramExecutionResult.ExecutionResultType.UsedProcessorTimeLimitReached);
                    }

                    /* Memory usage limit */
                    if (ExecutionRequest.RuntimeLimitsFeature.MemoryUsageLimit.Enabled)
                    {
                        if (ExecutionRequest.RuntimeLimitsFeature.MemoryUsageLimit.Limit <
                            ExecutionResult.RuntimeResourcesUsage.PeakMemoryUsage)
                            FinalizeExecution(ProgramExecutionResult.ExecutionResultType.WorkingSetLimitReached);
                    }
                    
                    /* Real execution time limit */
                    if (ExecutionRequest.RuntimeLimitsFeature.RealTimeLimit.Enabled)
                    {
                        if ((DateTime.Now - _process.StartTime).TotalMilliseconds >
                            ExecutionRequest.RuntimeLimitsFeature.RealTimeLimit.Limit)
                            FinalizeExecution(ProgramExecutionResult.ExecutionResultType.RealExecutionTimeLimitReached);
                    }
                }
            }
            catch (Exception) { /* We don't need to do anything */ }
        }
        
        private void FinalizeExecution(ProgramExecutionResult.ExecutionResultType executionResult)
        {
            _stopExecutionSemaphore.Wait();

            if (ExecutionResult.Result != ProgramExecutionResult.ExecutionResultType.Unset)
            {
                _stopExecutionSemaphore.Release();
                return;
            }
            
            try
            {
                ExecutionResult.Result = executionResult;
                
                // Try to kill an associated process tree
                if (!_process.HasExited)
                {
                    _process.Kill(true);
                    try { _process.WaitForExit(); }
                    catch (Exception) { /* Ignore, just to workaround an old good issue in .NET Core */ }
                }
                
                UpdateProcessResourcesUsage();

                if (ExecutionResult.Result != ProgramExecutionResult.ExecutionResultType.RealExecutionTimeLimitReached)
                {
                    ExecutionResult.StandardOutputData = _process.StandardOutput.ReadToEnd();
                    ExecutionResult.StandardErrorData = _process.StandardError.ReadToEnd();
                }
                
                ExecutionResult.ProcessExitCode = _process.ExitCode;
                
                if (executionResult == ProgramExecutionResult.ExecutionResultType.Successful && ExecutionResult.ProcessExitCode != 0)
                    ExecutionResult.Result = ProgramExecutionResult.ExecutionResultType.RuntimeError;
                
                /*
                 * Save output data
                 */
                
                if (!string.IsNullOrWhiteSpace(ExecutionRequest.StandardStreamsRedirectionOptions.OutputRedirectionFileName))
                {
                    File.WriteAllText(
                        Path.Combine(
                            ExecutionRequest.StartInformation.WorkingDirectory,
                            ExecutionRequest.StandardStreamsRedirectionOptions.OutputRedirectionFileName
                        ),
                        ExecutionResult.StandardOutputData, Encoding.UTF8
                    );
                }
            }
            finally
            {
                _stopExecutionSemaphore.Release();
            }
        }
        
        private void UpdateProcessResourcesUsage()
        {
            /*
             * Need to find possible workarounds to do disk space usage calculations faster.
             * 
             * var diskSpaceUsage = new DirectoryInfo(ExecutionRequest.StartInformation.WorkingDirectory)
             *    .GetFiles("*", SearchOption.AllDirectories).Sum(f => f.Length);
             *
             * ExecutionResult.RuntimeResourcesUsage.DiskSpaceUsage = diskSpaceUsage;
             */
            
            if (_process.TotalProcessorTime.TotalMilliseconds >= 0)
                ExecutionResult.RuntimeResourcesUsage.ProcessorTime = Convert.ToInt64(_process.TotalProcessorTime.TotalMilliseconds);

            if (_process.HasExited)
            {
                ExecutionResult.RuntimeResourcesUsage.RealExecutionTime = Convert.ToInt64(Math.Round((_process.ExitTime - _process.StartTime).TotalMilliseconds));
            }
            else
            {
                ExecutionResult.RuntimeResourcesUsage.PeakMemoryUsage = _process.PeakWorkingSet64;
            }
        }
        
        public void Dispose()
        {
            _stopExecutionSemaphore?.Dispose();
            _process?.Close();
            _process?.Dispose();
        }
    }
}