﻿using System.IO;
using System.Text;
using Microsoft.Extensions.Configuration;
using NLog;
using NLog.Config;
using NLog.Layouts;
using NLog.Targets;

namespace Sirkadirov.Overtest.SharedLibraries.Shared
{
    public static class LoggingSharedMethods
    {
        public static LoggingConfiguration ConfigureLogging(this IConfiguration configuration)
        {
            var loggingConfiguration = new LoggingConfiguration();
            IConfiguration userLoggingConfiguration = configuration.GetSection("diagnostics:logging");
            
            if (!userLoggingConfiguration.GetValue<bool>("enabled"))
                return loggingConfiguration;
            
            var userLoggingTargets = userLoggingConfiguration.GetSection("targets");
            var loggingLayout = new SimpleLayout("${longdate}|${logger}|${level:uppercase=true}|${message} ${all-event-properties:includeEmptyValues=true}");
            
            ConfigureConsoleLoggingTarget();
            ConfigureFilesLoggingTarget();
                
            void ConfigureConsoleLoggingTarget()
            {
                var consoleTargetConfiguration = userLoggingTargets.GetSection("console");
                
                if (!consoleTargetConfiguration.GetValue<bool>("enabled"))
                    return;
                
                var nLogColoredConsoleTarget = new ColoredConsoleTarget("console")
                {
                    Encoding = Encoding.UTF8,
                    
                    DetectConsoleAvailable = true,
                    DetectOutputRedirected = true,
                    
                    UseDefaultRowHighlightingRules = false,
                    Layout = loggingLayout
                    //AutoFlush = true
                };
                
                nLogColoredConsoleTarget.RowHighlightingRules.Add(
                    new ConsoleRowHighlightingRule(
                        "level == LogLevel.Trace",
                        ConsoleOutputColor.DarkGray,
                        ConsoleOutputColor.NoChange
                    )
                );
                
                nLogColoredConsoleTarget.RowHighlightingRules.Add(
                    new ConsoleRowHighlightingRule(
                        "level == LogLevel.Debug",
                        ConsoleOutputColor.Gray,
                        ConsoleOutputColor.NoChange
                    )
                );
                
                nLogColoredConsoleTarget.RowHighlightingRules.Add(
                    new ConsoleRowHighlightingRule(
                        "level == LogLevel.Info",
                        ConsoleOutputColor.Cyan,
                        ConsoleOutputColor.NoChange
                    )
                );
                
                nLogColoredConsoleTarget.RowHighlightingRules.Add(
                    new ConsoleRowHighlightingRule(
                        "level == LogLevel.Warn",
                        ConsoleOutputColor.DarkYellow,
                        ConsoleOutputColor.NoChange
                    )
                );
                
                nLogColoredConsoleTarget.RowHighlightingRules.Add(
                    new ConsoleRowHighlightingRule(
                        "level == LogLevel.Error",
                        ConsoleOutputColor.Magenta,
                        ConsoleOutputColor.NoChange
                    )
                );
                
                nLogColoredConsoleTarget.RowHighlightingRules.Add(
                    new ConsoleRowHighlightingRule(
                        "level == LogLevel.Fatal",
                        ConsoleOutputColor.Red,
                        ConsoleOutputColor.NoChange
                    )
                );
                
                loggingConfiguration.AddRule(
                    LogLevel.FromString(consoleTargetConfiguration["levels:minimum"]),
                    LogLevel.FromString(consoleTargetConfiguration["levels:maximum"]),
                    nLogColoredConsoleTarget
                );
            }
            
            void ConfigureFilesLoggingTarget()
            {
                var fileTargetConfiguration = userLoggingTargets.GetSection("files");
                
                if (!fileTargetConfiguration.GetValue<bool>("enabled"))
                    return;
                
                var nLogFileTarget = new FileTarget
                {
                    FileName = Path.Combine(fileTargetConfiguration["path"], "overtest.log"),
                    CreateDirs = true,
                    
                    KeepFileOpen = true,
                    Layout = loggingLayout,
                    Encoding = Encoding.UTF8,
                    WriteBom = false,
                    LineEnding = LineEndingMode.CRLF,
                    
                    ArchiveEvery = FileArchivePeriod.Hour,
                    ArchiveNumbering = ArchiveNumberingMode.Sequence
                };
                
                loggingConfiguration.AddRule(
                    LogLevel.FromString(fileTargetConfiguration["levels:minimum"]),
                    LogLevel.FromString(fileTargetConfiguration["levels:maximum"]),
                    nLogFileTarget
                );
            }
            
            return loggingConfiguration;
        }
    }
}