﻿using System;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Identity;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Sirkadirov.Overtest.SharedLibraries.Database;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.Identity;
using Sirkadirov.Overtest.WebApplication.Areas.Administration.Models.CuratorUsersController;
using Sirkadirov.Overtest.WebApplication.Extensions.Filters;

namespace Sirkadirov.Overtest.WebApplication.Areas.Administration.Controllers
{
    
    [Area("Administration"), Route("/Administration/Social/Curators/")]
    [AllowedUserTypesFilter(UserType.SuperUser)]
    public class CuratorUsersController : Controller
    {
        private const string ViewsDirectoryPath = "~/Areas/Administration/Views/Social/CuratorUsersController/";
        
        private readonly OvertestDatabaseContext _databaseContext;
        private readonly UserManager<User> _userManager;

        public CuratorUsersController(OvertestDatabaseContext databaseContext, UserManager<User> userManager)
        {
            _databaseContext = databaseContext;
            _userManager = userManager;
        }
        
        [HttpGet, Route(nameof(List))]
        public async Task<IActionResult> List()
        {
            const string actionViewPath = ViewsDirectoryPath + nameof(List) + ".cshtml";
            
            var curatorsList = await _databaseContext.Users
                .Where(u => u.Type != UserType.Student || u.UserGroupId == null)
                .Select(s => new CuratorUserInfoModel
                {
                    Id = s.Id,
                    FullName = s.FullName,
                    InstitutionName = s.InstitutionName,
                    Type = s.Type
                })
                .OrderBy(o => o.FullName)
                .ToListAsync();
            
            return View(actionViewPath, curatorsList);
        }
        
        [HttpGet, Route(nameof(Create))]
        public async Task<IActionResult> Create()
        {
            const string actionViewPath = ViewsDirectoryPath + nameof(Create) + ".cshtml";
            return View(actionViewPath, new CuratorUserCreationModel
            {
                InstitutionName = await _databaseContext.Users
                    .Where(u => u.Id.ToString() == _userManager.GetUserId(HttpContext.User))
                    .Select(s => s.InstitutionName)
                    .FirstAsync()
            });
        }
        
        [HttpPost, Route(nameof(Create))]
        public async Task<IActionResult> Create(CuratorUserCreationModel model)
        {
            const string actionViewPath = ViewsDirectoryPath + nameof(Create) + ".cshtml";

            if (!ModelState.IsValid)
                return View(actionViewPath, model);

            var identityResult = await _userManager.CreateAsync(new User
            {
                Type = UserType.Curator,
                IsBanned = false,
                
                Email = model.Email,
                UserName = model.Email,
                
                FullName = model.FullName,
                InstitutionName = model.InstitutionName,
                
                Registered = DateTime.UtcNow,
                LastSeen = DateTime.UtcNow,
                
                UserGroupId = null,
                UserPhotoId = null
            }, model.Password);

            if (!identityResult.Succeeded)
            {
                foreach (var error in identityResult.Errors)
                {
                    ModelState.AddModelError(string.Empty, error.Description);
                }
                return View(actionViewPath, model);
            }

            return RedirectToAction("List", "CuratorUsers", new { area = "Administration" });
        }
        
    }
}