﻿using System;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Identity;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Localization;
using Sirkadirov.Overtest.SharedLibraries.Database;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.Identity;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications;
using Sirkadirov.Overtest.SharedLibraries.Database.Storage.TestingApplications.Extras;
using Sirkadirov.Overtest.WebApplication.Areas.TasksArchive.ViewComponents.Models.ProgrammingTasksBrowsingController;

namespace Sirkadirov.Overtest.WebApplication.Areas.TasksArchive.Controllers
{
    [Area("TasksArchive")]
    [Route("/TasksArchive/TestingApplications")]
    public class ProgrammingTasksTestingApplicationsController : Controller
    {
        private const string ViewsDirectoryPath = "~/Areas/TasksArchive/Views/ProgrammingTasksTestingApplicationsController/";
        
        private readonly OvertestDatabaseContext _databaseContext;
        private readonly UserManager<User> _userManager;
        private readonly IStringLocalizer<ProgrammingTasksTestingApplicationsController> _localizer;

        public ProgrammingTasksTestingApplicationsController(OvertestDatabaseContext databaseContext,
            UserManager<User> userManager, IStringLocalizer<ProgrammingTasksTestingApplicationsController> localizer)
        {
            _databaseContext = databaseContext;
            _userManager = userManager;
            _localizer = localizer;
        }
        
        #region View and submit a single testing application
        
        [HttpGet, Route(nameof(View) + "/{testingApplicationId:guid}")]
        public async Task<IActionResult> View(Guid testingApplicationId)
        {
            const string actionViewPath = ViewsDirectoryPath + nameof(View) + ".cshtml";
            
            var currentUserId = new Guid(_userManager.GetUserId(HttpContext.User));
            var testingApplicationQuery = _databaseContext.TestingApplications
                .Where(a => a.Id == testingApplicationId)
                .Include(i => i.TestingResult)
                .ThenInclude(r => r.CompilationResult)
                .AsNoTracking();

            if (!await testingApplicationQuery.AnyAsync())
                return NotFound();

            var hasAccessRights = await _databaseContext.UserPermissionsOperator.GetUserDataEditPermissionAsync(
                await testingApplicationQuery.Select(s => s.AuthorId).FirstAsync(),
                currentUserId
            );

            if (!hasAccessRights)
                return Forbid();

            var testingApplication = await testingApplicationQuery.FirstAsync();
            
            testingApplication.TestingResult ??= new TestingApplicationResult();

            return View(actionViewPath, testingApplication);
        }

        [HttpGet, Route(nameof(GetCurrentVerificationStatus) + "/{testingApplicationId:guid}")]
        public async Task<IActionResult> GetCurrentVerificationStatus(Guid testingApplicationId)
        {
            var currentUserId = new Guid(_userManager.GetUserId(HttpContext.User));
            var testingApplicationQuery = _databaseContext.TestingApplications
                .Where(a => a.Id == testingApplicationId)
                .AsNoTracking();

            if (!await testingApplicationQuery.AnyAsync())
                return NotFound();

            var hasAccessRights = await _databaseContext.UserPermissionsOperator.GetUserDataEditPermissionAsync(
                await testingApplicationQuery.Select(s => s.AuthorId).FirstAsync(),
                currentUserId
            );
            
            if (!hasAccessRights)
                return Forbid();
            
            // Return current solution's verification process' status as plain text
            return Content((await testingApplicationQuery.Select(s => s.Status).FirstAsync()).ToString());
        }
        
        [HttpPost, ValidateAntiForgeryToken, Route(nameof(SubmitTestingApplication))]
        public async Task<IActionResult> SubmitTestingApplication(TestingApplicationCreationPartialModel model)
        {
            if (!ModelState.IsValid || !model.ProgrammingLanguageId.HasValue || !model.TestingType.HasValue)
                return BadRequest(ModelState.ToList());
            
            var currentUserId = new Guid(_userManager.GetUserId(HttpContext.User));
            var isSuperUser = await _databaseContext.UserPermissionsOperator.GetUserHasSpecifiedTypeAsync(currentUserId, UserType.SuperUser);
            
            if (!await _databaseContext.ProgrammingTasks.AnyAsync(t => t.Id == model.ProgrammingTaskId))
                return NotFound();
            
            // TODO: Disabled for development and testing stages only
            if (!await _databaseContext.ProgrammingTasks.AnyAsync(t => t.Id == model.ProgrammingTaskId && t.TestingData.DataPackageFile != null))
                return BadRequest(_localizer["Обране вами завдання не містить даних, необхідних для його тестування! Зв'яжіться з автором завдання чи адміністратором системи."].Value);
            
            var programmingTaskInfo = await _databaseContext.ProgrammingTasks
                .Where(t => t.Id == model.ProgrammingTaskId)
                .Select(s => new
                {
                    s.VisibleInFreeMode,
                    s.VisibleInCompetitionMode
                })
                .FirstAsync();
            
            var isCompetitionModeApplication = false;
            
            if (!isSuperUser)
            {
                // ReSharper disable once ConditionIsAlwaysTrueOrFalse
                if (!isCompetitionModeApplication && !programmingTaskInfo.VisibleInFreeMode)
                    return Forbid();

                // ReSharper disable once ConditionIsAlwaysTrueOrFalse
                if (isCompetitionModeApplication && !programmingTaskInfo.VisibleInCompetitionMode)
                    return Forbid();
            }
            
            // Create testing application prototype
            var testingApplication = new TestingApplication
            {
                ProgrammingTaskId = model.ProgrammingTaskId,
                AuthorId = currentUserId,
                CompetitionId = null,
                
                Created = DateTime.UtcNow,
                
                SourceCode = new TestingApplicationSourceCode
                {
                    ProgrammingLanguageId = model.ProgrammingLanguageId.Value,
                    SourceCode = Encoding.UTF8.GetBytes(model.SourceCode.Trim().Replace("\r", string.Empty).Replace("\n", "\r\n"))
                },
                TestingType = model.TestingType.Value
            };
            
            // TODO: Implement FLOOD and DDOS security

            try
            {
                await using var transaction = await _databaseContext.Database.BeginTransactionAsync();
                
                var previousWaitingSubmissionsQuery = _databaseContext.TestingApplications
                    .Where(a => a.Status == TestingApplication.ApplicationStatus.Waiting)
                    .Where(a => a.ProgrammingTaskId == model.ProgrammingTaskId)
                    .Where(a => a.AuthorId == currentUserId);
                
                if (await previousWaitingSubmissionsQuery.AnyAsync())
                {
                    var previousWaitingSubmissions = await previousWaitingSubmissionsQuery
                        .Select(s => new TestingApplication { Id = s.Id, Status = s.Status})
                        .AsNoTracking()
                        .ToListAsync();
                    
                    _databaseContext.TestingApplications.AttachRange(previousWaitingSubmissions);
                    _databaseContext.TestingApplications.RemoveRange(previousWaitingSubmissions);
                }

                await _databaseContext.SaveChangesAsync();
                await transaction.CommitAsync();
            }
            catch (Exception) { /* Not required */ }
            
            try
            {
                var submissionsCurrentlyInTesting = await _databaseContext.TestingApplications
                    .Where(a => a.AuthorId == testingApplication.AuthorId)
                    .Where(a => a.CompetitionId == testingApplication.CompetitionId)
                    .Where(a => a.ProgrammingTaskId == testingApplication.ProgrammingTaskId)
                    .Where(a => a.Status != TestingApplication.ApplicationStatus.Waiting)
                    .Where(a => a.Status != TestingApplication.ApplicationStatus.Verified)
                    .AsNoTracking()
                    .AnyAsync();

                if (submissionsCurrentlyInTesting)
                    return BadRequest("Other submissions are currently in testing! Please, wait!"); // TODO: Make something beautiful
            }
            catch (Exception) { /* Not required */ }
            
            if (testingApplication.TestingType == TestingApplication.ApplicationTestingType.ReleaseMode)
            {
                var previousSubmissionsList = await _databaseContext.TestingApplications
                    .Where(a => a.AuthorId == testingApplication.AuthorId)
                    .Where(a => a.ProgrammingTaskId == testingApplication.ProgrammingTaskId)
                    .Where(a => a.CompetitionId == testingApplication.CompetitionId)
                    .Where(a => a.Status != TestingApplication.ApplicationStatus.Selected)
                    .Select(s => new TestingApplication { Id = s.Id, Status = s.Status })
                    .ToListAsync();
            
                foreach (var item in previousSubmissionsList)
                    _databaseContext.TestingApplications.Remove(item);
            }
            
            // Add a newly-created testing application to the database
            await _databaseContext.TestingApplications.AddAsync(testingApplication);
            
            // Save changes made to the database
            await _databaseContext.SaveChangesAsync();
            
            return RedirectToAction(nameof(View), new { testingApplicationId = testingApplication.Id });
        }
        
        #endregion
        
        #region Submissions listing for specified user
        
        [HttpGet, Route(nameof(List) + "/{userId:guid}")]
        public async Task<IActionResult> List(Guid userId, Guid? competitionId = null)
        {
            throw new NotImplementedException();
        }
        
        [HttpGet, Route("User/{userId:guid}/ProgrammingTask/{programmingTaskId:guid}")]
        public async Task<IActionResult> GetUserSubmissionsForProgrammingTask(Guid userId, Guid programmingTaskId, Guid? competitionId = null)
        {
            const string actionViewPath = ViewsDirectoryPath + nameof(GetUserSubmissionsForProgrammingTask) + ".cshtml";
            
            var currentUserId = new Guid(_userManager.GetUserId(HttpContext.User));
            var hasAccessRights = await _databaseContext.UserPermissionsOperator.GetUserDataEditPermissionAsync(userId, currentUserId);

            if (!hasAccessRights)
                return Forbid();

            ViewBag.UserId = userId;
            ViewBag.ProgrammingTaskId = programmingTaskId;
            ViewBag.CompetitionId = competitionId;

            var testingApplicationsList = await _databaseContext.TestingApplications
                .Include(i => i.TestingResult)
                .Where(a => a.CompetitionId == competitionId)
                .Where(a => a.AuthorId == userId)
                .Where(a => a.ProgrammingTaskId == programmingTaskId)
                .OrderByDescending(o => o.Created)
                .Select(s => new TestingApplication
                {
                    Id = s.Id,
                    Created = s.Created,
                    TestingType = s.TestingType,
                    Status = s.Status,
                    TestingResult = new TestingApplicationResult
                    {
                        GivenDifficulty = s.TestingResult.GivenDifficulty,
                        SolutionAdjudgement = s.TestingResult.SolutionAdjudgement
                    }
                }).AsNoTracking()
                .ToListAsync();

            return View(actionViewPath, testingApplicationsList);
        }
        
        #endregion
    }
}