/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.rewriter.helpers.impl;

import javax.faces.context.FacesContext;
import javax.portlet.PortletURL;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.dto.FileNodeDto;
import jp.sf.pal.cms.helper.DownloadHelper;
import jp.sf.pal.cms.rewriter.helpers.RewriteHandler;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * @author shinsuke
 * 
 */
public class RewriteHandlerImpl extends DefaultHandler implements
        RewriteHandler {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(RewriteHandlerImpl.class);

    private static final String BODY_TAG = "body";

    private StringBuffer buf;

    private boolean write;

    public RewriteHandlerImpl() {
        buf = new StringBuffer();
        write = false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.cms.rewriter.helpers.RewriteHandler#getContent()
     */
    public String getContent() {
        if (log.isDebugEnabled()) {
            log.debug("getContent() - content=" + buf.toString());
        }

        return buf.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int length)
            throws SAXException {
        if (!write) {
            return;
        }
        buf.append(new String(ch, start, length));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.xml.sax.helpers.DefaultHandler#endDocument()
     */
    public void endDocument() throws SAXException {
        if (log.isDebugEnabled()) {
            log.debug("endDocument()");
            buf.append("<!-- End Document(RewriteHandlerImpl) -->");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qName)
            throws SAXException {
        if (write && qName.equalsIgnoreCase(BODY_TAG)) {
            write = false;
            return;
        }
        if (!write) {
            return;
        }
        buf.append("</");
        buf.append(qName.toLowerCase());
        buf.append(">");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.xml.sax.helpers.DefaultHandler#startDocument()
     */
    public void startDocument() throws SAXException {
        if (log.isDebugEnabled()) {
            log.debug("startDocument()");
            buf.append("<!-- Start Document(RewriteHandlerImpl) -->");
        }
        S2Container container = SingletonS2ContainerFactory.getContainer();
        DownloadHelper downloadHelper = (DownloadHelper) container
                .getComponent(DownloadHelper.class);
        downloadHelper.reset();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String,
     *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qName,
            Attributes attributes) throws SAXException {
        if (log.isDebugEnabled()) {
            log.debug("startElement(String, String, String, Attributes) - uri="
                    + uri + ", localName=" + localName + ", qName=" + qName
                    + ", attributes=" + attributes);
        }
        if (!write && qName.equalsIgnoreCase(BODY_TAG)) {
            write = true;
            return;
        }
        if (!write) {
            return;
        }
        buf.append("<");
        buf.append(qName.toLowerCase());
        if (qName.equalsIgnoreCase("a")) {
            // a
            String href = null;
            String target = null;
            for (int i = 0; i < attributes.getLength(); i++) {
                String attrQName = attributes.getQName(i);
                String attrValue = attributes.getValue(i);
                if (attrQName.equalsIgnoreCase("href")) {
                    href = attrValue;
                } else if (attrQName.equalsIgnoreCase("target")) {
                    target = attrValue;
                } else {
                    buf.append(" ");
                    buf.append(attrQName);
                    buf.append("=\"");
                    buf.append(attrValue);
                    buf.append("\"");
                }
            }
            if (log.isDebugEnabled()) {
                log
                        .debug("startElement(String, String, String, Attributes) - href="
                                + href + ", target=" + target);
            }
            if (href != null) {
                if (target != null) {
                    if (target.startsWith(CMSConstants.CONTENT_TARGET_PREFIX)) {
                        buf.append(" ");
                        buf.append("href");
                        buf.append("=\"");
                        buf.append(rewriteUrl(href, target
                                .substring(CMSConstants.CONTENT_TARGET_PREFIX
                                        .length())));
                        buf.append("\"");
                    } else {
                        buf.append(" ");
                        buf.append("href");
                        buf.append("=\"");
                        buf.append(rewriteUrl(href, null));
                        buf.append("\"");

                        buf.append(" ");
                        buf.append("target");
                        buf.append("=\"");
                        buf.append(target);
                        buf.append("\"");
                    }
                } else {
                    buf.append(" ");
                    buf.append("href");
                    buf.append("=\"");
                    buf.append(rewriteUrl(href, null));
                    buf.append("\"");
                }
            } else if (target != null) {
                buf.append(" ");
                buf.append("target");
                buf.append("=\"");
                buf.append(target);
                buf.append("\"");
            }
        } else if (qName.equalsIgnoreCase("img")) {
            // img
            for (int i = 0; i < attributes.getLength(); i++) {
                String attrQName = attributes.getQName(i);
                String attrValue = attributes.getValue(i);
                if (attrQName.equalsIgnoreCase("src")) {
                    // src
                    buf.append(" ");
                    buf.append(attrQName);
                    buf.append("=\"");
                    buf.append(rewriteUrl(attrValue, null));
                    buf.append("\"");
                } else {
                    buf.append(" ");
                    buf.append(attrQName);
                    buf.append("=\"");
                    buf.append(attrValue);
                    buf.append("\"");
                }
            }
        } else {
            for (int i = 0; i < attributes.getLength(); i++) {
                buf.append(" ");
                buf.append(attributes.getQName(i));
                buf.append("=\"");
                buf.append(attributes.getValue(i));
                buf.append("\"");
            }
        }
        buf.append(">");

    }

    protected String rewriteUrl(String url, String target) {
        if (log.isDebugEnabled()) {
            log.debug("rewriteUrl(String) - url=" + url);
        }
        if (url.startsWith(CMSConstants.CONTENT_MANAGEMENT_PROTOCOL)) {
            String path = url
                    .substring(CMSConstants.CONTENT_MANAGEMENT_PROTOCOL
                            .length());
            if (log.isDebugEnabled()) {
                log.debug("rewriteUrl(String) - path=" + path);
            }
            try {
                S2Container container = SingletonS2ContainerFactory
                        .getContainer();
                FileNodeDao fileNodeDao = (FileNodeDao) container
                        .getComponent(FileNodeDao.class);
                FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                        .getScopeFromRequest(), path);
                if (fileNode != null) {
                    if (fileNode.getMimeType() != null
                            && fileNode.getMimeType().equals(
                                    CMSConstants.DEFAULT_HTML_MIMETYPE)) {
                        PortletURL portletUrl = CMSUtil.getRenderResponse()
                                .createActionURL();
                        portletUrl
                                .setParameter(CMSConstants.CONTENT_PATH, path);
                        if (target == null) {
                            target = CMSUtil.getRenderResponse().getNamespace();
                        }
                        portletUrl.setParameter(CMSConstants.CONTENT_TARGET,
                                target);
                        if (log.isDebugEnabled()) {
                            log.debug("rewriteUrl(String) - url=" + url
                                    + ", path=" + path + ", target=" + target);
                            log.debug("rewriteUrl(String) - portletUrl="
                                    + portletUrl);
                        }
                        return portletUrl.toString();
                    } else {
                        DownloadHelper downloadHelper = (DownloadHelper) container
                                .getComponent(DownloadHelper.class);
                        String key = downloadHelper.publish(CMSUtil
                                .getScopeFromRequest()
                                + CMSConstants.KEY_SEPARATOR + path);
                        if (key == null) {
                            log.error("Invalid url: " + url);
                            return url;
                        }
                        StringBuffer buf = new StringBuffer(FacesContext
                                .getCurrentInstance().getExternalContext()
                                .getRequestContextPath());
                        buf.append(CMSConstants.FILE_SERVLET);
                        buf.append("?");
                        buf.append(CMSConstants.FILE_ID);
                        buf.append("=");
                        buf.append(key);
                        return buf.toString();
                    }
                } else {
                    log.error("Invalid url: " + url);
                    return url;
                }
            } catch (CMSException e) {
                log.error("Invalid url: " + url, e);
                return url;
            }
        } else {
            return url;
        }
    }

}
