/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.util;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Map;

import javax.faces.context.FacesContext;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.ValueFormatException;
import javax.jcr.lock.LockException;
import javax.jcr.version.VersionException;
import javax.portlet.PortletConfig;
import javax.portlet.PortletContext;
import javax.portlet.PortletRequest;
import javax.portlet.RenderResponse;
import javax.servlet.ServletContext;

import jp.sf.pal.cms.CMSConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class CMSUtil {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(CMSUtil.class);

    public static String normalizedPath(String path) {
        // TODO
        path = new String(path).replaceAll(CMSConstants.PATH_SEPARATOR + "+",
                CMSConstants.PATH_SEPARATOR);
        return path;
    }

    public static String getParentPath(String path) {
        path = normalizedPath(path);
        int last = path.lastIndexOf(CMSConstants.PATH_SEPARATOR);
        if (path.endsWith(CMSConstants.PATH_SEPARATOR)) {
            last = path.lastIndexOf(CMSConstants.PATH_SEPARATOR, last);
        }
        if (log.isDebugEnabled()) {
            log.debug("getParentPath(String) - path=" + path);
            log.debug("getParentPath(String) - parentPath="
                    + path.substring(0, last));
            log.debug("getParentPath(String) - last=" + last);
        }
        if (last != 0) {
            return path.substring(0, last);
        } else {
            return CMSConstants.PATH_SEPARATOR;
        }
    }

    public static void setScopeToRequest(PortletRequest request,
            String scopeName) {
        request.setAttribute(CMSConstants.SCOPE, scopeName);
    }

    /**
     * Returns the scope name during init()
     * 
     * @param request
     * @return
     */
    public static String getScopeFromRequest(PortletRequest request) {
        String scopeName = (String) request.getAttribute(CMSConstants.SCOPE);
        if (scopeName != null) {
            return scopeName;
        }
        // TODO NPE from getPortletConfig
        scopeName = getPortletConfig(request).getInitParameter(
                CMSConstants.SCOPE);
        if (scopeName != null) {
            return scopeName;
        }
        return CMSConstants.DEFAULT_SCOPE_VALUE;
    }

    public static PortletConfig getPortletConfig(PortletRequest request) {
        String name = "javax.portlet.PortletConfig";
        return (PortletConfig) request.getAttribute(name);
    }

    /**
     * Returns the scope name during processAction() and render()
     * 
     * @param request
     * @return
     */
    public static String getScopeFromRequest() {
        String scopeName = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getRequestMap().get(CMSConstants.SCOPE);
        if (scopeName != null) {
            return scopeName;
        }
        // TODO NPE from getPortletConfig
        scopeName = getPortletConfig().getInitParameter(CMSConstants.SCOPE);
        if (scopeName != null) {
            return scopeName;
        }
        return CMSConstants.DEFAULT_SCOPE_VALUE;

    }

    public static PortletConfig getPortletConfig() {
        String name = "javax.portlet.PortletConfig";
        return (PortletConfig) FacesContext.getCurrentInstance()
                .getExternalContext().getRequestMap().get(name);
    }

    public static void clearSession() {
        Map sessionMap = FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap();
        if (sessionMap != null) {
            sessionMap.remove(CMSConstants.NEW_OBJECT_NAME);
            sessionMap.remove(CMSConstants.SELECTED_OBJECT_NAME);
        } else {
            log.warn("Cannot get the session.");
        }
    }

    public static String escapeAttributeValue(String str) {
        // TODO review conversion
        // return StringEscapeUtils.escapeJava(str);
        return str.replaceAll("'", "\\'");
    }

    public static boolean setProperty(Node node, String key, String value)
            throws VersionException, LockException, ValueFormatException,
            RepositoryException {
        if (value == null) {
            return false;
        }
        node.setProperty(key, value);
        return true;
    }

    public static boolean setProperty(Node node, String key, String value,
            String defaultValue) throws VersionException, LockException,
            ValueFormatException, RepositoryException {
        if (!setProperty(node, key, value)) {
            if (defaultValue == null) {
                return false;
            }
            node.setProperty(key, defaultValue);
        }
        return true;
    }

    public static boolean setProperty(Node node, String key, Calendar value)
            throws VersionException, LockException, ValueFormatException,
            RepositoryException {
        if (value == null) {
            return false;
        }
        node.setProperty(key, value);
        return true;
    }

    public static boolean setProperty(Node node, String key, Calendar value,
            Calendar defaultValue) throws VersionException, LockException,
            ValueFormatException, RepositoryException {
        if (!setProperty(node, key, value)) {
            if (defaultValue == null) {
                return false;
            }
            node.setProperty(key, defaultValue);
        }
        return true;
    }

    public static boolean setProperty(Node node, String key, InputStream value)
            throws VersionException, LockException, ValueFormatException,
            RepositoryException {
        if (value == null) {
            return false;
        }
        node.setProperty(key, value);
        return true;
    }

    public static boolean setProperty(Node node, String key, InputStream value,
            InputStream defaultValue) throws VersionException, LockException,
            ValueFormatException, RepositoryException {
        if (!setProperty(node, key, value)) {
            if (defaultValue == null) {
                return false;
            }
            node.setProperty(key, defaultValue);
        }
        return true;
    }

    public static String getNamespace() {
        return FacesContext.getCurrentInstance().getExternalContext()
                .encodeNamespace(CMSConstants.PAGE_NAME);
    }

    public static RenderResponse getRenderResponse() {
        Object response = FacesContext.getCurrentInstance()
                .getExternalContext().getResponse();
        if (response instanceof RenderResponse) {
            return (RenderResponse) response;
        }
        throw new IllegalStateException("Could not get RenderResponse.");
    }

    public static String getMimeType(String filename) {
        return getMimeType(FacesContext.getCurrentInstance()
                .getExternalContext().getContext(), filename);
    }

    public static String getMimeType(Object context, String filename) {
        if (context instanceof ServletContext) {
            ServletContext servletContext = (ServletContext) context;
            return servletContext.getMimeType(filename);
        } else if (context instanceof PortletContext) {
            PortletContext portletContext = (PortletContext) context;
            return portletContext.getMimeType(filename);
        }
        return null;
    }

    public static boolean isTextMimeType(String mimeType) {
        if (mimeType != null && mimeType.startsWith("text")) {
            return true;
        }
        return false;
    }

    public static boolean isHtmlMimeType(String mimeType) {
        if (mimeType != null
                && mimeType.equals(CMSConstants.DEFAULT_HTML_MIMETYPE)) {
            return true;
        }
        return false;
    }

}
