# testblazermodule.rb: testcases for additional methods on WEBrick::HTTPResponse

require 'test/unit'
load 'blazer-ja-proxy'

require 'nkf'
require 'uri'

class TestBlazerModule < Test::Unit::TestCase

	def test_charset
		{
			'iso-8859-1' => NKF::ASCII,
			'EUC-JP' => NKF::EUC,
			'utf-8' => NKF::UTF8,
			'ISO-2022-JP' => NKF::JIS,
			'x-sjis' => NKF::SJIS,
			'shift-jis' => NKF::SJIS,
			nil => nil,
		}.each_pair do |charset_string, constant|
			assert_equal(constant, BlazerProxy::charset_to_nkf(charset_string))
		end
		{
			NKF::ASCII => 'ISO-8859-1',
			NKF::EUC => 'EUC-JP',
			NKF::JIS => 'ISO-2022-JP',
			NKF::SJIS => 'Shift-JIS',
			NKF::UTF8 => 'UTF-8',
			NKF::UTF16 => 'UTF-16',
			NKF::BINARY => nil,
			nil => nil,
		}.each_pair do |constant, str|
			assert_equal(str, BlazerProxy::nkf_to_charset(constant))
		end
	end

	def test_content_type
		{
			'text/html; charset=iso-8859-1' => NKF::ASCII,
			'text/html; charset=EUC-JP' => NKF::EUC,
			'text/html; charset=utf-8' => NKF::UTF8,
			'text/html; charset=UTF-8' => NKF::UTF8,
			'text/plain; charset=utf-8' => NKF::UTF8,
			'text/html' => nil,
			'text/plain' => nil,
			'text/css' => nil,
			'image/gif' => NKF::BINARY,
			nil => nil,
		}.each_pair do |content_type, charset|
			assert_equal(charset, BlazerProxy::content_type_to_nkf(content_type), "Content-type: " + content_type.inspect)
		end
	end

	def test_registry
		input = {
			'http://A.B.C/' => NKF::ASCII,
			'http://b.c.d/' => NKF::EUC,
			'http://b.c.d/EFG#moge' => NKF::UTF8,
		}
		reg = {
			'http://a.b.c/' => NKF::ASCII,
			'http://b.c.d/' => NKF::EUC,
			'http://b.c.d/EFG' => NKF::UTF8,
		}

		t = BlazerProxy::CharsetRegistry.new
		input.each_pair do |url, const|
			t[URI::parse(url)] = const
		end
		reg.each_pair do |url, const|
			assert_equal(const, t[URI::parse(url)])
		end
	end

	def test_registry_max
		input = [
			['http://A.B.C/', NKF::ASCII],
			['http://b.c.d/', NKF::EUC],
			['http://b.c.d/EFG#moge', NKF::UTF8],
			['http://b.c.d/', NKF::SJIS],
		]
		reg = {
			'http://b.c.d/' => NKF::SJIS,
		}

		t = BlazerProxy::CharsetRegistry.new(2)
		input.each do |e|
			t[URI::parse(e[0])] = e[1]
		end
		reg.each_pair do |url, const|
			assert_equal(const, t[URI::parse(url)])
		end
	end

	def test_convert_form_data
		src = NKF.nkf("-e -Lw", <<'_END')
--------------------------------117511377109921378543360
Content-Disposition: form-data; name="file"; filename="test"
Content-Type: application/octet-stream


--------------------------------117511377109921378543360
Content-Disposition: form-data; name="ja"

ܸʸϤǤ

--------------------------------117511377109921378543360
Content-Disposition: form-data; name="en"

English text

--------------------------------117511377109921378543360--
_END
		dst = NKF.nkf("-s -E", src)
		converted = BlazerProxy::convert_form_data(src, '------------------------------117511377109921378543360', NKF::SJIS)
		assert_equal(dst, converted)
	end

	def test_multibyte
		src = 'testǤȡdesuto'
		%w( -e -j -s -w ).each do |code|
			assert(BlazerProxy::have_multibyte?(NKF::nkf(code, src)), code)
		end
		#assert(!BlazerProxy::have_multibyte?("Gr\xfc\xdf Gott"), "Gruess Got")
		# Some ISO-8859-1 8-bit stream seem like EUC Kanji, unfortunately.
	end

	def test_convert_urlencoded
		{'x=B%26B' => 'x=B%26B'}.each_pair do |src, dst|
			assert_equal(dst, BlazerProxy::convert_urlencoded(src, NKF::EUC))
		end
	end

	def test_sanitize
		{
			["Ascii\244\310\306\374\313\334\270\354.\n", NKF::EUC] =>
			"Ascii\244\310\306\374\313\334\270\354.\n",
			["Ascii\244\310\306\374\313\334\270\354.\n", 'EUC-JP'] =>
			"Ascii\244\310\306\374\313\334\270\354.\n",
			["Ascii\202\306\223\372\226{\214\352.\n", NKF::SJIS] =>
			"Ascii\202\306\223\372\226{\214\352.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", NKF::JIS] =>
			"Ascii\e$B$HF|K\\8l\e(B.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'ISO-2022-JP'] =>
			"Ascii\e$B$HF|K\\8l\e(B.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'iso-2022-jp'] =>
			"Ascii\e$B$HF|K\\8l\e(B.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'ISO-2022-WHATEVER'] =>
			"Ascii\e$B$HF|K\\8l\e(B.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'iso-2022-whatever'] =>
			"Ascii\e$B$HF|K\\8l\e(B.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", NKF::EUC] => "Ascii$HF|K\\8l.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'EUC-JP'] => "Ascii$HF|K\\8l.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", NKF::SJIS] => "Ascii$HF|K\\8l.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'iso-8859-1'] => "Ascii$HF|K\\8l.\n",
			["Ascii\e$B$HF|K\\8l\e(B.\n", 'utf-8'] => "Ascii$HF|K\\8l.\n",
		}.each_pair do |src, dst|
			src_string, src_code = src
			assert_equal(dst, BlazerProxy::sanitize(src_string, src_code))
		end
	end

	def test_sanitize_in_convert
		euc_target = "\306\374\313\334\270\354\244\310Ascii\244\310\306\374\313\334\270\354.\n"
		contaminated_target = "\306\374\313\334\270\354\244\310Ascii$HF|K\\8l.\n"
		{
			["\e$BF|K\\8l$H\e(BAscii\e$B$HF|K\\8l\e(B.\n", NKF::JIS] => euc_target,
			["\223\372\226{\214\352\202\306Ascii\202\306\223\372\226{\214\352.\n", NKF::SJIS] => euc_target,
			["\e$BF|K\\8l$H\e(BAscii\e$B$HF|K\\8l\e(B.\n", NKF::JIS] => euc_target,
			["\306\374\313\334\270\354\244\310Ascii\e$B$HF|K\\8l\e(B.\n", NKF::EUC] => contaminated_target,
			["\223\372\226{\214\352\202\306Ascii\e$B$HF|K\\8l\e(B.\n", NKF::SJIS] => contaminated_target,
		}.each_pair do |src, dst|
			src_string, src_code = src
			assert_equal(dst, BlazerProxy::convert(src_string, NKF::EUC, src_code), src)
		end
	end

end
