/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.IssueSyncDBConnection;
import jp.valtech.bts.connection.VoteSyncDBConnection;
import jp.valtech.bts.dao.AttachmentDAO;
import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.dao.CommentHistoryDAO;
import jp.valtech.bts.dao.IssueDAO;
import jp.valtech.bts.dao.IssueHistoryDAO;
import jp.valtech.bts.dao.VoteDAO;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueSyncHeader;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.Vote;
import jp.valtech.bts.util.AttachmentUtility;
import jp.valtech.bts.util.Logging;

/**
 * サーバ側同期処理用のFacadeクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SyncServerFacade implements Logging {

	/** リクエスト元に返す課題票情報 */
	private Issue[] syncGarbages;
	
	/** リクエスト元に返すごみ箱情報 */
	private Issue[] syncIssues;
	
	/**
	 * 何もしない。
	 */
	public SyncServerFacade() {
		;
	}

	/**
	 * リクエスト元に返す課題票情報を読み込みます。
	 * 
	 * @param		issueSyncHeaderList		同期対象となる課題票のIssueSyncHeaderの配列
	 * @param		localGarbageHeader		同期対象となるごみ箱のFingerPrintのString配列
	 */
	public void readSyncIssues(List issueSyncHeaderList, List localGarbageHeader) throws BtsDBException ,IOException {
		
		// DBコネクション取得(*** Sync処理用のコネクションを使うこと ***)
		IssueSyncDBConnection issueDbcon = new IssueSyncDBConnection();
		VoteSyncDBConnection voteDbcon = new VoteSyncDBConnection();

		try {
			// DAO生成
			IssueDAO issueDAO = new IssueDAO(issueDbcon);
			CommentHistoryDAO commentDAO = new CommentHistoryDAO(issueDbcon);
			IssueHistoryDAO historyDAO = new IssueHistoryDAO(issueDbcon);
			AttachmentDAO attachmentDAO = new AttachmentDAO(issueDbcon);
			
			VoteDAO voteDAO = new VoteDAO(voteDbcon);
			
			// リクエスト元に戻す課題票情報を生成
			if(issueSyncHeaderList != null ) {
				
				syncIssues = new Issue[issueSyncHeaderList.size()];
				
				IssueSyncHeader[] headers 
					= (IssueSyncHeader[])issueSyncHeaderList.toArray(new IssueSyncHeader[0]);

				for (int i = 0; i < headers.length; i++) {
					
					// FingerPrint取得
					String fingerPrint = headers[i].getFingerPrint();

					// 課題票情報取得
					Issue issue = issueDAO.getByFingerPrint(fingerPrint, IssueType.RELEASE_VALUE);
					
					// コメント履歴取得
					CommentHistory[] comments = commentDAO.getByFingerPrint(fingerPrint, IssueType.RELEASE_VALUE);
					
					// 課題票履歴取得
					IssueHistory[] histories = historyDAO.getByFingerPrint(fingerPrint, IssueType.RELEASE_VALUE);
					
					// 添付ファイル取得 
					ArrayList attachmentList = attachmentDAO.getByFingerPrint(fingerPrint, IssueType.RELEASE_VALUE);
					AttachmentUtility utility = new AttachmentUtility();
					utility.read(attachmentList);
					
					// 投票情報取得
					Vote[] votes = voteDAO.get(fingerPrint);

					// 課題票に取得した情報を設定
					issue.setCommentHistories(comments);
					issue.setIssueHistories(histories);
					issue.setAttachmentList(attachmentList);
					issue.setVotes(votes);
					
					// Listに加える
					syncIssues[i] = issue;
				}
			}
			
			
			// リクエスト元に戻すごみ箱情報を生成
			if(localGarbageHeader != null) {

				syncGarbages = new Issue[localGarbageHeader.size()];
				
				String[] fingerPrints = (String[])localGarbageHeader.toArray(new String[0]);
				
				for (int i = 0; i < fingerPrints.length; i++) {
					// FingerPrintを元に対象となる課題票を取得
					Issue issue = 
					  issueDAO.getByFingerPrint( fingerPrints[i] , IssueType.GARBAGE_VALUE);
					
					// 課題票がごみ箱に入った時の履歴を取得
					IssueHistory history = historyDAO.getMovedGarbageHistory(fingerPrints[i]);
					issue.setIssueHistories(new IssueHistory[]{history});
					
					// Listに加える
					syncGarbages[i] = issue;
				}
			}

		} finally {
			issueDbcon.close();
			issueDbcon = null;
			
			voteDbcon.close();
			voteDbcon = null;
		}
	}
	

	/**
	 * リクエスト元に返す課題票情報取得
	 * 
	 * @return		同期リクエスト元に返す課題票情報
	 */
	public Issue[] getSyncGarbages(){
		return syncGarbages;
	}
	
	/**
	 * リクエスト元に返すごみ箱情報取得
	 * 
	 * @return		同期リクエスト元に返すごみ箱情報
	 */
	public Issue[] getSyncIssues() {
		return syncIssues;
	}

}