/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network.command;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.MulticastSocket;

import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.Response;
import jp.valtech.bts.util.IOUtility;

/**
 * マルチキャストデータ通信を行うコマンド機能のための基底クラス。
 * 指定されたホストに対してデータを送る処理は{@link TCPClientCommand#send(Request)}を使用します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public abstract class MulticastClientCommand {

	
	protected NetworkConfig myconfig;
	
	/**
	 * インスタンスにネットワーク情報を設定します。
	 * 
	 * @param		myconfig 	ローカルマシンのネットワーク情報
	 */
	public MulticastClientCommand( NetworkConfig myconfig ) {
		this.myconfig = myconfig;
	}

	/**
	 * データを送信する処理。
	 * 指定されたホストにデータを送信する処理です。相手先に対して送信するデータは{@link Request}
	 * で構築します。またこの通信は、UDP通信として定義されているので、リクエストに対するレスポンスは
	 * ありません。この処理が正常に完了した場合、{@link Response}に対してコード{@link Response#OK}
	 * のみを含むインスタンスが返ります。
	 */
	protected void send( Request request ) throws IOException {

		MulticastSocket socket = null;
		InetAddress multicastAdrs = null;
		try{
			// マルチキャスト接続情報取得
			int port = myconfig.getMulticastPort();
			String address = myconfig.getMulticastAddress();
			
			// 接続
			socket = new MulticastSocket();
			multicastAdrs = InetAddress.getByName(address);

			// リクエストオブジェクトをバイト配列に変換します。
			byte[] bytes = IOUtility.toByte(request);
			
			// マルチキャスト配信
			DatagramPacket packet = new DatagramPacket( bytes, bytes.length, multicastAdrs, port);
			socket.send( packet );

		}finally{
			if(socket != null) {
				socket.close();
				socket = null;
			}
		}
	}
}
