/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network.command;

import java.io.ObjectOutputStream;
import java.net.InetAddress;
import java.net.Socket;

import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.Response;
import jp.valtech.bts.util.Logging;

/**
 * TCP通信処理を受け付ける受信側コマンド処理の基底クラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public abstract class TCPServerCommand implements Runnable, Logging {

	/** ローカル端末のネットワーク設定 */
	private NetworkConfig myconfig	= null;
	
	/** リクエストデータ */
	private Request request		= null;

	/** 接続要求により生成されたソケットインスタンス。 派生クラスからは直接参照できません。*/
	private Socket socket = null;

	/**
	 * サーバコマンドインスタンスを生成します。
	 */
	public TCPServerCommand() {
		super();
	}

	/**
	 * TCP通信のレシーバインスタンスを格納します。
	 * 
	 * @param		tcpReceiver		TCP通信のレシーバインスタンス
	 */
	public void setSocket( Socket socket ) {
		this.socket = socket;
	}

	/**
	 * <DL><DT><B>configを設定します。</B></DT>
	 * <DD></DD>
	 * </DL>
	 * @param config config。
	 */
	public void setConfig( NetworkConfig myconfig ) {
		this.myconfig = myconfig;
	}

	/**
	 * <DL><DT><B>リクエストデータを設定します。</B></DT>
	 * <DD></DD>
	 * </DL>
	 * @param request リクエストデータ。
	 */
	public void setRequest( Request request ) {
		this.request = request;
	}

	/**
	 * TCP接続を要求してきた送信元に返事を戻します。
	 * 送信する情報は引数の<code>response</code>に設定します。
	 * 
	 * @param		response			返信データ
	 */
	protected void reply(Response response) {
		try {
			// Responseを返してデータ受信を送信側に知らせる。
			ObjectOutputStream os = new ObjectOutputStream( this.socket.getOutputStream() );
			os.writeObject(response);
			os.flush();
			os.close();
		} catch (Exception e) {
			logger.fatal(e.getMessage(), e);
		}
	}

	
	/**
	 * リクエストの送信元ホストの情報を{@link InetAddress}の形式で返します。
	 * 
	 * @return		リクエスト送信元のホスト情報
	 */
	protected InetAddress getRequestClient() {
		return this.socket.getInetAddress();
	}

	/**
	 * <DL><DT><B>コマンド実行処理(スレッド処理)</B></DL>
	 *  <DD>コマンド実行をスレッドで行うためのメソッドです。
	 *  {@link Thread}クラスを利用して実行してください。
	 *  </DD>
	 * </DL>
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		Response response = new Response();
		execute(request, response, myconfig);
	}

	/**
	 * TCPサーバコマンドを実行します。
	 * 
	 * @param		request			リクエストオブジェクト
	 * @param		response		レスポンスオブジェクト
	 * @param		myconfig		ローカル端末のネットワーク情報
	 */
	public abstract void execute(Request request, Response response, NetworkConfig myconfig);

}
